/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.glowroot.agent;

import java.lang.instrument.ClassFileTransformer;
import java.security.ProtectionDomain;

import org.glowroot.agent.shaded.org.checkerframework.checker.nullness.qual.MonotonicNonNull;
import org.glowroot.agent.shaded.org.checkerframework.checker.nullness.qual.Nullable;
import org.glowroot.agent.shaded.org.objectweb.asm.ClassReader;
import org.glowroot.agent.shaded.org.objectweb.asm.ClassVisitor;
import org.glowroot.agent.shaded.org.objectweb.asm.ClassWriter;
import org.glowroot.agent.shaded.org.objectweb.asm.MethodVisitor;
import org.glowroot.agent.shaded.org.objectweb.asm.Type;
import org.glowroot.agent.shaded.org.objectweb.asm.commons.AdviceAdapter;
import org.glowroot.agent.shaded.org.slf4j.Logger;
import org.glowroot.agent.shaded.org.slf4j.LoggerFactory;

import static org.glowroot.agent.shaded.org.objectweb.asm.Opcodes.ASM7;

// needs to be public, loadingClass() method is used by generated bytecode, see below
public class DebuggingClassFileTransformer implements ClassFileTransformer {

    private static volatile @MonotonicNonNull Logger logger;

    @Override
    public byte /*@Nullable*/ [] transform(@Nullable ClassLoader loader, @Nullable String className,
            @Nullable Class<?> classBeingRedefined, @Nullable ProtectionDomain protectionDomain,
            byte[] bytes) {
        try {
            if ("sun/misc/Launcher$AppClassLoader".equals(className)) {
                ClassWriter cw = new ClassWriter(ClassWriter.COMPUTE_MAXS);
                ClassVisitor cv = new DebuggingClassVisitor(cw);
                ClassReader cr = new ClassReader(bytes);
                cr.accept(cv, ClassReader.EXPAND_FRAMES);
                return cw.toByteArray();
            }
        } catch (Throwable t) {
            if (logger == null) {
                t.printStackTrace();
            } else {
                logger.error(t.getMessage(), t);
            }
        }
        return null;
    }

    // used via generated bytecode, see DebuggingMethodAdvice below
    public static void loadingClass(String name) {
        if (name.startsWith("org.glowroot.")) {
            new Exception(name).printStackTrace();
        }
    }

    public static void initLogger() {
        logger = LoggerFactory.getLogger(DebuggingClassFileTransformer.class);
    }

    private static class DebuggingClassVisitor extends ClassVisitor {

        private final ClassWriter cw;

        private DebuggingClassVisitor(ClassWriter cw) {
            super(ASM7, cw);
            this.cw = cw;
        }

        @Override
        public MethodVisitor visitMethod(int access, String name, String descriptor,
                @Nullable String signature, String /*@Nullable*/ [] exceptions) {
            MethodVisitor mv = cw.visitMethod(access, name, descriptor, signature, exceptions);
            if (name.equals("loadClass")
                    && descriptor.equals("(Ljava/lang/String;Z)Ljava/lang/Class;")) {
                return new DebuggingMethodAdvice(mv, access, name, descriptor);
            } else {
                return mv;
            }
        }
    }

    private static class DebuggingMethodAdvice extends AdviceAdapter {

        private DebuggingMethodAdvice(MethodVisitor mv, int access, String name,
                String descriptor) {
            super(ASM7, mv, access, name, descriptor);
        }

        @Override
        protected void onMethodEnter() {
            visitVarInsn(ALOAD, 1);
            visitMethodInsn(INVOKESTATIC, Type.getInternalName(DebuggingClassFileTransformer.class),
                    "loadingClass", "(Ljava/lang/String;)V", false);
        }
    }
}
