package com.apple.foundationdb;

/**
 * A set of options that can be set on a {@link Transaction}.
 */
public class TransactionOptions extends OptionsSet {
	public TransactionOptions( OptionConsumer consumer ) { super(consumer); }

	/**
	 * The transaction, if not self-conflicting, may be committed a second time after commit succeeds, in the event of a fault.
	 */
	public void setCausalWriteRisky() { setOption(10); }

	/**
	 * The read version will be committed, and usually will be the latest committed, but might not be the latest committed in the event of a simultaneous fault and misbehaving clock.
	 */
	public void setCausalReadRisky() { setOption(20); }

	/**
	 * Addresses returned by get_addresses_for_key include the port when enabled. As of api version 630, this option is enabled by default and setting this has no effect.
	 */
	public void setIncludePortInAddress() { setOption(23); }

	/**
	 * The next write performed on this transaction will not generate a write conflict range. As a result, other transactions which read the key(s) being modified by the next write will not conflict with this transaction. Care needs to be taken when using this option on a transaction that is shared between multiple threads. When setting this option, write conflict ranges will be disabled on the next write operation, regardless of what thread it is on.
	 */
	public void setNextWriteNoWriteConflictRange() { setOption(30); }

	/**
	 * Reads performed by a transaction will not see any prior mutations that occurred in that transaction, instead seeing the value which was in the database at the transaction's read version. This option may provide a small performance benefit for the client, but also disables a number of client-side optimizations which are beneficial for transactions which tend to read and write the same keys within a single transaction. It is an error to set this option after performing any reads or writes on the transaction.
	 */
	public void setReadYourWritesDisable() { setOption(51); }

	/**
	 * Deprecated.
	 */
	@Deprecated
	public void setReadAheadDisable() { setOption(52); }

	/**
	 * Storage server should cache disk blocks needed for subsequent read requests in this transaction.  This is the default behavior.
	 */
	public void setReadServerSideCacheEnable() { setOption(507); }

	/**
	 * Storage server should not cache disk blocks needed for subsequent read requests in this transaction.  This can be used to avoid cache pollution for reads not expected to be repeated.
	 */
	public void setReadServerSideCacheDisable() { setOption(508); }

	/**
	 * Use normal read priority for subsequent read requests in this transaction.  This is the default read priority.
	 */
	public void setReadPriorityNormal() { setOption(509); }

	/**
	 * Use low read priority for subsequent read requests in this transaction.
	 */
	public void setReadPriorityLow() { setOption(510); }

	/**
	 * Use high read priority for subsequent read requests in this transaction.
	 */
	public void setReadPriorityHigh() { setOption(511); }

	/**
	 * Deprecated.
	 */
	@Deprecated
	public void setDurabilityDevNullIsWebScale() { setOption(130); }

	/**
	 * Specifies that this transaction should be treated as highest priority and that lower priority transactions should block behind this one. Use is discouraged outside of low-level tools.
	 */
	public void setPrioritySystemImmediate() { setOption(200); }

	/**
	 * Specifies that this transaction should be treated as low priority and that default priority transactions will be processed first. Batch priority transactions will also be throttled at load levels smaller than for other types of transactions and may be fully cut off in the event of machine failures. Useful for doing batch work simultaneously with latency-sensitive work.
	 */
	public void setPriorityBatch() { setOption(201); }

	/**
	 * This is a write-only transaction which sets the initial configuration. This option is designed for use by database system tools only.
	 */
	public void setInitializeNewDatabase() { setOption(300); }

	/**
	 * Allows this transaction to read and modify system keys (those that start with the byte 0xFF). Implies raw_access.
	 */
	public void setAccessSystemKeys() { setOption(301); }

	/**
	 * Allows this transaction to read system keys (those that start with the byte 0xFF). Implies raw_access.
	 */
	public void setReadSystemKeys() { setOption(302); }

	/**
	 * Allows this transaction to access the raw key-space when tenant mode is on.
	 */
	public void setRawAccess() { setOption(303); }

	/**
	 * Allows this transaction to bypass storage quota enforcement. Should only be used for transactions that directly or indirectly decrease the size of the tenant group's data.
	 */
	public void setBypassStorageQuota() { setOption(304); }

	/**
	 * Deprecated.
	 * 
	 * @param value String identifier to be used in the logs when tracing this transaction. The identifier must not exceed 100 characters.
	 */
	@Deprecated
	public void setTransactionLoggingEnable(String value) { setOption(402, value); }

	/**
	 * Sets a client provided identifier for the transaction that will be used in scenarios like tracing or profiling. Client trace logging or transaction profiling must be separately enabled.
	 * 
	 * @param value String identifier to be used when tracing or profiling this transaction. The identifier must not exceed 100 characters.
	 */
	public void setDebugTransactionIdentifier(String value) { setOption(403, value); }

	/**
	 * Enables tracing for this transaction and logs results to the client trace logs. The DEBUG_TRANSACTION_IDENTIFIER option must be set before using this option, and client trace logging must be enabled to get log output.
	 */
	public void setLogTransaction() { setOption(404); }

	/**
	 * Sets the maximum escaped length of key and value fields to be logged to the trace file via the LOG_TRANSACTION option, after which the field will be truncated. A negative value disables truncation.
	 * 
	 * @param value Maximum length of escaped key and value fields.
	 */
	public void setTransactionLoggingMaxFieldLength(long value) { setOption(405, value); }

	/**
	 * Sets an identifier for server tracing of this transaction. When committed, this identifier triggers logging when each part of the transaction authority encounters it, which is helpful in diagnosing slowness in misbehaving clusters. The identifier is randomly generated. When there is also a debug_transaction_identifier, both IDs are logged together.
	 */
	public void setServerRequestTracing() { setOption(406); }

	/**
	 * Set a timeout in milliseconds which, when elapsed, will cause the transaction automatically to be cancelled. Valid parameter values are {@code [0, INT_MAX]}. If set to 0, will disable all timeouts. All pending and any future uses of the transaction will throw an exception. The transaction can be used again after it is reset. Prior to API version 610, like all other transaction options, the timeout must be reset after a call to {@code onError}. If the API version is 610 or greater, the timeout is not reset after an {@code onError} call. This allows the user to specify a longer timeout on specific transactions than the default timeout specified through the {@code transaction_timeout} database option without the shorter database timeout cancelling transactions that encounter a retryable error. Note that at all API versions, it is safe and legal to set the timeout each time the transaction begins, so most code written assuming the older behavior can be upgraded to the newer behavior without requiring any modification, and the caller is not required to implement special logic in retry loops to only conditionally set this option.
	 * 
	 * @param value value in milliseconds of timeout
	 */
	public void setTimeout(long value) { setOption(500, value); }

	/**
	 * Set a maximum number of retries after which additional calls to {@code onError} will throw the most recently seen error code. Valid parameter values are {@code [-1, INT_MAX]}. If set to -1, will disable the retry limit. Prior to API version 610, like all other transaction options, the retry limit must be reset after a call to {@code onError}. If the API version is 610 or greater, the retry limit is not reset after an {@code onError} call. Note that at all API versions, it is safe and legal to set the retry limit each time the transaction begins, so most code written assuming the older behavior can be upgraded to the newer behavior without requiring any modification, and the caller is not required to implement special logic in retry loops to only conditionally set this option.
	 * 
	 * @param value number of times to retry
	 */
	public void setRetryLimit(long value) { setOption(501, value); }

	/**
	 * Set the maximum amount of backoff delay incurred in the call to {@code onError} if the error is retryable. Defaults to 1000 ms. Valid parameter values are {@code [0, INT_MAX]}. If the maximum retry delay is less than the current retry delay of the transaction, then the current retry delay will be clamped to the maximum retry delay. Prior to API version 610, like all other transaction options, the maximum retry delay must be reset after a call to {@code onError}. If the API version is 610 or greater, the retry limit is not reset after an {@code onError} call. Note that at all API versions, it is safe and legal to set the maximum retry delay each time the transaction begins, so most code written assuming the older behavior can be upgraded to the newer behavior without requiring any modification, and the caller is not required to implement special logic in retry loops to only conditionally set this option.
	 * 
	 * @param value value in milliseconds of maximum delay
	 */
	public void setMaxRetryDelay(long value) { setOption(502, value); }

	/**
	 * Set the transaction size limit in bytes. The size is calculated by combining the sizes of all keys and values written or mutated, all key ranges cleared, and all read and write conflict ranges. (In other words, it includes the total size of all data included in the request to the cluster to commit the transaction.) Large transactions can cause performance problems on FoundationDB clusters, so setting this limit to a smaller value than the default can help prevent the client from accidentally degrading the cluster's performance. This value must be at least 32 and cannot be set to higher than 10,000,000, the default transaction size limit.
	 * 
	 * @param value value in bytes
	 */
	public void setSizeLimit(long value) { setOption(503, value); }

	/**
	 * Automatically assign a random 16 byte idempotency id for this transaction. Prevents commits from failing with {@code commit_unknown_result}. WARNING: If you are also using the multiversion client or transaction timeouts, if either cluster_version_changed or transaction_timed_out was thrown during a commit, then that commit may have already succeeded or may succeed in the future. This feature is in development and not ready for general use.
	 */
	public void setAutomaticIdempotency() { setOption(505); }

	/**
	 * Snapshot read operations will see the results of writes done in the same transaction. This is the default behavior.
	 */
	public void setSnapshotRywEnable() { setOption(600); }

	/**
	 * Snapshot read operations will not see the results of writes done in the same transaction. This was the default behavior prior to API version 300.
	 */
	public void setSnapshotRywDisable() { setOption(601); }

	/**
	 * The transaction can read and write to locked databases, and is responsible for checking that it took the lock.
	 */
	public void setLockAware() { setOption(700); }

	/**
	 * By default, operations that are performed on a transaction while it is being committed will not only fail themselves, but they will attempt to fail other in-flight operations (such as the commit) as well. This behavior is intended to help developers discover situations where operations could be unintentionally executed after the transaction has been reset. Setting this option removes that protection, causing only the offending operation to fail.
	 */
	public void setUsedDuringCommitProtectionDisable() { setOption(701); }

	/**
	 * The transaction can read from locked databases.
	 */
	public void setReadLockAware() { setOption(702); }

	/**
	 * This option should only be used by tools which change the database configuration.
	 */
	public void setUseProvisionalProxies() { setOption(711); }

	/**
	 * The transaction can retrieve keys that are conflicting with other transactions.
	 */
	public void setReportConflictingKeys() { setOption(712); }

	/**
	 * By default, the special key space will only allow users to read from exactly one module (a subspace in the special key space). Use this option to allow reading from zero or more modules. Users who set this option should be prepared for new modules, which may have different behaviors than the modules they're currently reading. For example, a new module might block or return an error.
	 */
	public void setSpecialKeySpaceRelaxed() { setOption(713); }

	/**
	 * By default, users are not allowed to write to special keys. Enable this option will implicitly enable all options required to achieve the configuration change.
	 */
	public void setSpecialKeySpaceEnableWrites() { setOption(714); }

	/**
	 * Adds a tag to the transaction that can be used to apply manual targeted throttling. At most 5 tags can be set on a transaction.
	 * 
	 * @param value String identifier used to associated this transaction with a throttling group. Must not exceed 16 characters.
	 */
	public void setTag(String value) { setOption(800, value); }

	/**
	 * Adds a tag to the transaction that can be used to apply manual or automatic targeted throttling. At most 5 tags can be set on a transaction.
	 * 
	 * @param value String identifier used to associated this transaction with a throttling group. Must not exceed 16 characters.
	 */
	public void setAutoThrottleTag(String value) { setOption(801, value); }

	/**
	 * Adds a parent to the Span of this transaction. Used for transaction tracing. A span can be identified with a 33 bytes serialized binary format which consists of: 8 bytes protocol version, e.g. {@code 0x0FDB00B073000000LL} in little-endian format, 16 bytes trace id, 8 bytes span id, 1 byte set to 1 if sampling is enabled.
	 * 
	 * @param value A serialized binary byte string of length 33 used to associate the span of this transaction with a parent
	 */
	public void setSpanParent(byte[] value) { setOption(900, value); }

	/**
	 * Asks storage servers for how many bytes a clear key range contains. Otherwise uses the location cache to roughly estimate this.
	 */
	public void setExpensiveClearCostEstimationEnable() { setOption(1000); }

	/**
	 * Allows {@code get} operations to read from sections of keyspace that have become unreadable because of versionstamp operations. These reads will view versionstamp operations as if they were set operations that did not fill in the versionstamp.
	 */
	public void setBypassUnreadable() { setOption(1100); }

	/**
	 * Allows this transaction to use cached GRV from the database context. Defaults to off. Upon first usage, starts a background updater to periodically update the cache to avoid stale read versions. The disable_client_bypass option must also be set.
	 */
	public void setUseGrvCache() { setOption(1101); }

	/**
	 * Attach given authorization token to the transaction such that subsequent tenant-aware requests are authorized.
	 * 
	 * @param value A JSON Web Token authorized to access data belonging to one or more tenants, indicated by 'tenants' claim of the token's payload.
	 */
	public void setAuthorizationToken(String value) { setOption(2000, value); }

	/**
	 * Enables replica consistency check, which compares the results returned by storage server replicas (as many as specified by consistency_check_required_replicas option) for a given read request, in client-side load balancer.
	 */
	public void setEnableReplicaConsistencyCheck() { setOption(4000); }

	/**
	 * Specifies the number of storage server replica results that the load balancer needs to compare when enable_replica_consistency_check option is set.
	 * 
	 * @param value Number of storage replicas over which the load balancer consistency check is done.
	 */
	public void setConsistencyCheckRequiredReplicas(long value) { setOption(4001, value); }

	public void setCausalReadDisable() { setOption(21); }

	public void setDurabilityDatacenter() { setOption(110); }

	public void setDurabilityRisky() { setOption(120); }

	public void setDebugRetryLogging(String value) { setOption(401, value); }
}
