package cdm.product.asset.floatingrate.functions;

import cdm.base.math.Rounding;
import cdm.observable.asset.Price;
import cdm.product.asset.FloatingRateSpecification;
import cdm.product.asset.InterestRatePayout;
import cdm.product.asset.NegativeInterestRateTreatmentEnum;
import cdm.product.asset.RateSpecification;
import cdm.product.asset.RateTreatmentEnum;
import cdm.product.asset.floatingrate.FloatingRateProcessingParameters;
import cdm.product.asset.floatingrate.FloatingRateProcessingParameters.FloatingRateProcessingParametersBuilder;
import cdm.product.common.schedule.CalculationPeriodBase;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperS;
import java.math.BigDecimal;
import java.util.Optional;
import javax.inject.Inject;


@ImplementedBy(GetFloatingRateProcessingParameters.GetFloatingRateProcessingParametersDefault.class)
public abstract class GetFloatingRateProcessingParameters implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected CapRateAmount capRateAmount;
	@Inject protected FloorRateAmount floorRateAmount;
	@Inject protected MultiplierAmount multiplierAmount;
	@Inject protected SpreadAmount spreadAmount;

	/**
	* @param interestRatePayout An interest rate stream.
	* @param calculationPeriod The calculation period for which the calculation is being perfmored (needed to look up paramters).
	* @return processingParameters The processing parameters.
	*/
	public FloatingRateProcessingParameters evaluate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
		FloatingRateProcessingParameters.FloatingRateProcessingParametersBuilder processingParametersBuilder = doEvaluate(interestRatePayout, calculationPeriod);
		
		final FloatingRateProcessingParameters processingParameters;
		if (processingParametersBuilder == null) {
			processingParameters = null;
		} else {
			processingParameters = processingParametersBuilder.build();
			objectValidator.validate(FloatingRateProcessingParameters.class, processingParameters);
		}
		
		return processingParameters;
	}

	protected abstract FloatingRateProcessingParameters.FloatingRateProcessingParametersBuilder doEvaluate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<BigDecimal> spreadRate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<BigDecimal> multiplier(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<BigDecimal> cap(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<BigDecimal> floor(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<? extends Rounding> rounding(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<NegativeInterestRateTreatmentEnum> negativeTreatment(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<RateTreatmentEnum> treatment(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod);

	public static class GetFloatingRateProcessingParametersDefault extends GetFloatingRateProcessingParameters {
		@Override
		protected FloatingRateProcessingParameters.FloatingRateProcessingParametersBuilder doEvaluate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
			FloatingRateProcessingParameters.FloatingRateProcessingParametersBuilder processingParameters = FloatingRateProcessingParameters.builder();
			return assignOutput(processingParameters, interestRatePayout, calculationPeriod);
		}
		
		protected FloatingRateProcessingParameters.FloatingRateProcessingParametersBuilder assignOutput(FloatingRateProcessingParameters.FloatingRateProcessingParametersBuilder processingParameters, InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
			processingParameters
				.setInitialRate(MapperS.of(interestRatePayout).<RateSpecification>map("getRateSpecification", _interestRatePayout -> _interestRatePayout.getRateSpecification()).<FloatingRateSpecification>map("getFloatingRate", rateSpecification -> rateSpecification.getFloatingRate()).<Price>map("getInitialRate", floatingRateSpecification -> floatingRateSpecification.getInitialRate()).get());
			
			processingParameters
				.setSpread(MapperS.of(spreadRate(interestRatePayout, calculationPeriod).get()).get());
			
			processingParameters
				.setMultiplier(MapperS.of(multiplier(interestRatePayout, calculationPeriod).get()).get());
			
			processingParameters
				.setTreatment(MapperS.of(treatment(interestRatePayout, calculationPeriod).get()).get());
			
			processingParameters
				.setCapRate(MapperS.of(cap(interestRatePayout, calculationPeriod).get()).get());
			
			processingParameters
				.setFloorRate(MapperS.of(floor(interestRatePayout, calculationPeriod).get()).get());
			
			processingParameters
				.setRounding(MapperS.of(rounding(interestRatePayout, calculationPeriod).get()).get());
			
			processingParameters
				.setNegativeTreatment(MapperS.of(negativeTreatment(interestRatePayout, calculationPeriod).get()).get());
			
			return Optional.ofNullable(processingParameters)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<BigDecimal> spreadRate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(spreadAmount.evaluate(MapperS.of(interestRatePayout).get(), MapperS.of(calculationPeriod).get()));
		}
		
		@Override
		protected Mapper<BigDecimal> multiplier(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(multiplierAmount.evaluate(MapperS.of(interestRatePayout).get(), MapperS.of(calculationPeriod).get()));
		}
		
		@Override
		protected Mapper<BigDecimal> cap(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(capRateAmount.evaluate(MapperS.of(interestRatePayout).get(), MapperS.of(calculationPeriod).get()));
		}
		
		@Override
		protected Mapper<BigDecimal> floor(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(floorRateAmount.evaluate(MapperS.of(interestRatePayout).get(), MapperS.of(calculationPeriod).get()));
		}
		
		@Override
		protected Mapper<? extends Rounding> rounding(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(interestRatePayout).<RateSpecification>map("getRateSpecification", _interestRatePayout -> _interestRatePayout.getRateSpecification()).<FloatingRateSpecification>map("getFloatingRate", rateSpecification -> rateSpecification.getFloatingRate()).<Rounding>map("getFinalRateRounding", floatingRateSpecification -> floatingRateSpecification.getFinalRateRounding());
		}
		
		@Override
		protected Mapper<NegativeInterestRateTreatmentEnum> negativeTreatment(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(interestRatePayout).<RateSpecification>map("getRateSpecification", _interestRatePayout -> _interestRatePayout.getRateSpecification()).<FloatingRateSpecification>map("getFloatingRate", rateSpecification -> rateSpecification.getFloatingRate()).<NegativeInterestRateTreatmentEnum>map("getNegativeInterestRateTreatment", floatingRateSpecification -> floatingRateSpecification.getNegativeInterestRateTreatment());
		}
		
		@Override
		protected Mapper<RateTreatmentEnum> treatment(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(interestRatePayout).<RateSpecification>map("getRateSpecification", _interestRatePayout -> _interestRatePayout.getRateSpecification()).<FloatingRateSpecification>map("getFloatingRate", rateSpecification -> rateSpecification.getFloatingRate()).<RateTreatmentEnum>map("getRateTreatment", floatingRate -> floatingRate.getRateTreatment());
		}
	}
}
