package cdm.product.asset.floatingrate.functions;

import cdm.observable.asset.FloatingRateOption;
import cdm.observable.asset.fro.functions.IndexValueObservation;
import cdm.observable.asset.metafields.ReferenceWithMetaFloatingRateOption;
import cdm.product.asset.FloatingRate;
import cdm.product.asset.floatingrate.FloatingRateSettingDetails;
import cdm.product.asset.floatingrate.FloatingRateSettingDetails.FloatingRateSettingDetailsBuilder;
import cdm.product.common.schedule.CalculationPeriodBase;
import cdm.product.common.schedule.ResetDates;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.records.Date;
import java.math.BigDecimal;
import java.util.Optional;
import javax.inject.Inject;


@ImplementedBy(EvaluateScreenRate.EvaluateScreenRateDefault.class)
public abstract class EvaluateScreenRate implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected DetermineFixingDate determineFixingDate;
	@Inject protected DetermineResetDate determineResetDate;
	@Inject protected IndexValueObservation indexValueObservation;

	/**
	* @param rateDef Floating rate definition.
	* @param resetDates Reset dates for observing the rate.
	* @param calculationPeriod Calculation period for which you want the rate.
	* @return details Resulting details of the rate setting .
	*/
	public FloatingRateSettingDetails evaluate(FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod) {
		FloatingRateSettingDetails.FloatingRateSettingDetailsBuilder detailsBuilder = doEvaluate(rateDef, resetDates, calculationPeriod);
		
		final FloatingRateSettingDetails details;
		if (detailsBuilder == null) {
			details = null;
		} else {
			details = detailsBuilder.build();
			objectValidator.validate(FloatingRateSettingDetails.class, details);
		}
		
		return details;
	}

	protected abstract FloatingRateSettingDetails.FloatingRateSettingDetailsBuilder doEvaluate(FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<Date> resetDate(FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<Date> fixingDate(FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod);

	protected abstract Mapper<BigDecimal> observedRate(FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod);

	public static class EvaluateScreenRateDefault extends EvaluateScreenRate {
		@Override
		protected FloatingRateSettingDetails.FloatingRateSettingDetailsBuilder doEvaluate(FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod) {
			FloatingRateSettingDetails.FloatingRateSettingDetailsBuilder details = FloatingRateSettingDetails.builder();
			return assignOutput(details, rateDef, resetDates, calculationPeriod);
		}
		
		protected FloatingRateSettingDetails.FloatingRateSettingDetailsBuilder assignOutput(FloatingRateSettingDetails.FloatingRateSettingDetailsBuilder details, FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod) {
			details
				.setResetDate(MapperS.of(resetDate(rateDef, resetDates, calculationPeriod).get()).get());
			
			details
				.setObservationDate(MapperS.of(fixingDate(rateDef, resetDates, calculationPeriod).get()).get());
			
			details
				.setFloatingRate(MapperS.of(observedRate(rateDef, resetDates, calculationPeriod).get()).get());
			
			return Optional.ofNullable(details)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<Date> resetDate(FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(determineResetDate.evaluate(MapperS.of(resetDates).get(), MapperS.of(calculationPeriod).get()));
		}
		
		@Override
		protected Mapper<Date> fixingDate(FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(determineFixingDate.evaluate(MapperS.of(resetDates).get(), MapperS.of(resetDate(rateDef, resetDates, calculationPeriod).get()).get()));
		}
		
		@Override
		protected Mapper<BigDecimal> observedRate(FloatingRate rateDef, ResetDates resetDates, CalculationPeriodBase calculationPeriod) {
			return MapperS.of(indexValueObservation.evaluate(MapperS.of(fixingDate(rateDef, resetDates, calculationPeriod).get()).get(), MapperS.of(rateDef).<ReferenceWithMetaFloatingRateOption>map("getRateOption", floatingRateBase -> floatingRateBase.getRateOption()).<FloatingRateOption>map("getValue", _f->_f.getValue()).get()));
		}
	}
}
