package cdm.product.asset.floatingrate.functions;

import cdm.product.asset.floatingrate.FloatingRateProcessingParameters;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import java.math.BigDecimal;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(ApplyCapsAndFloors.ApplyCapsAndFloorsDefault.class)
public abstract class ApplyCapsAndFloors implements RosettaFunction {

	/**
	* @param processing 
	* @param inputRate The floating rate prior to treatment, either a single term rate, or a calculated rate such as an OIS or lookback compounded rate.
	* @return cappedAndFlooredRate The rate after application of cap and/or floor.
	*/
	public BigDecimal evaluate(FloatingRateProcessingParameters processing, BigDecimal inputRate) {
		BigDecimal cappedAndFlooredRate = doEvaluate(processing, inputRate);
		
		return cappedAndFlooredRate;
	}

	protected abstract BigDecimal doEvaluate(FloatingRateProcessingParameters processing, BigDecimal inputRate);

	protected abstract Mapper<BigDecimal> cap(FloatingRateProcessingParameters processing, BigDecimal inputRate);

	protected abstract Mapper<BigDecimal> floor(FloatingRateProcessingParameters processing, BigDecimal inputRate);

	protected abstract Mapper<BigDecimal> cappedRate(FloatingRateProcessingParameters processing, BigDecimal inputRate);

	protected abstract Mapper<BigDecimal> flooredRate(FloatingRateProcessingParameters processing, BigDecimal inputRate);

	public static class ApplyCapsAndFloorsDefault extends ApplyCapsAndFloors {
		@Override
		protected BigDecimal doEvaluate(FloatingRateProcessingParameters processing, BigDecimal inputRate) {
			BigDecimal cappedAndFlooredRate = null;
			return assignOutput(cappedAndFlooredRate, processing, inputRate);
		}
		
		protected BigDecimal assignOutput(BigDecimal cappedAndFlooredRate, FloatingRateProcessingParameters processing, BigDecimal inputRate) {
			cappedAndFlooredRate = MapperS.of(flooredRate(processing, inputRate).get()).get();
			
			return cappedAndFlooredRate;
		}
		
		@Override
		protected Mapper<BigDecimal> cap(FloatingRateProcessingParameters processing, BigDecimal inputRate) {
			return MapperS.of(processing).<BigDecimal>map("getCapRate", floatingRateProcessingParameters -> floatingRateProcessingParameters.getCapRate());
		}
		
		@Override
		protected Mapper<BigDecimal> floor(FloatingRateProcessingParameters processing, BigDecimal inputRate) {
			return MapperS.of(processing).<BigDecimal>map("getFloorRate", floatingRateProcessingParameters -> floatingRateProcessingParameters.getFloorRate());
		}
		
		@Override
		protected Mapper<BigDecimal> cappedRate(FloatingRateProcessingParameters processing, BigDecimal inputRate) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(cap(processing, inputRate).get())).and(greaterThan(MapperS.of(inputRate), MapperS.of(cap(processing, inputRate).get()), CardinalityOperator.All)).getOrDefault(false)) {
					return MapperS.of(cap(processing, inputRate).get());
				}
				else {
					return MapperS.of(inputRate);
				}
			});
		}
		
		@Override
		protected Mapper<BigDecimal> flooredRate(FloatingRateProcessingParameters processing, BigDecimal inputRate) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(floor(processing, inputRate).get())).and(lessThan(MapperS.of(cappedRate(processing, inputRate).get()), MapperS.of(floor(processing, inputRate).get()), CardinalityOperator.All)).getOrDefault(false)) {
					return MapperS.of(floor(processing, inputRate).get());
				}
				else {
					return MapperS.of(cappedRate(processing, inputRate).get());
				}
			});
		}
	}
}
