package cdm.observable.asset.calculatedrate.functions;

import cdm.base.math.ArithmeticOperationEnum;
import cdm.base.math.functions.VectorOperation;
import cdm.observable.asset.calculatedrate.CalculatedRateDetails;
import cdm.observable.asset.calculatedrate.CalculatedRateDetails.CalculatedRateDetailsBuilder;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.MapperMaths;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import java.math.BigDecimal;
import java.util.List;
import java.util.Optional;
import javax.inject.Inject;


@ImplementedBy(ApplyAveragingFormula.ApplyAveragingFormulaDefault.class)
public abstract class ApplyAveragingFormula implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected VectorOperation vectorOperation;

	/**
	* @param observations a vector of observation value.
	* @param weights a vector of weights (should be same size as observations, 1 weight per observation.
	* @return results Details of the averaging calculation.
	*/
	public CalculatedRateDetails evaluate(List<BigDecimal> observations, List<BigDecimal> weights) {
		CalculatedRateDetails.CalculatedRateDetailsBuilder resultsBuilder = doEvaluate(observations, weights);
		
		final CalculatedRateDetails results;
		if (resultsBuilder == null) {
			results = null;
		} else {
			results = resultsBuilder.build();
			objectValidator.validate(CalculatedRateDetails.class, results);
		}
		
		return results;
	}

	protected abstract CalculatedRateDetails.CalculatedRateDetailsBuilder doEvaluate(List<BigDecimal> observations, List<BigDecimal> weights);

	protected abstract Mapper<BigDecimal> weightedObservations(List<BigDecimal> observations, List<BigDecimal> weights);

	protected abstract Mapper<BigDecimal> totalWeightedObservations(List<BigDecimal> observations, List<BigDecimal> weights);

	protected abstract Mapper<BigDecimal> totalWeight(List<BigDecimal> observations, List<BigDecimal> weights);

	protected abstract Mapper<BigDecimal> calculatedRate(List<BigDecimal> observations, List<BigDecimal> weights);

	public static class ApplyAveragingFormulaDefault extends ApplyAveragingFormula {
		@Override
		protected CalculatedRateDetails.CalculatedRateDetailsBuilder doEvaluate(List<BigDecimal> observations, List<BigDecimal> weights) {
			CalculatedRateDetails.CalculatedRateDetailsBuilder results = CalculatedRateDetails.builder();
			return assignOutput(results, observations, weights);
		}
		
		protected CalculatedRateDetails.CalculatedRateDetailsBuilder assignOutput(CalculatedRateDetails.CalculatedRateDetailsBuilder results, List<BigDecimal> observations, List<BigDecimal> weights) {
			results
				.setAggregateValue(MapperS.of(totalWeightedObservations(observations, weights).get()).get());
			
			results
				.setAggregateWeight(MapperS.of(totalWeight(observations, weights).get()).get());
			
			results
				.setCalculatedRate(MapperS.of(calculatedRate(observations, weights).get()).get());
			
			results
				.addWeightedRates(MapperC.<BigDecimal>of(weightedObservations(observations, weights).getMulti()).getMulti());
			
			return Optional.ofNullable(results)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<BigDecimal> weightedObservations(List<BigDecimal> observations, List<BigDecimal> weights) {
			return MapperC.<BigDecimal>of(vectorOperation.evaluate(MapperS.of(ArithmeticOperationEnum.MULTIPLY).get(), MapperC.<BigDecimal>of(observations).getMulti(), MapperC.<BigDecimal>of(weights).getMulti()));
		}
		
		@Override
		protected Mapper<BigDecimal> totalWeightedObservations(List<BigDecimal> observations, List<BigDecimal> weights) {
			return MapperC.<BigDecimal>of(weightedObservations(observations, weights).getMulti())
				.sumBigDecimal();
		}
		
		@Override
		protected Mapper<BigDecimal> totalWeight(List<BigDecimal> observations, List<BigDecimal> weights) {
			return MapperC.<BigDecimal>of(weights)
				.sumBigDecimal();
		}
		
		@Override
		protected Mapper<BigDecimal> calculatedRate(List<BigDecimal> observations, List<BigDecimal> weights) {
			return MapperMaths.<BigDecimal, BigDecimal, BigDecimal>divide(MapperS.of(totalWeightedObservations(observations, weights).get()), MapperS.of(totalWeight(observations, weights).get()));
		}
	}
}
