package cdm.event.common.functions;

import cdm.base.staticdata.party.AncillaryParty;
import cdm.base.staticdata.party.Counterparty;
import cdm.event.common.TermsChangeInstruction;
import cdm.event.common.Trade;
import cdm.event.common.TradeState;
import cdm.event.common.TradeState.TradeStateBuilder;
import cdm.product.common.NotionalAdjustmentEnum;
import cdm.product.template.Product;
import cdm.product.template.TradableProduct;
import cdm.product.template.TradeLot;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(Create_TermsChange.Create_TermsChangeDefault.class)
public abstract class Create_TermsChange implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;

	/**
	* @param termsChange Instructions to be used as an input to the function
	* @param before current trade to be ammended
	* @return tradeState 
	*/
	public TradeState evaluate(TermsChangeInstruction termsChange, TradeState before) {
		TradeState.TradeStateBuilder tradeStateBuilder = doEvaluate(termsChange, before);
		
		final TradeState tradeState;
		if (tradeStateBuilder == null) {
			tradeState = null;
		} else {
			tradeState = tradeStateBuilder.build();
			objectValidator.validate(TradeState.class, tradeState);
		}
		
		return tradeState;
	}

	protected abstract TradeState.TradeStateBuilder doEvaluate(TermsChangeInstruction termsChange, TradeState before);

	protected abstract Mapper<? extends Product> newProduct(TermsChangeInstruction termsChange, TradeState before);

	protected abstract Mapper<? extends AncillaryParty> newAncillaryParty(TermsChangeInstruction termsChange, TradeState before);

	protected abstract Mapper<NotionalAdjustmentEnum> newAdjustment(TermsChangeInstruction termsChange, TradeState before);

	public static class Create_TermsChangeDefault extends Create_TermsChange {
		@Override
		protected TradeState.TradeStateBuilder doEvaluate(TermsChangeInstruction termsChange, TradeState before) {
			TradeState.TradeStateBuilder tradeState = TradeState.builder();
			return assignOutput(tradeState, termsChange, before);
		}
		
		protected TradeState.TradeStateBuilder assignOutput(TradeState.TradeStateBuilder tradeState, TermsChangeInstruction termsChange, TradeState before) {
			tradeState = toBuilder(MapperS.of(before).get());
			
			tradeState
				.getOrCreateTrade()
				.setTradableProduct(MapperS.of(TradableProduct.builder()
					.setProduct(MapperS.of(newProduct(termsChange, before).get()).get())
					.setTradeLot(MapperS.of(tradeState).<Trade>map("getTrade", _tradeState -> _tradeState.getTrade()).<TradableProduct>map("getTradableProduct", trade -> trade.getTradableProduct()).<TradeLot>mapC("getTradeLot", tradableProduct -> tradableProduct.getTradeLot()).getMulti())
					.setCounterparty(MapperS.of(tradeState).<Trade>map("getTrade", _tradeState -> _tradeState.getTrade()).<TradableProduct>map("getTradableProduct", trade -> trade.getTradableProduct()).<Counterparty>mapC("getCounterparty", tradableProduct -> tradableProduct.getCounterparty()).getMulti())
					.setAncillaryParty(MapperC.<AncillaryParty>of(newAncillaryParty(termsChange, before).getMulti()).getMulti())
					.setAdjustment(MapperS.of(newAdjustment(termsChange, before).get()).get())
					.build())
				.get());
			
			return Optional.ofNullable(tradeState)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<? extends Product> newProduct(TermsChangeInstruction termsChange, TradeState before) {
			return MapperUtils.runSinglePolymorphic(() -> {
				if (exists(MapperS.of(termsChange).<Product>map("getProduct", termsChangeInstruction -> termsChangeInstruction.getProduct())).getOrDefault(false)) {
					return MapperS.of(termsChange).<Product>map("getProduct", termsChangeInstruction -> termsChangeInstruction.getProduct());
				}
				else {
					return MapperS.of(before).<Trade>map("getTrade", tradeState -> tradeState.getTrade()).<TradableProduct>map("getTradableProduct", trade -> trade.getTradableProduct()).<Product>map("getProduct", tradableProduct -> tradableProduct.getProduct());
				}
			});
		}
		
		@Override
		protected Mapper<? extends AncillaryParty> newAncillaryParty(TermsChangeInstruction termsChange, TradeState before) {
			return MapperUtils.runMultiPolymorphic(() -> {
				if (exists(MapperS.of(termsChange).<AncillaryParty>mapC("getAncillaryParty", termsChangeInstruction -> termsChangeInstruction.getAncillaryParty())).getOrDefault(false)) {
					return MapperS.of(termsChange).<AncillaryParty>mapC("getAncillaryParty", termsChangeInstruction -> termsChangeInstruction.getAncillaryParty());
				}
				else {
					return MapperS.of(before).<Trade>map("getTrade", tradeState -> tradeState.getTrade()).<TradableProduct>map("getTradableProduct", trade -> trade.getTradableProduct()).<AncillaryParty>mapC("getAncillaryParty", tradableProduct -> tradableProduct.getAncillaryParty());
				}
			});
		}
		
		@Override
		protected Mapper<NotionalAdjustmentEnum> newAdjustment(TermsChangeInstruction termsChange, TradeState before) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(termsChange).<NotionalAdjustmentEnum>map("getAdjustment", termsChangeInstruction -> termsChangeInstruction.getAdjustment())).getOrDefault(false)) {
					return MapperS.of(termsChange).<NotionalAdjustmentEnum>map("getAdjustment", termsChangeInstruction -> termsChangeInstruction.getAdjustment());
				}
				else {
					return MapperS.of(before).<Trade>map("getTrade", tradeState -> tradeState.getTrade()).<TradableProduct>map("getTradableProduct", trade -> trade.getTradableProduct()).<NotionalAdjustmentEnum>map("getAdjustment", tradableProduct -> tradableProduct.getAdjustment());
				}
			});
		}
	}
}
