package cdm.base.datetime.functions;

import cdm.base.datetime.BusinessCenterEnum;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import com.rosetta.model.lib.records.Date;
import java.util.ArrayList;
import java.util.List;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(GenerateDateList.GenerateDateListDefault.class)
public abstract class GenerateDateList implements RosettaFunction {
	
	// RosettaFunction dependencies
	//
	@Inject protected AddBusinessDays addBusinessDays;
	@Inject protected AppendDateToList appendDateToList;
	@Inject protected cdm.base.datetime.functions.GenerateDateList generateDateList;
	@Inject protected IsBusinessDay isBusinessDay;

	/**
	* @param startDate Start of the date range to be generated.
	* @param endDate End of the date range to be generated
	* @param businessCenters Business centers to be used to generate the list of good business days
	* @return dateList Resulting list of good business days.
	*/
	public List<Date> evaluate(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters) {
		List<Date> dateList = doEvaluate(startDate, endDate, businessCenters);
		
		return dateList;
	}

	protected abstract List<Date> doEvaluate(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters);

	protected abstract Mapper<Boolean> active(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters);

	protected abstract Mapper<Boolean> isGoodBusinessDay(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters);

	protected abstract Mapper<Date> priorDate(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters);

	protected abstract Mapper<Date> priorList(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters);

	protected abstract Mapper<Date> newList(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters);

	public static class GenerateDateListDefault extends GenerateDateList {
		@Override
		protected List<Date> doEvaluate(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters) {
			List<Date> dateList = new ArrayList<>();
			return assignOutput(dateList, startDate, endDate, businessCenters);
		}
		
		protected List<Date> assignOutput(List<Date> dateList, Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters) {
			List<Date> addVar = MapperUtils.runMulti(() -> {
				if (MapperS.of(active(startDate, endDate, businessCenters).get()).getOrDefault(false)) {
					return MapperC.<Date>of(newList(startDate, endDate, businessCenters).getMulti());
				}
				else {
					return null;
				}
			}).getMulti();
			dateList.addAll(addVar);
			
			return dateList;
		}
		
		@Override
		protected Mapper<Boolean> active(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters) {
			return lessThanEquals(MapperS.of(startDate), MapperS.of(endDate), CardinalityOperator.All);
		}
		
		@Override
		protected Mapper<Boolean> isGoodBusinessDay(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters) {
			return MapperS.of(isBusinessDay.evaluate(MapperS.of(endDate).get(), MapperC.<BusinessCenterEnum>of(businessCenters).getMulti()));
		}
		
		@Override
		protected Mapper<Date> priorDate(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters) {
			return MapperS.of(addBusinessDays.evaluate(MapperS.of(endDate).get(), MapperS.of(Integer.valueOf(-1)).get(), MapperC.<BusinessCenterEnum>of(businessCenters).getMulti()));
		}
		
		@Override
		protected Mapper<Date> priorList(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters) {
			return MapperC.<Date>of(generateDateList.evaluate(MapperS.of(startDate).get(), MapperS.of(priorDate(startDate, endDate, businessCenters).get()).get(), MapperC.<BusinessCenterEnum>of(businessCenters).getMulti()));
		}
		
		@Override
		protected Mapper<Date> newList(Date startDate, Date endDate, List<BusinessCenterEnum> businessCenters) {
			return MapperUtils.runMulti(() -> {
				if (MapperS.of(isGoodBusinessDay(startDate, endDate, businessCenters).get()).getOrDefault(false)) {
					return MapperC.<Date>of(appendDateToList.evaluate(MapperC.<Date>of(priorList(startDate, endDate, businessCenters).getMulti()).getMulti(), MapperS.of(endDate).get()));
				}
				else {
					return MapperC.<Date>of(priorList(startDate, endDate, businessCenters).getMulti());
				}
			});
		}
	}
}
