/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server.handler.gzip;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.WritePendingException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.zip.CRC32;
import java.util.zip.Deflater;
import org.eclipse.jetty.http.CompressedContentFormat;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpFields;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.PreEncodedHttpField;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.Response;
import org.eclipse.jetty.server.handler.gzip.GzipFactory;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.IteratingCallback;
import org.eclipse.jetty.util.IteratingNestedCallback;
import org.eclipse.jetty.util.compression.CompressionPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GzipResponse
extends Response.Wrapper {
    public static Logger LOG = LoggerFactory.getLogger(GzipResponse.class);
    private static final byte OS_UNKNOWN = -1;
    private static final byte[] GZIP_HEADER = new byte[]{31, -117, 8, 0, 0, 0, 0, 0, 0, -1};
    public static final int GZIP_TRAILER_SIZE = 8;
    public static final HttpField VARY_ACCEPT_ENCODING = new PreEncodedHttpField(HttpHeader.VARY, HttpHeader.ACCEPT_ENCODING.asString());
    private final AtomicReference<GZState> _state = new AtomicReference<GZState>(GZState.MIGHT_COMPRESS);
    private final CRC32 _crc = new CRC32();
    private final GzipFactory _factory;
    private final HttpField _vary;
    private final int _bufferSize;
    private final boolean _syncFlush;
    private CompressionPool.Entry _deflaterEntry;
    private ByteBuffer _buffer;

    public GzipResponse(Request request, Response wrapped, GzipFactory factory, HttpField vary, int bufferSize, boolean syncFlush) {
        super(request, wrapped);
        this._factory = factory;
        this._vary = vary;
        this._bufferSize = Math.max(GZIP_HEADER.length + 8, bufferSize);
        this._syncFlush = syncFlush;
    }

    @Override
    public void write(boolean last, ByteBuffer content, Callback callback) {
        switch (this._state.get()) {
            case MIGHT_COMPRESS: {
                this.commit(last, callback, content);
                break;
            }
            case NOT_COMPRESSING: {
                super.write(last, content, callback);
                break;
            }
            case COMMITTING: {
                callback.failed((Throwable)new WritePendingException());
                break;
            }
            case COMPRESSING: {
                this.gzip(last, callback, content);
                break;
            }
            default: {
                callback.failed((Throwable)new IllegalStateException("state=" + String.valueOf((Object)this._state.get())));
            }
        }
    }

    private void addTrailer(ByteBuffer outputBuffer) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("addTrailer: _crc={}, _totalIn={})", (Object)this._crc.getValue(), (Object)((Deflater)this._deflaterEntry.get()).getTotalIn());
        }
        outputBuffer.putInt((int)this._crc.getValue());
        outputBuffer.putInt(((Deflater)this._deflaterEntry.get()).getTotalIn());
    }

    private void gzip(boolean complete, Callback callback, ByteBuffer content) {
        if (content != null || complete) {
            new GzipBufferCB(complete, callback, content).iterate();
        } else {
            callback.succeeded();
        }
    }

    protected void commit(boolean last, Callback callback, ByteBuffer content) {
        String baseType;
        if (LOG.isDebugEnabled()) {
            LOG.debug("commit(last={}, callback={}, content={})", new Object[]{last, callback, BufferUtil.toDetailString((ByteBuffer)content)});
        }
        GzipResponse response = this;
        Request request = response.getRequest();
        int sc = response.getStatus();
        if (sc > 0 && (sc < 200 || sc == 204 || sc == 205 || sc >= 300)) {
            LOG.debug("{} exclude by status {}", (Object)this, (Object)sc);
            this.noCompression();
            if (sc == 304) {
                String requestEtags = (String)request.getAttribute("o.e.j.s.h.gzip.GzipHandler.etag");
                String responseEtag = response.getHeaders().get(HttpHeader.ETAG);
                if (requestEtags != null && responseEtag != null) {
                    String responseEtagGzip = this.etagGzip(responseEtag);
                    if (requestEtags.contains(responseEtagGzip)) {
                        response.getHeaders().put(HttpHeader.ETAG, responseEtagGzip);
                    }
                    if (this._vary != null) {
                        response.getHeaders().ensureField(this._vary);
                    }
                }
            }
            super.write(last, content, callback);
            return;
        }
        String ct = response.getHeaders().get(HttpHeader.CONTENT_TYPE);
        if (ct != null && !this._factory.isMimeTypeGzipable(baseType = HttpField.valueParameters((String)ct, null))) {
            LOG.debug("{} exclude by mimeType {}", (Object)this, (Object)ct);
            this.noCompression();
            super.write(last, content, callback);
            return;
        }
        HttpFields.Mutable fields = response.getHeaders();
        String ce = fields.get(HttpHeader.CONTENT_ENCODING);
        if (ce != null) {
            LOG.debug("{} exclude by content-encoding {}", (Object)this, (Object)ce);
            this.noCompression();
            super.write(last, content, callback);
            return;
        }
        if (this._state.compareAndSet(GZState.MIGHT_COMPRESS, GZState.COMMITTING)) {
            long contentLength;
            if (this._vary != null) {
                fields.ensureField(this._vary);
            }
            if ((contentLength = response.getHeaders().getLongField(HttpHeader.CONTENT_LENGTH)) < 0L && last) {
                contentLength = BufferUtil.length((ByteBuffer)content);
            }
            this._deflaterEntry = this._factory.getDeflaterEntry(request, contentLength);
            if (this._deflaterEntry == null) {
                LOG.debug("{} exclude no deflater", (Object)this);
                this._state.set(GZState.NOT_COMPRESSING);
                super.write(last, content, callback);
                return;
            }
            fields.put(CompressedContentFormat.GZIP.getContentEncoding());
            this._crc.reset();
            response.getHeaders().remove(HttpHeader.CONTENT_LENGTH);
            String etag = fields.get(HttpHeader.ETAG);
            if (etag != null) {
                fields.put(HttpHeader.ETAG, this.etagGzip(etag));
            }
            LOG.debug("{} compressing {}", (Object)this, (Object)this._deflaterEntry);
            this._state.set(GZState.COMPRESSING);
            if (BufferUtil.isEmpty((ByteBuffer)content)) {
                super.write(last, content, callback);
            } else {
                this.gzip(last, callback, content);
            }
        } else {
            callback.failed((Throwable)new WritePendingException());
        }
    }

    private String etagGzip(String etag) {
        return CompressedContentFormat.GZIP.etag(etag);
    }

    public void noCompression() {
        block4: while (true) {
            switch (this._state.get()) {
                case NOT_COMPRESSING: {
                    return;
                }
                case MIGHT_COMPRESS: {
                    if (!this._state.compareAndSet(GZState.MIGHT_COMPRESS, GZState.NOT_COMPRESSING)) continue block4;
                    return;
                }
            }
            break;
        }
        throw new IllegalStateException(this._state.get().toString());
    }

    private static enum GZState {
        MIGHT_COMPRESS,
        NOT_COMPRESSING,
        COMMITTING,
        COMPRESSING,
        FINISHING,
        FINISHED;

    }

    private class GzipBufferCB
    extends IteratingNestedCallback {
        private final ByteBuffer _content;
        private final boolean _last;

        public GzipBufferCB(boolean complete, Callback callback, ByteBuffer content) {
            super(callback);
            this._content = content;
            this._last = complete;
            if (this._content != null) {
                GzipResponse.this._crc.update(this._content.slice());
                Deflater deflater = (Deflater)GzipResponse.this._deflaterEntry.get();
                deflater.setInput(this._content);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("GzipBufferCB(complete={}, callback={}, content={})", new Object[]{complete, callback, BufferUtil.toDetailString((ByteBuffer)content)});
            }
        }

        protected void onCompleteFailure(Throwable x) {
            if (GzipResponse.this._deflaterEntry != null) {
                GzipResponse.this._deflaterEntry.release();
                GzipResponse.this._deflaterEntry = null;
            }
            super.onCompleteFailure(x);
        }

        protected IteratingCallback.Action process() throws Exception {
            GZState gzstate;
            if (LOG.isDebugEnabled()) {
                LOG.debug("GzipBufferCB.process(): _last={}, _buffer={}, _content={}", new Object[]{this._last, BufferUtil.toDetailString((ByteBuffer)GzipResponse.this._buffer), BufferUtil.toDetailString((ByteBuffer)this._content)});
            }
            if ((gzstate = GzipResponse.this._state.get()) == GZState.FINISHED) {
                this.cleanup();
                return IteratingCallback.Action.SUCCEEDED;
            }
            if (GzipResponse.this._buffer == null) {
                GzipResponse.this._buffer = GzipResponse.this.getRequest().getComponents().getByteBufferPool().acquire(GzipResponse.this._bufferSize, false);
                GzipResponse.this._buffer.order(ByteOrder.LITTLE_ENDIAN);
                BufferUtil.flipToFill((ByteBuffer)GzipResponse.this._buffer);
                GzipResponse.this._buffer.put(GZIP_HEADER, 0, GZIP_HEADER.length);
            } else {
                BufferUtil.clearToFill((ByteBuffer)GzipResponse.this._buffer);
            }
            Deflater deflater = (Deflater)GzipResponse.this._deflaterEntry.get();
            return switch (gzstate) {
                case GZState.COMPRESSING -> this.compressing(deflater, GzipResponse.this._buffer);
                case GZState.FINISHING -> this.finishing(deflater, GzipResponse.this._buffer);
                default -> throw new IllegalStateException("Unexpected state [" + String.valueOf((Object)GzipResponse.this._state.get()) + "]");
            };
        }

        private void cleanup() {
            if (GzipResponse.this._deflaterEntry != null) {
                GzipResponse.this._deflaterEntry.release();
                GzipResponse.this._deflaterEntry = null;
            }
            if (GzipResponse.this._buffer != null) {
                GzipResponse.this.getRequest().getComponents().getByteBufferPool().release(GzipResponse.this._buffer);
                GzipResponse.this._buffer = null;
            }
        }

        private int getFlushMode() {
            return GzipResponse.this._syncFlush ? 2 : 0;
        }

        private IteratingCallback.Action compressing(Deflater deflater, ByteBuffer outputBuffer) {
            int len;
            if (LOG.isDebugEnabled()) {
                LOG.debug("compressing() deflater={}, outputBuffer={}", (Object)deflater, (Object)BufferUtil.toDetailString((ByteBuffer)outputBuffer));
            }
            if (!deflater.finished() && !deflater.needsInput() && (len = deflater.deflate(outputBuffer, this.getFlushMode())) > 0) {
                BufferUtil.flipToFlush((ByteBuffer)outputBuffer, (int)0);
                this.write(false, outputBuffer);
                return IteratingCallback.Action.SCHEDULED;
            }
            if (this._last) {
                GzipResponse.this._state.set(GZState.FINISHING);
                deflater.finish();
                return this.finishing(deflater, outputBuffer);
            }
            BufferUtil.flipToFlush((ByteBuffer)outputBuffer, (int)0);
            if (outputBuffer.hasRemaining()) {
                this.write(false, outputBuffer);
                return IteratingCallback.Action.SCHEDULED;
            }
            if (BufferUtil.isEmpty((ByteBuffer)this._content)) {
                return IteratingCallback.Action.SUCCEEDED;
            }
            throw new AssertionError((Object)("No progress on deflate made for " + String.valueOf((Object)this)));
        }

        private IteratingCallback.Action finishing(Deflater deflater, ByteBuffer outputBuffer) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("finishing() deflater={}, outputBuffer={}", (Object)deflater, (Object)BufferUtil.toDetailString((ByteBuffer)outputBuffer));
            }
            if (!deflater.finished()) {
                int len = deflater.deflate(outputBuffer, this.getFlushMode());
                if (deflater.finished() && outputBuffer.remaining() >= 8) {
                    GzipResponse.this._state.set(GZState.FINISHED);
                    GzipResponse.this.addTrailer(outputBuffer);
                    BufferUtil.flipToFlush((ByteBuffer)outputBuffer, (int)0);
                    this.write(true, outputBuffer);
                    return IteratingCallback.Action.SCHEDULED;
                }
                if (len > 0) {
                    BufferUtil.flipToFlush((ByteBuffer)outputBuffer, (int)0);
                    this.write(false, outputBuffer);
                    return IteratingCallback.Action.SCHEDULED;
                }
                throw new AssertionError((Object)("No progress on deflate made for " + String.valueOf((Object)this)));
            }
            GzipResponse.this._state.set(GZState.FINISHED);
            GzipResponse.this.addTrailer(outputBuffer);
            BufferUtil.flipToFlush((ByteBuffer)outputBuffer, (int)0);
            this.write(true, outputBuffer);
            return IteratingCallback.Action.SCHEDULED;
        }

        private void write(boolean last, ByteBuffer outputBuffer) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("write() last={}, outputBuffer={}", (Object)last, (Object)BufferUtil.toDetailString((ByteBuffer)outputBuffer));
            }
            GzipResponse.super.write(last, outputBuffer, (Callback)this);
        }

        public String toString() {
            return String.format("%s[content=%s last=%b buffer=%s deflate=%s %s]", new Object[]{super.toString(), BufferUtil.toDetailString((ByteBuffer)this._content), this._last, BufferUtil.toDetailString((ByteBuffer)GzipResponse.this._buffer), GzipResponse.this._deflaterEntry, GzipResponse.this._state.get()});
        }
    }
}

