/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server.handler.gzip;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.zip.Deflater;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.http.HttpVersion;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.http.PathMap;
import org.eclipse.jetty.server.HttpOutput;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.HandlerWrapper;
import org.eclipse.jetty.server.handler.gzip.GzipFactory;
import org.eclipse.jetty.server.handler.gzip.GzipHttpOutputInterceptor;
import org.eclipse.jetty.util.ConcurrentHashSet;
import org.eclipse.jetty.util.URIUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class GzipHandler
extends HandlerWrapper
implements GzipFactory {
    private static final Logger LOG = Log.getLogger(GzipHandler.class);
    public static final String GZIP = "gzip";
    public static final String DEFLATE = "deflate";
    public static final String ETAG_GZIP = "--gzip";
    public static final String ETAG = "o.e.j.s.Gzip.ETag";
    public static final int DEFAULT_MIN_GZIP_SIZE = 16;
    private int _minGzipSize = 16;
    private int _compressionLevel = -1;
    private boolean _checkGzExists = true;
    private final ThreadLocal<Deflater> _deflater = new ThreadLocal();
    private final Set<String> _includedMethods = new HashSet<String>();
    private final Set<Pattern> _excludedAgentPatterns = new HashSet<Pattern>();
    private final PathMap<Boolean> _excludedPaths = new PathMap();
    private final PathMap<Boolean> _includedPaths = new PathMap();
    private final Set<String> _excludedMimeTypes = new HashSet<String>();
    private final Set<String> _includedMimeTypes = new HashSet<String>();
    private HttpField _vary;
    private final Set<String> _uaCache = new ConcurrentHashSet<String>();
    private int _uaCacheSize = 1024;

    public GzipHandler() {
        this._includedMethods.add(HttpMethod.GET.asString());
        for (String type : MimeTypes.getKnownMimeTypes()) {
            if (!type.startsWith("image/") && !type.startsWith("audio/") && !type.startsWith("video/")) continue;
            this._excludedMimeTypes.add(type);
        }
        this._excludedMimeTypes.add("application/compress");
        this._excludedMimeTypes.add("application/zip");
        this._excludedMimeTypes.add("application/gzip");
        this._excludedMimeTypes.add("application/bzip2");
        this._excludedMimeTypes.add("application/x-rar-compressed");
        LOG.debug("{} excluding mimes {}", this, this._excludedMimeTypes);
        this._excludedAgentPatterns.add(Pattern.compile(".*MSIE 6.0.*"));
    }

    public void addExcludedAgentPatterns(String ... patterns) {
        for (String s : patterns) {
            this._excludedAgentPatterns.add(Pattern.compile(s));
        }
    }

    public void addExcludedMimeTypes(String ... types) {
        this._excludedMimeTypes.addAll(Arrays.asList(types));
    }

    public void addExcludedPaths(String ... pathspecs) {
        for (String ps : pathspecs) {
            this._excludedPaths.put(ps, Boolean.TRUE);
        }
    }

    public void addIncludedMethods(String ... methods) {
        for (String m : methods) {
            this._includedMethods.add(m);
        }
    }

    public void addIncludedMimeTypes(String ... types) {
        this._includedMimeTypes.addAll(Arrays.asList(types));
    }

    public void addIncludedPaths(String ... pathspecs) {
        for (String ps : pathspecs) {
            this._includedPaths.put(ps, Boolean.TRUE);
        }
    }

    public boolean getCheckGzExists() {
        return this._checkGzExists;
    }

    public int getCompressionLevel() {
        return this._compressionLevel;
    }

    @Override
    public Deflater getDeflater(Request request, long content_length) {
        Deflater df;
        String ua = request.getHttpFields().get(HttpHeader.USER_AGENT);
        if (ua != null && this.isExcludedAgent(ua)) {
            LOG.debug("{} excluded user agent {}", this, request);
            return null;
        }
        if (content_length >= 0L && content_length < (long)this._minGzipSize) {
            LOG.debug("{} excluded minGzipSize {}", this, request);
            return null;
        }
        if (request.getHttpVersion() != HttpVersion.HTTP_2) {
            HttpField accept = request.getHttpFields().getField(HttpHeader.ACCEPT_ENCODING);
            if (accept == null) {
                LOG.debug("{} excluded !accept {}", this, request);
                return null;
            }
            boolean gzip = accept.contains(GZIP);
            if (!gzip) {
                LOG.debug("{} excluded not gzip accept {}", this, request);
                return null;
            }
        }
        if ((df = this._deflater.get()) == null) {
            df = new Deflater(this._compressionLevel, true);
        } else {
            this._deflater.set(null);
        }
        return df;
    }

    public String[] getExcludedAgentPatterns() {
        Pattern[] ps = this._excludedAgentPatterns.toArray(new Pattern[this._excludedAgentPatterns.size()]);
        String[] s = new String[ps.length];
        int i = 0;
        for (Pattern p : ps) {
            s[i++] = p.toString();
        }
        return s;
    }

    public String[] getExcludedMimeTypes() {
        return this._excludedMimeTypes.toArray(new String[this._excludedMimeTypes.size()]);
    }

    public String[] getExcludedPaths() {
        String[] ps = this._excludedPaths.keySet().toArray(new String[this._excludedPaths.size()]);
        return ps;
    }

    public String[] getIncludedMimeTypes() {
        return this._includedMimeTypes.toArray(new String[this._includedMimeTypes.size()]);
    }

    public String[] getIncludedPaths() {
        String[] ps = this._includedPaths.keySet().toArray(new String[this._includedPaths.size()]);
        return ps;
    }

    public String[] getMethods() {
        return this._includedMethods.toArray(new String[this._includedMethods.size()]);
    }

    public int getMinGzipSize() {
        return this._minGzipSize;
    }

    @Override
    protected void doStart() throws Exception {
        this._vary = this._excludedAgentPatterns.size() > 0 ? GzipHttpOutputInterceptor.VARY_ACCEPT_ENCODING_USER_AGENT : GzipHttpOutputInterceptor.VARY_ACCEPT_ENCODING;
        super.doStart();
    }

    @Override
    public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
        File gz;
        String realpath;
        String mimeType;
        ServletContext context = baseRequest.getServletContext();
        String path = context == null ? baseRequest.getRequestURI() : URIUtil.addPaths(baseRequest.getServletPath(), baseRequest.getPathInfo());
        LOG.debug("{} handle {} in {}", this, baseRequest, context);
        HttpOutput out = baseRequest.getResponse().getHttpOutput();
        for (HttpOutput.Interceptor interceptor = out.getInterceptor(); interceptor != null; interceptor = interceptor.getNextInterceptor()) {
            if (!(interceptor instanceof GzipHttpOutputInterceptor)) continue;
            LOG.debug("{} already intercepting {}", this, request);
            this._handler.handle(target, baseRequest, request, response);
            return;
        }
        if (!this._includedMethods.contains(baseRequest.getMethod())) {
            LOG.debug("{} excluded by method {}", this, request);
            this._handler.handle(target, baseRequest, request, response);
            return;
        }
        if (!this.isPathGzipable(path)) {
            LOG.debug("{} excluded by path {}", this, request);
            this._handler.handle(target, baseRequest, request, response);
            return;
        }
        String string = mimeType = context == null ? null : context.getMimeType(path);
        if (mimeType != null && !this.isMimeTypeGzipable(mimeType = MimeTypes.getContentTypeWithoutCharset(mimeType))) {
            LOG.debug("{} excluded by path suffix mime type {}", this, request);
            this._handler.handle(target, baseRequest, request, response);
            return;
        }
        if (this._checkGzExists && context != null && (realpath = request.getServletContext().getRealPath(path)) != null && (gz = new File(realpath + ".gz")).exists()) {
            LOG.debug("{} gzip exists {}", this, request);
            this._handler.handle(target, baseRequest, request, response);
            return;
        }
        String etag = request.getHeader("If-None-Match");
        if (etag != null && etag.contains(ETAG_GZIP)) {
            request.setAttribute(ETAG, etag.replace(ETAG_GZIP, ""));
        }
        out.setInterceptor(new GzipHttpOutputInterceptor(this, this._vary, baseRequest.getHttpChannel(), out.getInterceptor()));
        this._handler.handle(target, baseRequest, request, response);
    }

    protected boolean isExcludedAgent(String ua) {
        if (ua == null) {
            return false;
        }
        if (this._excludedAgentPatterns != null) {
            if (this._uaCache.contains(ua)) {
                return true;
            }
            for (Pattern pattern : this._excludedAgentPatterns) {
                if (!pattern.matcher(ua).matches()) continue;
                if (this._uaCache.size() > this._uaCacheSize) {
                    this._uaCache.clear();
                }
                this._uaCache.add(ua);
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean isMimeTypeGzipable(String mimetype) {
        if (this._includedMimeTypes.size() > 0 && this._includedMimeTypes.contains(mimetype)) {
            return true;
        }
        return !this._excludedMimeTypes.contains(mimetype);
    }

    protected boolean isPathGzipable(String requestURI) {
        if (requestURI == null) {
            return true;
        }
        if (this._includedPaths.size() > 0 && this._includedPaths.containsMatch(requestURI)) {
            return true;
        }
        return this._excludedPaths.size() <= 0 || !this._excludedPaths.containsMatch(requestURI);
    }

    @Override
    public void recycle(Deflater deflater) {
        deflater.reset();
        if (this._deflater.get() == null) {
            this._deflater.set(deflater);
        }
    }

    public void setCheckGzExists(boolean checkGzExists) {
        this._checkGzExists = checkGzExists;
    }

    public void setCompressionLevel(int compressionLevel) {
        this._compressionLevel = compressionLevel;
    }

    public void setExcludedAgentPatterns(String ... patterns) {
        this._excludedAgentPatterns.clear();
        this.addExcludedAgentPatterns(patterns);
    }

    public void setExcludedMimeTypes(String ... types) {
        this._excludedMimeTypes.clear();
        this.addExcludedMimeTypes(types);
    }

    public void setExcludedPaths(String ... pathspecs) {
        this._excludedPaths.clear();
        this.addExcludedPaths(pathspecs);
    }

    public void setIncludedMethods(String ... methods) {
        this._includedMethods.clear();
        this.addIncludedMethods(methods);
    }

    public void setIncludedMimeTypes(String ... types) {
        this._includedMimeTypes.clear();
        this.addIncludedMimeTypes(types);
    }

    public void setIncludedPaths(String ... pathspecs) {
        this._includedPaths.clear();
        this.addIncludedPaths(pathspecs);
    }

    public void setMinGzipSize(int minGzipSize) {
        this._minGzipSize = minGzipSize;
    }
}

