/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements.tcp.netty;

import io.netty.channel.Channel;
import io.netty.handler.ssl.SslHandler;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.security.Principal;
import java.security.cert.X509Certificate;
import java.util.concurrent.CancellationException;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLPeerUnverifiedException;
import org.eclipse.californium.elements.EndpointContext;
import org.eclipse.californium.elements.EndpointContextMatcher;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.TlsEndpointContext;
import org.eclipse.californium.elements.auth.X509CertPath;
import org.eclipse.californium.elements.config.CertificateAuthenticationMode;
import org.eclipse.californium.elements.config.Configuration;
import org.eclipse.californium.elements.config.TcpConfig;
import org.eclipse.californium.elements.tcp.netty.TcpClientConnector;
import org.eclipse.californium.elements.tcp.netty.TlsContextUtil;
import org.eclipse.californium.elements.util.CertPathUtil;
import org.eclipse.californium.elements.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TlsClientConnector
extends TcpClientConnector {
    private static final Logger LOG = LoggerFactory.getLogger(TlsClientConnector.class);
    private final SSLContext sslContext;
    private final String[] weakCipherSuites;
    private final int handshakeTimeoutMillis;
    private final boolean verifyServerSubject;

    public TlsClientConnector(SSLContext sslContext, Configuration configuration) {
        super(configuration, new TlsContextUtil(CertificateAuthenticationMode.NEEDED));
        this.sslContext = sslContext;
        this.handshakeTimeoutMillis = configuration.getTimeAsInt(TcpConfig.TLS_HANDSHAKE_TIMEOUT, TimeUnit.MILLISECONDS);
        this.verifyServerSubject = configuration.get(TcpConfig.TLS_VERIFY_SERVER_CERTIFICATES_SUBJECT);
        this.weakCipherSuites = TlsContextUtil.getWeakCipherSuites(sslContext);
    }

    @Override
    protected void send(final Channel channel, final EndpointContextMatcher endpointMatcher, final RawData msg) {
        SslHandler sslHandler = channel.pipeline().get(SslHandler.class);
        if (sslHandler == null) {
            msg.onError(new IllegalStateException("Missing SslHandler"));
        } else {
            Future<Channel> handshakeFuture = sslHandler.handshakeFuture();
            handshakeFuture.addListener(new GenericFutureListener<Future<Channel>>(){

                @Override
                public void operationComplete(Future<Channel> future) throws Exception {
                    if (future.isSuccess()) {
                        EndpointContext context = TlsClientConnector.this.contextUtil.buildEndpointContext(channel);
                        if (context == null || context.get(TlsEndpointContext.KEY_SESSION_ID) == null) {
                            msg.onError(new IllegalStateException("Missing TlsEndpointContext " + context));
                            return;
                        }
                        if (TlsClientConnector.this.verifyServerSubject) {
                            try {
                                Principal principal = context.getPeerIdentity();
                                if (principal instanceof X509CertPath) {
                                    X509Certificate target = ((X509CertPath)principal).getTarget();
                                    InetSocketAddress address = context.getPeerAddress();
                                    String hostname = context.getVirtualHost();
                                    TlsClientConnector.this.verifyCertificatesSubject(hostname, address, target);
                                }
                            }
                            catch (SSLPeerUnverifiedException ex) {
                                msg.onError(ex);
                                return;
                            }
                        }
                        TlsClientConnector.super.send(future.getNow(), endpointMatcher, msg);
                    } else if (future.isCancelled()) {
                        msg.onError(new CancellationException());
                    } else {
                        msg.onError(future.cause());
                    }
                }
            });
        }
    }

    @Override
    protected void onNewChannelCreated(SocketAddress remote, Channel ch) {
        SSLEngine sslEngine = this.createSllEngine(remote);
        sslEngine.setUseClientMode(true);
        if (this.weakCipherSuites != null) {
            sslEngine.setEnabledCipherSuites(this.weakCipherSuites);
        }
        SslHandler sslHandler = new SslHandler(sslEngine);
        sslHandler.setHandshakeTimeoutMillis(this.handshakeTimeoutMillis);
        ch.pipeline().addFirst(sslHandler);
    }

    @Override
    public String getProtocol() {
        return "TLS";
    }

    private SSLEngine createSllEngine(SocketAddress remoteAddress) {
        if (remoteAddress instanceof InetSocketAddress) {
            LOG.info("Connection to inet {}", StringUtil.toLog(remoteAddress));
            InetSocketAddress remote = (InetSocketAddress)remoteAddress;
            return this.sslContext.createSSLEngine(remote.getAddress().getHostAddress(), remote.getPort());
        }
        LOG.info("Connection to {}", StringUtil.toLog(remoteAddress));
        return this.sslContext.createSSLEngine();
    }

    private void verifyCertificatesSubject(String serverName, InetSocketAddress peer, X509Certificate certificate) throws SSLPeerUnverifiedException {
        String cn;
        if (certificate == null) {
            throw new NullPointerException("Certficate must not be null!");
        }
        if (serverName == null && peer == null) {
            return;
        }
        String literalIp = null;
        String hostname = serverName;
        if (peer != null) {
            InetAddress destination = peer.getAddress();
            if (destination != null) {
                literalIp = destination.getHostAddress();
            }
            if (hostname == null) {
                hostname = peer.getHostString();
            }
        }
        if (hostname != null && hostname.equals(literalIp)) {
            hostname = null;
        }
        if (hostname != null) {
            if (!CertPathUtil.matchDestination(certificate, hostname)) {
                cn = CertPathUtil.getSubjectsCn(certificate);
                LOG.debug("Certificate {} validation failed: destination doesn't match", (Object)cn);
                throw new SSLPeerUnverifiedException("Certificate " + cn + ": Destination '" + hostname + "' doesn't match!");
            }
        } else if (!CertPathUtil.matchLiteralIP(certificate, literalIp)) {
            cn = CertPathUtil.getSubjectsCn(certificate);
            LOG.debug("Certificate {} validation failed: literal IP doesn't match", (Object)cn);
            throw new SSLPeerUnverifiedException("Certificate " + cn + ": Literal IP " + literalIp + " doesn't match!");
        }
    }
}

