/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements.util;

import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivilegedAction;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import javax.crypto.Cipher;
import org.eclipse.californium.elements.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JceProviderUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(JceProviderUtil.class);
    private static volatile JceProviderUtil features;
    private static final String[] ED25519_ALIASES;
    private static final String[] ED448_ALIASES;
    private static final String[][] ALGORITHM_ALIASES;
    private static final String NET_I2P_CRYPTO_EDDSA = "net.i2p.crypto.eddsa";
    private static final String NET_I2P_CRYPTO_EDDSA_PROVIDER = "net.i2p.crypto.eddsa.EdDSASecurityProvider";
    private static final String BOUNCY_CASTLE_JCE_PROVIDER = "org.bouncycastle.jce.provider.BouncyCastleProvider";
    private static final String BOUNCY_CASTLE_JSSE_PROVIDER = "org.bouncycastle.jsse.provider.BouncyCastleJsseProvider";
    private static final String JSSE_PROVIDER_BOUNCY_CASTLE = "BCJSSE";
    private static final String AES = "AES";
    private final boolean useBc;
    private final boolean rsa;
    private final boolean ec;
    private final boolean ed25519;
    private final boolean ed448;
    private final boolean strongEncryption;
    private final String providerVersion;

    private static void doPrivileged() {
        AccessController.doPrivileged(new PrivilegedAction<Void>(){

            @Override
            public Void run() {
                try {
                    JceProviderUtil.setupJce();
                }
                catch (Throwable t) {
                    LOGGER.error("JCE:", t);
                }
                return null;
            }
        });
    }

    private static boolean isBouncyCastle(Provider provider) {
        return provider != null && provider.getName().equals("BC");
    }

    private static boolean isNetI2PEdDsa(Provider provider) {
        return provider != null && provider.getClass().getName().equals(NET_I2P_CRYPTO_EDDSA_PROVIDER);
    }

    private static void configure(Provider provider, String key, String value) {
        String current = provider.getProperty(key);
        if (!value.equals(current)) {
            provider.setProperty(key, value);
        }
    }

    private static Provider loadProvider(String clzName) {
        try {
            Class<?> clz = Class.forName(clzName);
            Provider provider = (Provider)clz.getConstructor(new Class[0]).newInstance(new Object[0]);
            LOGGER.debug("Loaded {}", (Object)clzName);
            return provider;
        }
        catch (Throwable e) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Loading {} failed!", (Object)clzName, (Object)e);
            } else {
                LOGGER.debug("Loading {} failed!", (Object)clzName);
            }
            return null;
        }
    }

    private static void setupLoggingBridge() {
        try {
            Class<?> clz = Class.forName("org.slf4j.bridge.SLF4JBridgeHandler");
            Method method = clz.getMethod("removeHandlersForRootLogger", new Class[0]);
            method.invoke(null, new Object[0]);
            method = clz.getMethod("install", new Class[0]);
            method.invoke(null, new Object[0]);
        }
        catch (Throwable e) {
            LOGGER.warn("Setup BC logging failed!", e);
        }
    }

    private static String setupNonBlockingSecureRandom() {
        String strongAlgorithms = Security.getProperty("securerandom.strongAlgorithms");
        if (strongAlgorithms != null) {
            if (strongAlgorithms.contains("NativePRNGBlocking")) {
                String weakerAlgorithms = strongAlgorithms.replaceAll("NativePRNGBlocking", "NativePRNGNonBlocking");
                Security.setProperty("securerandom.strongAlgorithms", weakerAlgorithms);
            } else {
                SecureRandom random = new SecureRandom();
                String defaultAlgorithm = random.getAlgorithm() + ":";
                if (!strongAlgorithms.contains(defaultAlgorithm)) {
                    String defaultProvider = random.getProvider().getName();
                    String weakerAlgorithms = defaultAlgorithm + defaultProvider + "," + strongAlgorithms;
                    Security.setProperty("securerandom.strongAlgorithms", weakerAlgorithms);
                } else {
                    LOGGER.info("Random: {} already in {}", (Object)defaultAlgorithm, (Object)strongAlgorithms);
                }
            }
        }
        return strongAlgorithms;
    }

    private static void setupJce() {
        boolean tryJce = false;
        boolean tryBc = false;
        boolean tryEd25519Java = false;
        boolean nonBlockingRandom = false;
        String jce = StringUtil.getConfiguration("CALIFORNIUM_JCE_PROVIDER");
        if (jce != null && !jce.isEmpty()) {
            LOGGER.info("JCE setup: {}", (Object)jce);
            if ("SYSTEM".equalsIgnoreCase(jce)) {
                tryJce = true;
            } else if ("BC".equalsIgnoreCase(jce)) {
                tryBc = true;
            } else if ("BC_NON_BLOCKING_RANDOM".equalsIgnoreCase(jce)) {
                tryBc = true;
                nonBlockingRandom = true;
            } else if ("I2P".equalsIgnoreCase(jce)) {
                tryEd25519Java = true;
            }
        } else {
            LOGGER.info("JCE default setup");
            tryJce = true;
            tryEd25519Java = true;
        }
        boolean found = false;
        Provider provider = null;
        try {
            KeyFactory factory = KeyFactory.getInstance("EdDSA");
            provider = factory.getProvider();
            if (tryJce) {
                found = true;
                LOGGER.trace("EdDSA from default jce {}", (Object)provider.getName());
            }
        }
        catch (NoSuchAlgorithmException factory) {
            // empty catch block
        }
        if (!found && tryBc) {
            if (JceProviderUtil.isBouncyCastle(provider)) {
                found = true;
                LOGGER.trace("EdDSA from BC");
            } else {
                JceProviderUtil.setupLoggingBridge();
                String strongAlgorithms = nonBlockingRandom ? JceProviderUtil.setupNonBlockingSecureRandom() : null;
                Provider newProvider2 = JceProviderUtil.loadProvider(BOUNCY_CASTLE_JCE_PROVIDER);
                if (newProvider2 != null) {
                    try {
                        KeyFactory.getInstance("EdDSA", newProvider2);
                        Security.removeProvider(newProvider2.getName());
                        Security.insertProviderAt(newProvider2, 1);
                        provider = newProvider2;
                        found = true;
                        SecureRandom start = new SecureRandom();
                        start.nextInt();
                        String algorithms = Security.getProperty("securerandom.strongAlgorithms");
                        if (algorithms == null) {
                            algorithms = "not available";
                        }
                        LOGGER.info("StrongRandom: {}", (Object)algorithms);
                        LOGGER.trace("EdDSA added from BC");
                    }
                    catch (SecurityException start) {
                    }
                    catch (NoSuchAlgorithmException start) {
                        // empty catch block
                    }
                }
                if (strongAlgorithms != null) {
                    Security.setProperty("securerandom.strongAlgorithms", strongAlgorithms);
                }
                if (found && Security.getProvider(JSSE_PROVIDER_BOUNCY_CASTLE) == null && (newProvider2 = JceProviderUtil.loadProvider(BOUNCY_CASTLE_JSSE_PROVIDER)) != null) {
                    Security.setProperty("ssl.KeyManagerFactory.algorithm", "PKIX");
                    Security.setProperty("ssl.TrustManagerFactory.algorithm", "PKIX");
                    try {
                        Security.insertProviderAt(newProvider2, 2);
                        LOGGER.trace("TLS from added BC");
                    }
                    catch (SecurityException start) {
                        // empty catch block
                    }
                }
            }
        }
        if (!found && tryEd25519Java) {
            if (JceProviderUtil.isNetI2PEdDsa(provider)) {
                found = true;
                LOGGER.trace("EdDSA from {}", (Object)NET_I2P_CRYPTO_EDDSA);
            } else {
                Provider newProvider = JceProviderUtil.loadProvider(NET_I2P_CRYPTO_EDDSA_PROVIDER);
                if (newProvider != null) {
                    try {
                        KeyFactory.getInstance("EdDSA", newProvider);
                        Security.removeProvider(newProvider.getName());
                        Security.addProvider(newProvider);
                        provider = newProvider;
                        found = true;
                        LOGGER.trace("EdDSA added from {}", (Object)NET_I2P_CRYPTO_EDDSA);
                    }
                    catch (SecurityException newProvider2) {
                    }
                    catch (NoSuchAlgorithmException newProvider2) {
                        // empty catch block
                    }
                }
            }
        }
        boolean ec = false;
        boolean rsa = false;
        String aesPermission = "not supported";
        int aesMaxAllowedKeyLength = 0;
        try {
            aesMaxAllowedKeyLength = Cipher.getMaxAllowedKeyLength(AES);
            aesPermission = aesMaxAllowedKeyLength == Integer.MAX_VALUE ? "not restricted" : "restricted to " + aesMaxAllowedKeyLength + " bits key length";
        }
        catch (NoSuchAlgorithmException noSuchAlgorithmException) {
            // empty catch block
        }
        LOGGER.debug("AES: {}", (Object)aesPermission);
        try {
            KeyFactory.getInstance("RSA");
            rsa = true;
        }
        catch (NoSuchAlgorithmException noSuchAlgorithmException) {
            // empty catch block
        }
        LOGGER.debug("RSA: {}", (Object)rsa);
        try {
            KeyFactory.getInstance("EC");
            ec = true;
        }
        catch (NoSuchAlgorithmException noSuchAlgorithmException) {
            // empty catch block
        }
        LOGGER.debug("EC: {}", (Object)ec);
        if (!LOGGER.isDebugEnabled()) {
            LOGGER.info("RSA: {}, EC: {}, AES: {}", rsa, ec, aesPermission);
        }
        String version = provider == null ? "n.a." : Double.toString(provider.getVersion());
        boolean ed25519 = false;
        boolean ed448 = false;
        if (found && provider != null) {
            if (JceProviderUtil.isBouncyCastle(provider)) {
                JceProviderUtil.configure(provider, "Alg.Alias.KeyFactory.OID.1.3.101.112", "Ed25519");
                JceProviderUtil.configure(provider, "Alg.Alias.KeyFactory.OID.1.3.101.113", "Ed448");
            } else if (JceProviderUtil.isNetI2PEdDsa(provider)) {
                JceProviderUtil.configure(provider, "Alg.Alias.KeyFactory.Ed25519", "EdDSA");
            }
            try {
                KeyFactory.getInstance("Ed25519");
                ed25519 = true;
            }
            catch (NoSuchAlgorithmException noSuchAlgorithmException) {
                // empty catch block
            }
            try {
                KeyFactory.getInstance("Ed448");
                ed448 = true;
            }
            catch (NoSuchAlgorithmException noSuchAlgorithmException) {
                // empty catch block
            }
            LOGGER.info("EdDSA supported by {}, Ed25519: {}, Ed448: {}", provider.getName(), ed25519, ed448);
        } else {
            provider = null;
            LOGGER.info("EdDSA not supported!");
        }
        JceProviderUtil newSupport = new JceProviderUtil(JceProviderUtil.isBouncyCastle(provider), rsa, ec, ed25519, ed448, aesMaxAllowedKeyLength >= 256, version);
        if (!newSupport.equals(features)) {
            features = newSupport;
        }
        LOGGER.info("JCE setup: {}, ready.", (Object)provider);
        if (LOGGER.isDebugEnabled()) {
            Provider[] providers = Security.getProviders();
            for (int index = 0; index < providers.length; ++index) {
                provider = providers[index];
                LOGGER.debug("Security Provider [{}]: {}.", (Object)index, (Object)provider);
            }
            LOGGER.trace("JCE setup callstack:", new Throwable("JCE setup"));
        }
    }

    public static void init() {
    }

    public static boolean usesBouncyCastle() {
        return JceProviderUtil.features.useBc;
    }

    public static boolean hasStrongEncryption() {
        return JceProviderUtil.features.strongEncryption;
    }

    public static boolean isSupported(String algorithm) {
        if ("EC".equalsIgnoreCase(algorithm)) {
            return JceProviderUtil.features.ec;
        }
        if ("RSA".equalsIgnoreCase(algorithm)) {
            return JceProviderUtil.features.rsa;
        }
        String oid = JceProviderUtil.getEdDsaStandardAlgorithmName(algorithm, null);
        if ("OID.1.3.101.112".equals(oid)) {
            return JceProviderUtil.features.ed25519;
        }
        if ("OID.1.3.101.113".equals(oid)) {
            return JceProviderUtil.features.ed448;
        }
        if ("EdDSA".equalsIgnoreCase(algorithm)) {
            return JceProviderUtil.features.ed25519 || JceProviderUtil.features.ed448;
        }
        return false;
    }

    public static String getEdDsaStandardAlgorithmName(String algorithm, String def) {
        if ("EdDSA".equalsIgnoreCase(algorithm)) {
            return "EdDSA";
        }
        if (StringUtil.containsIgnoreCase(ED25519_ALIASES, algorithm)) {
            return "OID.1.3.101.112";
        }
        if (StringUtil.containsIgnoreCase(ED448_ALIASES, algorithm)) {
            return "OID.1.3.101.113";
        }
        return def;
    }

    public static boolean equalKeyAlgorithmSynonyms(String keyAlgorithm1, String keyAlgorithm2) {
        if (keyAlgorithm1 != null && keyAlgorithm1.equals(keyAlgorithm2)) {
            return true;
        }
        for (String[] aliases : ALGORITHM_ALIASES) {
            if (!StringUtil.containsIgnoreCase(aliases, keyAlgorithm1) || !StringUtil.containsIgnoreCase(aliases, keyAlgorithm2)) continue;
            return true;
        }
        return false;
    }

    public static String getProviderVersion() {
        return JceProviderUtil.features.providerVersion;
    }

    private JceProviderUtil(boolean useBc, boolean rsa, boolean ec, boolean ed25519, boolean ed448, boolean strongEncryption, String providerVersion) {
        this.useBc = useBc;
        this.rsa = rsa;
        this.ec = ec;
        this.ed25519 = ed25519;
        this.ed448 = ed448;
        this.strongEncryption = strongEncryption;
        this.providerVersion = providerVersion;
    }

    public int hashCode() {
        int prime = 31;
        int result = 1;
        result = 31 * result + (this.ed25519 ? 41 : 37);
        result = 31 * result + (this.ed448 ? 41 : 37);
        result = 31 * result + (this.strongEncryption ? 41 : 37);
        result = 31 * result + (this.ec ? 41 : 37);
        result = 31 * result + (this.rsa ? 41 : 37);
        result = 31 * result + (this.useBc ? 41 : 37);
        result = 31 * result + this.providerVersion.hashCode();
        return result;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        JceProviderUtil other = (JceProviderUtil)obj;
        if (this.ed25519 != other.ed25519) {
            return false;
        }
        if (this.ed448 != other.ed448) {
            return false;
        }
        if (this.strongEncryption != other.strongEncryption) {
            return false;
        }
        if (this.ec != other.ec) {
            return false;
        }
        if (this.rsa != other.rsa) {
            return false;
        }
        if (this.useBc != other.useBc) {
            return false;
        }
        return this.providerVersion.equals(other.providerVersion);
    }

    static {
        ED25519_ALIASES = new String[]{"Ed25519", "1.3.101.112", "OID.1.3.101.112", "EdDSA", "Ed25519.v2"};
        ED448_ALIASES = new String[]{"Ed448", "1.3.101.113", "OID.1.3.101.113", "EdDSA", "Ed448.v2"};
        ALGORITHM_ALIASES = new String[][]{{"DH", "DiffieHellman"}, {"EC", "EC.v2"}, ED25519_ALIASES, ED448_ALIASES, {"X25519", "X25519.v2", "OID.1.3.101.110"}, {"X448", "X448.v2", "OID.1.3.101.111"}};
        try {
            Class.forName(AccessController.class.getName());
            JceProviderUtil.doPrivileged();
        }
        catch (ClassNotFoundException e) {
            try {
                JceProviderUtil.setupJce();
            }
            catch (Throwable t) {
                LOGGER.error("JCE:", t);
            }
        }
    }
}

