/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.security.auth.x500.X500Principal;
import org.eclipse.californium.elements.util.Asn1DerDecoder;
import org.eclipse.californium.elements.util.CertPathUtil;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.NoPublicAPI;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.dtls.SignatureAndHashAlgorithm;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NoPublicAPI
public final class CertificateRequest
extends HandshakeMessage {
    private static final Logger LOGGER = LoggerFactory.getLogger(CertificateRequest.class);
    private static final String THREE_TABS = "\t\t\t";
    private static final int CERTIFICATE_TYPES_LENGTH_BITS = 8;
    private static final int CERTIFICATE_TYPE_BITS = 8;
    private static final int SUPPORTED_SIGNATURE_LENGTH_BITS = 16;
    private static final int CERTIFICATE_AUTHORITIES_LENGTH_BITS = 16;
    private static final int CERTIFICATE_AUTHORITY_LENGTH_BITS = 16;
    private static final int SUPPORTED_SIGNATURE_BITS = 8;
    private static final int MAX_LENGTH_CERTIFICATE_AUTHORITIES = 65535;
    private final List<ClientCertificateType> certificateTypes = new ArrayList<ClientCertificateType>();
    private final List<SignatureAndHashAlgorithm> supportedSignatureAlgorithms = new ArrayList<SignatureAndHashAlgorithm>();
    private final List<X500Principal> certificateAuthorities = new ArrayList<X500Principal>();
    private int certificateAuthoritiesEncodedLength = 0;

    public CertificateRequest() {
    }

    public CertificateRequest(List<ClientCertificateType> certificateTypes, List<SignatureAndHashAlgorithm> supportedSignatureAlgorithms, List<X500Principal> certificateAuthorities) {
        if (certificateTypes != null) {
            this.certificateTypes.addAll(certificateTypes);
        }
        if (!supportedSignatureAlgorithms.isEmpty()) {
            this.supportedSignatureAlgorithms.addAll(supportedSignatureAlgorithms);
        }
        if (certificateAuthorities != null) {
            this.addCerticiateAuthorities(certificateAuthorities);
        }
    }

    @Override
    public HandshakeType getMessageType() {
        return HandshakeType.CERTIFICATE_REQUEST;
    }

    @Override
    public int getMessageLength() {
        return 1 + this.certificateTypes.size() + 2 + this.supportedSignatureAlgorithms.size() * 2 + 2 + this.certificateAuthoritiesEncodedLength;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        if (!this.certificateTypes.isEmpty()) {
            sb.append("\t\tClient certificate type:").append(StringUtil.lineSeparator());
            for (ClientCertificateType type : this.certificateTypes) {
                sb.append(THREE_TABS).append((Object)type).append(StringUtil.lineSeparator());
            }
        }
        if (!this.supportedSignatureAlgorithms.isEmpty()) {
            sb.append("\t\tSignature and hash algorithm:").append(StringUtil.lineSeparator());
            for (SignatureAndHashAlgorithm algo : this.supportedSignatureAlgorithms) {
                sb.append(THREE_TABS).append(algo).append(StringUtil.lineSeparator());
            }
        }
        if (!this.certificateAuthorities.isEmpty()) {
            sb.append("\t\tCertificate authorities:").append(StringUtil.lineSeparator());
            for (X500Principal subject : this.certificateAuthorities) {
                sb.append(THREE_TABS).append(subject.getName()).append(StringUtil.lineSeparator());
            }
        }
        return sb.toString();
    }

    @Override
    public byte[] fragmentToByteArray() {
        DatagramWriter writer = new DatagramWriter();
        writer.write(this.certificateTypes.size(), 8);
        for (ClientCertificateType certificateType : this.certificateTypes) {
            writer.write(certificateType.getCode(), 8);
        }
        writer.write(this.supportedSignatureAlgorithms.size() * 2, 16);
        for (SignatureAndHashAlgorithm signatureAndHashAlgorithm : this.supportedSignatureAlgorithms) {
            writer.write(signatureAndHashAlgorithm.getHash().getCode(), 8);
            writer.write(signatureAndHashAlgorithm.getSignature().getCode(), 8);
        }
        writer.write(this.certificateAuthoritiesEncodedLength, 16);
        for (X500Principal distinguishedName : this.certificateAuthorities) {
            byte[] encoded = distinguishedName.getEncoded();
            writer.writeVarBytes(encoded, 16);
        }
        return writer.toByteArray();
    }

    public static HandshakeMessage fromReader(DatagramReader reader) {
        ArrayList<ClientCertificateType> certificateTypes = new ArrayList<ClientCertificateType>();
        int length = reader.read(8);
        DatagramReader rangeReader = reader.createRangeReader(length);
        while (rangeReader.bytesAvailable()) {
            int code = rangeReader.read(8);
            certificateTypes.add(ClientCertificateType.getTypeByCode(code));
        }
        ArrayList<SignatureAndHashAlgorithm> supportedSignatureAlgorithms = new ArrayList<SignatureAndHashAlgorithm>();
        length = reader.read(16);
        rangeReader = reader.createRangeReader(length);
        while (rangeReader.bytesAvailable()) {
            int codeHash = rangeReader.read(8);
            int codeSignature = rangeReader.read(8);
            supportedSignatureAlgorithms.add(new SignatureAndHashAlgorithm(codeHash, codeSignature));
        }
        ArrayList<X500Principal> certificateAuthorities = new ArrayList<X500Principal>();
        length = reader.read(16);
        rangeReader = reader.createRangeReader(length);
        while (rangeReader.bytesAvailable()) {
            byte[] name = rangeReader.readVarBytes(16);
            certificateAuthorities.add(new X500Principal(name));
        }
        return new CertificateRequest(certificateTypes, supportedSignatureAlgorithms, certificateAuthorities);
    }

    public void addCertificateType(ClientCertificateType certificateType) {
        this.certificateTypes.add(certificateType);
    }

    public void addSignatureAlgorithm(SignatureAndHashAlgorithm signatureAndHashAlgorithm) {
        this.supportedSignatureAlgorithms.add(signatureAndHashAlgorithm);
    }

    public void addSignatureAlgorithms(List<SignatureAndHashAlgorithm> signatureAndHashAlgorithms) {
        this.supportedSignatureAlgorithms.addAll(signatureAndHashAlgorithms);
    }

    public void selectSignatureAlgorithms(List<SignatureAndHashAlgorithm> supportedSignatureAndHashAlgorithms) {
        ArrayList<SignatureAndHashAlgorithm> removes = new ArrayList<SignatureAndHashAlgorithm>();
        for (SignatureAndHashAlgorithm algo : this.supportedSignatureAlgorithms) {
            if (supportedSignatureAndHashAlgorithms.contains(algo)) continue;
            removes.add(algo);
        }
        this.supportedSignatureAlgorithms.removeAll(removes);
    }

    public boolean addCertificateAuthority(X500Principal authority) {
        if (authority == null) {
            throw new NullPointerException("authority must not be null");
        }
        int encodedAuthorityLength = 2 + authority.getEncoded().length;
        if (this.certificateAuthoritiesEncodedLength + encodedAuthorityLength <= 65535) {
            this.certificateAuthorities.add(authority);
            this.certificateAuthoritiesEncodedLength += encodedAuthorityLength;
            return true;
        }
        return false;
    }

    public boolean addCerticiateAuthorities(List<X500Principal> authorities) {
        int authoritiesAdded = 0;
        for (X500Principal authority : authorities) {
            if (!this.addCertificateAuthority(authority)) {
                LOGGER.debug("could add only {} of {} certificate authorities, max length exceeded", (Object)authoritiesAdded, (Object)authorities.size());
                return false;
            }
            ++authoritiesAdded;
        }
        return true;
    }

    public List<ClientCertificateType> getCertificateTypes() {
        return Collections.unmodifiableList(this.certificateTypes);
    }

    boolean isSupportedKeyType(PublicKey key) {
        String algorithm = key.getAlgorithm();
        for (ClientCertificateType type : this.certificateTypes) {
            if (!type.isCompatibleWithKeyAlgorithm(algorithm)) continue;
            return true;
        }
        return false;
    }

    boolean isSupportedKeyType(X509Certificate cert) {
        Boolean clientUsage = null;
        String algorithm = cert.getPublicKey().getAlgorithm();
        for (ClientCertificateType type : this.certificateTypes) {
            if (!type.isCompatibleWithKeyAlgorithm(algorithm)) {
                LOGGER.debug("type: {}, is not compatible with KeyAlgorithm[{}]", (Object)type, (Object)algorithm);
                continue;
            }
            if (type.requiresSigningCapability()) {
                if (clientUsage == null) {
                    clientUsage = CertPathUtil.canBeUsedForAuthentication(cert, true);
                }
                if (!clientUsage.booleanValue()) {
                    LOGGER.error("type: {}, requires missing signing capability!", (Object)type);
                    continue;
                }
            }
            LOGGER.debug("type: {}, is compatible with KeyAlgorithm[{}] and meets signing requirements", (Object)type, (Object)algorithm);
            return true;
        }
        LOGGER.debug("certificate [{}] with public key {} is not of any supported type", (Object)cert, (Object)algorithm);
        return false;
    }

    public SignatureAndHashAlgorithm getSignatureAndHashAlgorithm(PublicKey key) {
        if (this.isSupportedKeyType(key)) {
            return SignatureAndHashAlgorithm.getSupportedSignatureAlgorithm(this.supportedSignatureAlgorithms, key);
        }
        return null;
    }

    public SignatureAndHashAlgorithm getSignatureAndHashAlgorithm(List<X509Certificate> chain) {
        SignatureAndHashAlgorithm signatureAndHashAlgorithm;
        X509Certificate certificate = chain.get(0);
        if (this.isSupportedKeyType(certificate) && (signatureAndHashAlgorithm = SignatureAndHashAlgorithm.getSupportedSignatureAlgorithm(this.supportedSignatureAlgorithms, certificate.getPublicKey())) != null && SignatureAndHashAlgorithm.isSignedWithSupportedAlgorithms(this.supportedSignatureAlgorithms, chain)) {
            return signatureAndHashAlgorithm;
        }
        return null;
    }

    public List<SignatureAndHashAlgorithm> getSupportedSignatureAlgorithms() {
        return Collections.unmodifiableList(this.supportedSignatureAlgorithms);
    }

    public List<X500Principal> getCertificateAuthorities() {
        return Collections.unmodifiableList(this.certificateAuthorities);
    }

    public static enum ClientCertificateType {
        RSA_SIGN(1, true, "RSA"),
        DSS_SIGN(2, true, "DSA"),
        RSA_FIXED_DH(3, false, "DH"),
        DSS_FIXED_DH(4, false, "DH"),
        RSA_EPHEMERAL_DH_RESERVED(5, false, "DH"),
        DSS_EPHEMERAL_DH_RESERVED(6, false, "DH"),
        FORTEZZA_DMS_RESERVED(20, false, "UNKNOWN"),
        ECDSA_SIGN(64, true, "EC", "EdDSA", "OID.1.3.101.112", "OID.1.3.101.113"),
        RSA_FIXED_ECDH(65, false, "DH"),
        ECDSA_FIXED_ECDH(66, false, "DH");

        private final int code;
        private final boolean requiresSigningCapability;
        private final String[] jcaAlgorithms;

        private ClientCertificateType(int code, boolean requiresSigningCapability, String ... algorithms) {
            this.code = code;
            this.jcaAlgorithms = algorithms;
            this.requiresSigningCapability = requiresSigningCapability;
        }

        public int getCode() {
            return this.code;
        }

        public boolean requiresSigningCapability() {
            return this.requiresSigningCapability;
        }

        public boolean isCompatibleWithKeyAlgorithm(String algorithm) {
            algorithm = Asn1DerDecoder.getEdDsaStandardAlgorithmName(algorithm, algorithm);
            for (String jcaAlgorithm : this.jcaAlgorithms) {
                if (!jcaAlgorithm.equalsIgnoreCase(algorithm)) continue;
                return true;
            }
            return false;
        }

        public static ClientCertificateType getTypeByCode(int code) {
            for (ClientCertificateType type : ClientCertificateType.values()) {
                if (type.code != code) continue;
                return type;
            }
            return null;
        }
    }
}

