/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements.util;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InterfaceAddress;
import java.net.NetworkInterface;
import java.net.SocketAddress;
import java.net.SocketException;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;
import java.util.regex.Pattern;
import org.eclipse.californium.elements.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetworkInterfacesUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(NetworkInterfacesUtil.class);
    public static final int MAX_MTU = 65535;
    public static final int DEFAULT_IPV6_MTU = 1280;
    public static final int DEFAULT_IPV4_MTU = 576;
    public static final String COAP_NETWORK_INTERFACES = "COAP_NETWORK_INTERFACES";
    public static final String COAP_NETWORK_INTERFACES_EXCLUDE = "COAP_NETWORK_INTERFACES_EXCLUDE";
    public static final String DEFAULT_COAP_NETWORK_INTERFACES_EXCLUDE = "(vxlan\\.calico|cali[0123456789abcdef]{10,}|cilium_\\w+|lxc[0123456789abcdef]{12,}|virbr\\d+|docker\\d+)";
    private static final Pattern DEFAULT_EXCLUDE = Pattern.compile("(vxlan\\.calico|cali[0123456789abcdef]{10,}|cilium_\\w+|lxc[0123456789abcdef]{12,}|virbr\\d+|docker\\d+)");
    private static int anyMtu;
    private static int ipv4Mtu;
    private static int ipv6Mtu;
    private static boolean anyIpv4;
    private static boolean anyIpv6;
    private static Inet4Address broadcastIpv4;
    private static Inet4Address multicastInterfaceIpv4;
    private static Inet6Address multicastInterfaceIpv6;
    private static NetworkInterface multicastInterface;
    private static final Set<InetAddress> broadcastAddresses;
    private static final Set<String> ipv6Scopes;

    private static synchronized void initialize() {
        if (anyMtu == 0) {
            NetworkInterfacesUtil.clear();
            int mtu = 65535;
            int ipv4mtu = 65535;
            int ipv6mtu = 65535;
            try {
                Filter interfaces = NetworkInterface.getNetworkInterfaces();
                if (interfaces == null) {
                    throw new SocketException("Network interfaces not available!");
                }
                interfaces = new Filter(interfaces);
                while (interfaces.hasMoreElements()) {
                    InetAddress address;
                    Enumeration<InetAddress> inetAddresses;
                    NetworkInterface iface = (NetworkInterface)interfaces.nextElement();
                    if (iface.isLoopback()) continue;
                    int ifaceMtu = iface.getMTU();
                    if (ifaceMtu > 0 && ifaceMtu < mtu) {
                        mtu = ifaceMtu;
                    }
                    if (iface.supportsMulticast()) {
                        inetAddresses = iface.getInetAddresses();
                        while (inetAddresses.hasMoreElements()) {
                            address = inetAddresses.nextElement();
                            if (!(address instanceof Inet6Address) || ((Inet6Address)address).getScopeId() <= 0) continue;
                            ipv6Scopes.add(iface.getName());
                        }
                    }
                    if (iface.supportsMulticast() && (multicastInterfaceIpv4 == null || multicastInterfaceIpv6 == null || broadcastIpv4 == null)) {
                        Inet4Address broad4 = null;
                        Inet4Address link4 = null;
                        Inet4Address site4 = null;
                        Inet6Address link6 = null;
                        Inet6Address site6 = null;
                        int countMultiFeatures = 0;
                        if (broadcastIpv4 != null) {
                            --countMultiFeatures;
                        }
                        if (multicastInterfaceIpv4 != null) {
                            --countMultiFeatures;
                        }
                        if (multicastInterfaceIpv6 != null) {
                            --countMultiFeatures;
                        }
                        Enumeration<InetAddress> inetAddresses2 = iface.getInetAddresses();
                        while (inetAddresses2.hasMoreElements()) {
                            InetAddress address2 = inetAddresses2.nextElement();
                            if (address2 instanceof Inet4Address) {
                                anyIpv4 = true;
                                if (ifaceMtu > 0 && ifaceMtu < ipv4mtu) {
                                    ipv4mtu = ifaceMtu;
                                }
                                if (site4 != null) continue;
                                if (address2.isSiteLocalAddress()) {
                                    site4 = (Inet4Address)address2;
                                    continue;
                                }
                                if (link4 != null || !address2.isLinkLocalAddress()) continue;
                                link4 = (Inet4Address)address2;
                                continue;
                            }
                            if (!(address2 instanceof Inet6Address)) continue;
                            Inet6Address address6 = (Inet6Address)address2;
                            anyIpv6 = true;
                            if (ifaceMtu > 0 && ifaceMtu < ipv6mtu) {
                                ipv6mtu = ifaceMtu;
                            }
                            if (site6 != null) continue;
                            if (address2.isSiteLocalAddress()) {
                                site6 = address6;
                                continue;
                            }
                            if (link4 != null || !address2.isLinkLocalAddress()) continue;
                            link6 = address6;
                        }
                        for (InterfaceAddress interfaceAddress : iface.getInterfaceAddresses()) {
                            InetAddress address3;
                            InetAddress broadcast = interfaceAddress.getBroadcast();
                            if (broadcast == null || broadcast.isAnyLocalAddress() || (address3 = interfaceAddress.getAddress()) == null || address3.equals(broadcast)) continue;
                            broadcastAddresses.add(broadcast);
                            LOGGER.debug("Found broadcast address {} - {}.", (Object)broadcast, (Object)iface.getName());
                            if (broad4 != null) continue;
                            broad4 = (Inet4Address)broadcast;
                            ++countMultiFeatures;
                        }
                        if (link4 != null || site4 != null) {
                            ++countMultiFeatures;
                        }
                        if (link6 != null || site6 != null) {
                            ++countMultiFeatures;
                        }
                        if (countMultiFeatures <= 0) continue;
                        multicastInterface = iface;
                        broadcastIpv4 = broad4;
                        multicastInterfaceIpv4 = site4 == null ? link4 : site4;
                        multicastInterfaceIpv6 = site6 == null ? link6 : site6;
                        continue;
                    }
                    inetAddresses = iface.getInetAddresses();
                    while (inetAddresses.hasMoreElements()) {
                        address = inetAddresses.nextElement();
                        if (address instanceof Inet4Address) {
                            anyIpv4 = true;
                            if (ifaceMtu <= 0 || ifaceMtu >= ipv4mtu) continue;
                            ipv4mtu = ifaceMtu;
                            continue;
                        }
                        if (!(address instanceof Inet6Address)) continue;
                        anyIpv6 = true;
                        if (ifaceMtu <= 0 || ifaceMtu >= ipv6mtu) continue;
                        ipv6mtu = ifaceMtu;
                    }
                }
            }
            catch (SocketException ex) {
                LOGGER.warn("discover the <any> interface failed!", ex);
                anyIpv4 = true;
                anyIpv6 = true;
            }
            if (broadcastAddresses.isEmpty()) {
                LOGGER.info("no broadcast address found!");
            }
            if (ipv4mtu == 65535) {
                ipv4mtu = 576;
            }
            if (ipv6mtu == 65535) {
                ipv6mtu = 1280;
            }
            if (mtu == 65535) {
                mtu = Math.min(ipv4mtu, ipv6mtu);
            }
            ipv4Mtu = ipv4mtu;
            ipv6Mtu = ipv6mtu;
            anyMtu = mtu;
        }
    }

    public static synchronized void clear() {
        anyMtu = 0;
        ipv4Mtu = 0;
        ipv6Mtu = 0;
        anyIpv4 = false;
        anyIpv6 = false;
        ipv6Scopes.clear();
        broadcastAddresses.clear();
        broadcastIpv4 = null;
        multicastInterfaceIpv4 = null;
        multicastInterfaceIpv6 = null;
        multicastInterface = null;
    }

    public static int getAnyMtu() {
        NetworkInterfacesUtil.initialize();
        return anyMtu;
    }

    public static int getIPv4Mtu() {
        NetworkInterfacesUtil.initialize();
        return ipv4Mtu;
    }

    public static int getIPv6Mtu() {
        NetworkInterfacesUtil.initialize();
        return ipv6Mtu;
    }

    public static boolean isAnyIpv4() {
        NetworkInterfacesUtil.initialize();
        return anyIpv4;
    }

    public static boolean isAnyIpv6() {
        NetworkInterfacesUtil.initialize();
        return anyIpv6;
    }

    public static Inet4Address getBroadcastIpv4() {
        NetworkInterfacesUtil.initialize();
        return broadcastIpv4;
    }

    public static Inet4Address getMulticastInterfaceIpv4() {
        NetworkInterfacesUtil.initialize();
        return multicastInterfaceIpv4;
    }

    public static Inet6Address getMulticastInterfaceIpv6() {
        NetworkInterfacesUtil.initialize();
        return multicastInterfaceIpv6;
    }

    public static NetworkInterface getMulticastInterface() {
        NetworkInterfacesUtil.initialize();
        return multicastInterface;
    }

    public static Collection<InetAddress> getNetworkInterfaces() {
        LinkedList<InetAddress> interfaces = new LinkedList<InetAddress>();
        try {
            Filter nets = NetworkInterface.getNetworkInterfaces();
            if (nets == null) {
                throw new SocketException("Network interfaces not available!");
            }
            nets = new Filter(nets);
            while (nets.hasMoreElements()) {
                NetworkInterface networkInterface = (NetworkInterface)nets.nextElement();
                Enumeration<InetAddress> inetAddresses = networkInterface.getInetAddresses();
                while (inetAddresses.hasMoreElements()) {
                    InetAddress address = inetAddresses.nextElement();
                    interfaces.add(address);
                }
            }
        }
        catch (SocketException e) {
            LOGGER.error("could not fetch all interface addresses", e);
        }
        return interfaces;
    }

    public static Set<String> getIpv6Scopes() {
        NetworkInterfacesUtil.initialize();
        return Collections.unmodifiableSet(ipv6Scopes);
    }

    public static boolean isBroadcastAddress(InetAddress address) {
        NetworkInterfacesUtil.initialize();
        return broadcastAddresses.contains(address);
    }

    public static boolean isMultiAddress(InetAddress address) {
        NetworkInterfacesUtil.initialize();
        return address != null && (address.isMulticastAddress() || broadcastAddresses.contains(address));
    }

    public static boolean equals(InetAddress address1, InetAddress address2) {
        return address1 == address2 || address1 != null && address1.equals(address2);
    }

    public static boolean equals(SocketAddress address1, SocketAddress address2) {
        return address1 == address2 || address1 != null && address1.equals(address2);
    }

    static {
        broadcastAddresses = new HashSet<InetAddress>();
        ipv6Scopes = new HashSet<String>();
    }

    private static class Filter
    implements Enumeration<NetworkInterface> {
        private NetworkInterface nextInterface;
        private final Enumeration<NetworkInterface> source;
        private final Pattern filter;
        private final Pattern excludeFilter;

        private Filter(Enumeration<NetworkInterface> source) {
            this.source = source;
            Pattern filter = null;
            Pattern excludeFilter = null;
            String regex = StringUtil.getConfiguration(NetworkInterfacesUtil.COAP_NETWORK_INTERFACES);
            String excludeRegex = StringUtil.getConfiguration(NetworkInterfacesUtil.COAP_NETWORK_INTERFACES_EXCLUDE);
            if (regex != null && !regex.isEmpty()) {
                filter = Pattern.compile(regex);
            } else if (excludeRegex == null || excludeRegex.isEmpty()) {
                excludeFilter = DEFAULT_EXCLUDE;
            }
            if (excludeRegex != null && !excludeRegex.isEmpty()) {
                excludeFilter = Pattern.compile(excludeRegex);
            }
            this.filter = filter;
            this.excludeFilter = excludeFilter;
            this.next();
        }

        @Override
        public boolean hasMoreElements() {
            return this.nextInterface != null;
        }

        @Override
        public NetworkInterface nextElement() {
            NetworkInterface result = this.nextInterface;
            this.next();
            return result;
        }

        private void next() {
            this.nextInterface = null;
            while (this.source.hasMoreElements()) {
                NetworkInterface iface = this.source.nextElement();
                String name = iface.getName();
                try {
                    if (!(!iface.isUp() || this.filter != null && !this.filter.matcher(name).matches() || this.excludeFilter != null && this.excludeFilter.matcher(name).matches())) {
                        this.nextInterface = iface;
                        break;
                    }
                }
                catch (SocketException socketException) {
                    // empty catch block
                }
                LOGGER.debug("skip {}", (Object)name);
            }
        }
    }
}

