/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.cli;

import java.io.IOException;
import java.io.PrintStream;
import java.net.InetSocketAddress;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import javax.crypto.SecretKey;
import org.eclipse.californium.cli.CliConnectorFactory;
import org.eclipse.californium.cli.ClientBaseConfig;
import org.eclipse.californium.cli.ConnectorConfig;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.network.CoapEndpoint;
import org.eclipse.californium.core.network.EndpointManager;
import org.eclipse.californium.core.network.interceptors.MessageTracer;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.UDPConnector;
import org.eclipse.californium.elements.config.Configuration;
import org.eclipse.californium.elements.util.SslContextUtil;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.DTLSConnector;
import org.eclipse.californium.scandium.config.DtlsConfig;
import org.eclipse.californium.scandium.config.DtlsConnectorConfig;
import org.eclipse.californium.scandium.dtls.CertificateType;
import org.eclipse.californium.scandium.dtls.ConnectionId;
import org.eclipse.californium.scandium.dtls.HandshakeResultHandler;
import org.eclipse.californium.scandium.dtls.PskPublicInformation;
import org.eclipse.californium.scandium.dtls.PskSecretResult;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.dtls.pskstore.AdvancedPskStore;
import org.eclipse.californium.scandium.dtls.x509.SingleCertificateProvider;
import org.eclipse.californium.scandium.dtls.x509.StaticNewAdvancedCertificateVerifier;
import org.eclipse.californium.scandium.util.ListUtils;
import org.eclipse.californium.scandium.util.SecretUtil;
import org.eclipse.californium.scandium.util.ServerNames;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import picocli.CommandLine;

public class ClientInitializer {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClientInitializer.class);
    private static final String DEFAULT_TCP_MODULE = "org.eclipse.californium.cli.tcp.netty.Initialize";
    private static final List<String> loadErrors = new ArrayList<String>();
    private static final Map<String, CliConnectorFactory> connectorFactories = new ConcurrentHashMap<String, CliConnectorFactory>();
    private static final Set<String> registeredProtocols = new TreeSet<String>();

    public static CliConnectorFactory registerConnectorFactory(String protocol, CliConnectorFactory factory) {
        registeredProtocols.add(protocol);
        return connectorFactories.put(protocol, factory);
    }

    public static CliConnectorFactory unregisterConnectorFactory(String protocol) {
        registeredProtocols.remove(protocol);
        return connectorFactories.remove(protocol);
    }

    public static void init(String[] args, ClientBaseConfig config) throws IOException {
        ClientInitializer.init(args, config, true);
    }

    public static void init(String[] args, ClientBaseConfig config, boolean createEndpoint) throws IOException {
        CommandLine cmd = new CommandLine(config);
        config.register(cmd);
        try {
            CommandLine.ParseResult result = cmd.parseArgs(args);
            if (result.isVersionHelpRequested()) {
                String version = StringUtil.CALIFORNIUM_VERSION == null ? "" : StringUtil.CALIFORNIUM_VERSION;
                System.out.println("\nCalifornium (Cf) " + cmd.getCommandName() + " " + version);
                cmd.printVersionHelp(System.out);
                System.out.println();
            }
            config.defaults();
            if (config.helpRequested) {
                cmd.usage(System.out);
                if (config.authHelpRequested) {
                    System.out.println();
                    System.out.println("   --auth: values");
                    ClientInitializer.print("      ", 60, Arrays.asList(ConnectorConfig.AuthenticationMode.values()), System.out);
                }
                if (config.cipherHelpRequested) {
                    ArrayList<CipherSuite> list = new ArrayList<CipherSuite>();
                    for (CipherSuite cipherSuite : CipherSuite.values()) {
                        if (!cipherSuite.isSupported() || CipherSuite.TLS_NULL_WITH_NULL_NULL.equals((Object)cipherSuite)) continue;
                        list.add(cipherSuite);
                    }
                    System.out.println();
                    System.out.println("   --cipher: values");
                    ClientInitializer.print("      ", 60, list, System.out);
                }
                return;
            }
        }
        catch (CommandLine.ParameterException ex) {
            ex.printStackTrace();
            System.err.println(ex.getMessage());
            System.err.println();
            cmd.usage(System.err);
            System.err.println();
            StringBuilder line = new StringBuilder();
            for (String arg : args) {
                line.append(arg).append(" ");
            }
            System.err.println(line);
            System.exit(-1);
        }
        if (createEndpoint) {
            ClientInitializer.registerEndpoint(config, null);
        }
    }

    public static void registerEndpoint(ClientBaseConfig config, ExecutorService executor) throws IOException {
        CoapEndpoint coapEndpoint = ClientInitializer.createEndpoint(config, null);
        coapEndpoint.start();
        LOGGER.info("endpoint started at {}", (Object)coapEndpoint.getAddress());
        EndpointManager.getEndpointManager().setDefaultEndpoint(coapEndpoint);
    }

    public static CoapEndpoint createEndpoint(ClientBaseConfig config, ExecutorService executor) {
        String scheme = CoAP.getSchemeFromUri(config.uri);
        if (scheme != null) {
            String protocol = CoAP.getProtocolForScheme(scheme);
            if (protocol != null) {
                CliConnectorFactory factory = connectorFactories.get(protocol);
                if (factory != null) {
                    CoapEndpoint.Builder builder = new CoapEndpoint.Builder();
                    builder.setLoggingTag(config.tag);
                    Connector connector = factory.create(config, executor);
                    builder.setConnector(connector);
                    builder.setConfiguration(config.configuration);
                    CoapEndpoint endpoint = builder.build();
                    if (config.verbose) {
                        endpoint.addInterceptor(new MessageTracer());
                    }
                    return endpoint;
                }
                if (CoAP.isTcpProtocol(protocol) && loadErrors.contains(DEFAULT_TCP_MODULE)) {
                    throw new IllegalArgumentException("Protocol '" + protocol + " is not supported! TCP-module not found!");
                }
                throw new IllegalArgumentException("Protocol '" + protocol + "' is not supported!");
            }
            throw new IllegalArgumentException("Scheme '" + scheme + "' is unknown!");
        }
        throw new IllegalArgumentException("Missing scheme in " + config.uri);
    }

    public static void print(String tab, int width, List<?> values, PrintStream out) {
        StringBuilder line = new StringBuilder();
        line.append(tab);
        for (Object value : values) {
            String name = value.toString();
            if (line.length() + name.length() > width) {
                out.println(line);
                line.setLength(tab.length());
            }
            line.append(name).append(" ");
        }
        out.println(line);
    }

    static {
        connectorFactories.put("UDP", new UdpConnectorFactory());
        connectorFactories.put("DTLS", new DtlsConnectorFactory());
        String factories = StringUtil.getConfiguration("CONNECTOR_FACTORIES");
        if (factories == null) {
            factories = DEFAULT_TCP_MODULE;
        }
        if (!factories.isEmpty()) {
            String[] initializers;
            for (String initializer : initializers = factories.split("#")) {
                registeredProtocols.clear();
                try {
                    Class.forName(initializer);
                }
                catch (ClassNotFoundException e) {
                    loadErrors.add(initializer);
                }
                if (registeredProtocols.isEmpty()) continue;
                LOGGER.info("loaded {} - {}", (Object)initializer, (Object)registeredProtocols);
                registeredProtocols.clear();
            }
        }
    }

    public static class PlugPskStore
    implements AdvancedPskStore {
        private final PskPublicInformation identity;
        private final SecretKey secret;

        public PlugPskStore(String id, byte[] secret) {
            this.identity = new PskPublicInformation(id);
            this.secret = secret == null ? ConnectorConfig.PSK_SECRET : SecretUtil.create(secret, "PSK");
            LOGGER.trace("DTLS-PSK-Identity: {}", (Object)this.identity);
        }

        public PlugPskStore(String id, SecretKey secret) {
            this.identity = new PskPublicInformation(id);
            this.secret = secret == null ? ConnectorConfig.PSK_SECRET : SecretUtil.create(secret);
            LOGGER.trace("DTLS-PSK-Identity: {}", (Object)this.identity);
        }

        public PlugPskStore(String id) {
            this.identity = new PskPublicInformation("cali." + id);
            this.secret = null;
            LOGGER.trace("DTLS-PSK-Identity: {} ({} random bytes)", (Object)this.identity, (Object)(id.length() / 2));
        }

        @Override
        public boolean hasEcdhePskSupported() {
            return true;
        }

        @Override
        public PskSecretResult requestPskSecretResult(ConnectionId cid, ServerNames serverName, PskPublicInformation identity, String hmacAlgorithm, SecretKey otherSecret, byte[] seed, boolean useExtendedMasterSecret) {
            SecretKey secret = null;
            if (this.identity.equals(identity)) {
                secret = this.secret == null && identity.getPublicInfoAsString().startsWith("cali.") ? SecretUtil.create(ConnectorConfig.PSK_SECRET) : SecretUtil.create(this.secret);
            }
            return new PskSecretResult(cid, this.identity, secret);
        }

        @Override
        public PskPublicInformation getIdentity(InetSocketAddress peerAddress, ServerNames virtualHost) {
            return this.identity;
        }

        @Override
        public void setResultHandler(HandshakeResultHandler resultHandler) {
        }
    }

    public static class DtlsConnectorFactory
    implements CliConnectorFactory {
        public static DtlsConnectorConfig.Builder createDtlsConfig(ClientBaseConfig clientConfig) {
            Configuration config = clientConfig.configuration;
            int localPort = clientConfig.localPort == null ? 0 : clientConfig.localPort;
            int extra = 97;
            Integer cidLength = clientConfig.cidLength;
            if (cidLength == null) {
                cidLength = config.get(DtlsConfig.DTLS_CONNECTION_ID_LENGTH);
            }
            if (cidLength != null) {
                extra += cidLength.intValue();
            }
            if (clientConfig.mtu != null && clientConfig.recordSizeLimit == null) {
                clientConfig.recordSizeLimit = clientConfig.mtu - extra;
            } else if (clientConfig.mtu == null && clientConfig.recordSizeLimit != null) {
                clientConfig.mtu = clientConfig.recordSizeLimit + extra;
            }
            if (clientConfig.mtu != null) {
                config.set(DtlsConfig.DTLS_MAX_TRANSMISSION_UNIT, clientConfig.mtu);
            }
            if (clientConfig.recordSizeLimit != null) {
                config.set(DtlsConfig.DTLS_RECORD_SIZE_LIMIT, clientConfig.recordSizeLimit);
            }
            if (clientConfig.cidLength != null) {
                config.set(DtlsConfig.DTLS_CONNECTION_ID_LENGTH, clientConfig.cidLength);
            }
            if (clientConfig.dtlsAutoHandshake != null) {
                config.setFromText(DtlsConfig.DTLS_AUTO_HANDSHAKE_TIMEOUT, clientConfig.dtlsAutoHandshake);
                LOGGER.info("set [{}] to {}", (Object)DtlsConfig.DTLS_AUTO_HANDSHAKE_TIMEOUT.getKey(), (Object)config.getAsText(DtlsConfig.DTLS_AUTO_HANDSHAKE_TIMEOUT));
            }
            if (clientConfig.noCertificatesSubjectVerification != null) {
                config.set(DtlsConfig.DTLS_VERIFY_SERVER_CERTIFICATES_SUBJECT, clientConfig.noCertificatesSubjectVerification == false);
            }
            if (clientConfig.noServerNameIndication != null) {
                config.set(DtlsConfig.DTLS_USE_SERVER_NAME_INDICATION, clientConfig.noServerNameIndication == false);
            }
            if (clientConfig.extendedMasterSecretMode != null) {
                config.set(DtlsConfig.DTLS_EXTENDED_MASTER_SECRET_MODE, clientConfig.extendedMasterSecretMode);
            }
            DtlsConnectorConfig.Builder dtlsConfig = DtlsConnectorConfig.builder(config);
            StaticNewAdvancedCertificateVerifier.Builder verifierBuilder = StaticNewAdvancedCertificateVerifier.builder();
            boolean psk = false;
            boolean cert = false;
            ArrayList<CipherSuite.KeyExchangeAlgorithm> keyExchangeAlgorithms = new ArrayList<CipherSuite.KeyExchangeAlgorithm>();
            ArrayList<CertificateType> certificateTypes = new ArrayList<CertificateType>();
            for (ConnectorConfig.AuthenticationMode auth : clientConfig.authenticationModes) {
                switch (auth) {
                    case NONE: {
                        break;
                    }
                    case PSK: {
                        psk = true;
                        keyExchangeAlgorithms.add(CipherSuite.KeyExchangeAlgorithm.PSK);
                        break;
                    }
                    case RPK: {
                        cert = true;
                        certificateTypes.add(CertificateType.RAW_PUBLIC_KEY);
                        ListUtils.addIfAbsent(keyExchangeAlgorithms, CipherSuite.KeyExchangeAlgorithm.EC_DIFFIE_HELLMAN);
                        verifierBuilder.setTrustAllRPKs();
                        break;
                    }
                    case X509: {
                        cert = true;
                        certificateTypes.add(CertificateType.X_509);
                        verifierBuilder.setTrustedCertificates(clientConfig.trust.trusts);
                        ListUtils.addIfAbsent(keyExchangeAlgorithms, CipherSuite.KeyExchangeAlgorithm.EC_DIFFIE_HELLMAN);
                        break;
                    }
                    case ECDHE_PSK: {
                        psk = true;
                        keyExchangeAlgorithms.add(CipherSuite.KeyExchangeAlgorithm.ECDHE_PSK);
                    }
                }
            }
            if (cert) {
                verifierBuilder.setSupportedCertificateTypes(certificateTypes);
                dtlsConfig.setAdvancedCertificateVerifier(verifierBuilder.build());
            }
            if (clientConfig.authentication != null && clientConfig.authentication.credentials != null) {
                SslContextUtil.Credentials identity = clientConfig.authentication.credentials;
                if (certificateTypes.contains((Object)CertificateType.X_509)) {
                    dtlsConfig.setCertificateIdentityProvider(new SingleCertificateProvider(identity.getPrivateKey(), (Certificate[])identity.getCertificateChain(), certificateTypes));
                } else if (certificateTypes.contains((Object)CertificateType.RAW_PUBLIC_KEY)) {
                    dtlsConfig.setCertificateIdentityProvider(new SingleCertificateProvider(identity.getPrivateKey(), identity.getPublicKey()));
                }
            }
            if (psk) {
                if (clientConfig.identity != null) {
                    dtlsConfig.setAdvancedPskStore(new PlugPskStore(clientConfig.identity, clientConfig.getPskSecretKey()));
                } else {
                    byte[] rid = new byte[8];
                    SecureRandom random = new SecureRandom();
                    random.nextBytes(rid);
                    dtlsConfig.setAdvancedPskStore(new PlugPskStore(StringUtil.byteArray2Hex(rid)));
                }
            }
            if (clientConfig.cipherSuites != null && !clientConfig.cipherSuites.isEmpty()) {
                dtlsConfig.set(DtlsConfig.DTLS_CIPHER_SUITES, clientConfig.cipherSuites);
                if (clientConfig.verbose) {
                    System.out.println("cipher suites:");
                    ClientInitializer.print("   ", 50, clientConfig.cipherSuites, System.out);
                }
            } else if (!keyExchangeAlgorithms.isEmpty()) {
                boolean recommendedOnly = config.get(DtlsConfig.DTLS_RECOMMENDED_CIPHER_SUITES_ONLY);
                List preselect = (List)((Object)config.get(DtlsConfig.DTLS_PRESELECTED_CIPHER_SUITES));
                List<CipherSuite> keyExchange = CipherSuite.getCipherSuitesByKeyExchangeAlgorithm(recommendedOnly, true, keyExchangeAlgorithms);
                if (preselect != null && !preselect.isEmpty()) {
                    keyExchange = CipherSuite.preselectCipherSuites(preselect, keyExchange);
                }
                dtlsConfig.set(DtlsConfig.DTLS_PRESELECTED_CIPHER_SUITES, keyExchange);
            }
            dtlsConfig.setAddress(new InetSocketAddress(localPort));
            return dtlsConfig;
        }

        @Override
        public Connector create(ClientBaseConfig clientConfig, ExecutorService executor) {
            DtlsConnectorConfig.Builder dtlsConfig = DtlsConnectorFactory.createDtlsConfig(clientConfig);
            DTLSConnector dtlsConnector = new DTLSConnector(dtlsConfig.build());
            if (executor != null) {
                dtlsConnector.setExecutor(executor);
            }
            return dtlsConnector;
        }
    }

    public static class UdpConnectorFactory
    implements CliConnectorFactory {
        @Override
        public Connector create(ClientBaseConfig clientConfig, ExecutorService executor) {
            int localPort = clientConfig.localPort == null ? 0 : clientConfig.localPort;
            return new UDPConnector(new InetSocketAddress(localPort), clientConfig.configuration);
        }
    }
}

