/**********************************************************************
Copyright (c) 2013 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.store.rdbms.mapping.java;

import java.sql.PreparedStatement;
import java.sql.ResultSet;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ExecutionContext;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.rdbms.RDBMSStoreManager;
import org.datanucleus.store.rdbms.table.Table;
import org.datanucleus.store.types.TypeManager;
import org.datanucleus.store.types.converters.TypeConverter;

/**
 * Mapping where the member has its value converted to/from some storable datastore type using a TypeConverter that 
 * uses a Long in the datastore.
 */
public class TypeConverterLongMapping extends SingleFieldMapping
{
    TypeConverter converter;

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.mapping.java.JavaTypeMapping#initialize(org.datanucleus.store.rdbms.RDBMSStoreManager, java.lang.String)
     */
    @Override
    public void initialize(RDBMSStoreManager storeMgr, String type)
    {
        ClassLoaderResolver clr = storeMgr.getNucleusContext().getClassLoaderResolver(null);
        Class fieldType = clr.classForName(type);
        converter = storeMgr.getNucleusContext().getTypeManager().getDefaultTypeConverterForType(fieldType);
        Class datastoreType = TypeManager.getDatastoreTypeForTypeConverter(converter, fieldType);
        if (!Long.class.isAssignableFrom(datastoreType))
        {
            throw new NucleusException("Attempt to create TypeConverterLongMapping for type " + type + " yet this is not using Long in the datastore");
        }

        super.initialize(storeMgr, type);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.mapping.SingleFieldMapping#initialize(org.datanucleus.metadata.AbstractMemberMetaData, org.datanucleus.store.rdbms.DatastoreContainerObject, org.datanucleus.ClassLoaderResolver)
     */
    @Override
    public void initialize(AbstractMemberMetaData fmd, Table table, ClassLoaderResolver clr)
    {
        if (fmd.getTypeConverterName() != null)
        {
            // Use specified converter (if found)
            converter = table.getStoreManager().getNucleusContext().getTypeManager().getTypeConverterForName(fmd.getTypeConverterName());
        }
        else
        {
            // Use default converter for the field type (if defined)
            converter = table.getStoreManager().getNucleusContext().getTypeManager().getDefaultTypeConverterForType(fmd.getType());
            if (converter == null)
            {
                throw new NucleusException("Attempt to create TypeConverterLongMapping when no type converter defined for member " + fmd.getFullFieldName());
            }
        }

        Class datastoreType = TypeManager.getDatastoreTypeForTypeConverter(converter, fmd.getType());
        if (!Long.class.isAssignableFrom(datastoreType))
        {
            throw new NucleusException("Attempt to create TypeConverterLongMapping for member " + fmd.getFullFieldName() + " yet this is not using LOng in the datastore");
        }

        super.initialize(fmd, table, clr);
    }

    /**
     * Accessor for the name of the java-type actually used when mapping the particular datastore
     * field. This java-type must have an entry in the datastore mappings.
     * @param index requested datastore field index.
     * @return the name of java-type for the requested datastore field.
     */
    public String getJavaTypeForDatastoreMapping(int index)
    {
        return Long.class.getName();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.mapping.JavaTypeMapping#getJavaType()
     */
    @Override
    public Class getJavaType()
    {
        return TypeManager.getMemberTypeForTypeConverter(converter, Long.class);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.mapping.SingleFieldMapping#setLong(org.datanucleus.ExecutionContext, java.lang.Object, int[], long)
     */
    @Override
    public void setLong(ExecutionContext ec, PreparedStatement ps, int[] exprIndex, long value)
    {
        if (exprIndex == null)
        {
            return;
        }

        getDatastoreMapping(0).setLong(ps, exprIndex[0], (Long)converter.toDatastoreType(value));
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.mapping.SingleFieldMapping#getLong(org.datanucleus.ExecutionContext, java.lang.Object, int[])
     */
    @Override
    public long getLong(ExecutionContext ec, ResultSet resultSet, int[] exprIndex)
    {
        if (exprIndex == null)
        {
            return 0;
        }

        return getDatastoreMapping(0).getLong(resultSet, exprIndex[0]);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.mapping.SingleFieldMapping#setObject(org.datanucleus.ExecutionContext, java.lang.Object, int[], java.lang.Object)
     */
    @Override
    public void setObject(ExecutionContext ec, PreparedStatement ps, int[] exprIndex, Object value)
    {
        if (exprIndex == null)
        {
            return;
        }

        getDatastoreMapping(0).setObject(ps, exprIndex[0], converter.toDatastoreType(value));
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.mapping.SingleFieldMapping#getObject(org.datanucleus.ExecutionContext, java.lang.Object, int[])
     */
    @Override
    public Object getObject(ExecutionContext ec, ResultSet resultSet, int[] exprIndex)
    {
        if (exprIndex == null)
        {
            return null;
        }

        Object datastoreValue = getDatastoreMapping(0).getObject(resultSet, exprIndex[0]);
        return (datastoreValue != null ? converter.toMemberType(datastoreValue) : null);
    }
}