/**********************************************************************
Copyright (c) 2007 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2008 Andy Jefferson - simplify interface. Add BULK_UPDATE, BULK_DELETE
2008 Andy Jefferson - strip majority of methods out into RDBMSQueryUtils.
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.query.legacy;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;

import org.datanucleus.ObjectManager;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.connection.ManagedConnectionResourceListener;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.StatementText;
import org.datanucleus.store.query.Query;
import org.datanucleus.store.query.QueryResult;
import org.datanucleus.store.query.ResultObjectFactory;
import org.datanucleus.store.rdbms.RDBMSStoreManager;
import org.datanucleus.store.rdbms.SQLController;
import org.datanucleus.store.rdbms.query.AbstractRDBMSQueryResult;
import org.datanucleus.store.rdbms.query.RDBMSQueryUtils;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.Localiser;

/**
 * Evaluator for SQL-based queries.
 * Takes the query, converts it into executable SQL and executes it, returning it in an appropriate form.
 */
public class SQLEvaluator
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER=Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    /** Query being evaluated. */
    Query query;

    /** ObjectManager for this query. */
    ObjectManager om;

    /** Factory for obtaining results from the result set. */
    ResultObjectFactory rof;

    Collection candidateCollection;

    /**
     * Constructor.
     * @param query The query
     * @param rof Factory for results
     * @param candidateCollection Candidate collection (optional)
     */
    public SQLEvaluator(Query query, ResultObjectFactory rof, Collection candidateCollection)
    {
        this.om = query.getObjectManager();
        this.query = query;
        this.rof = rof;
        this.candidateCollection = candidateCollection;
    }
    
    /**
     * Method to evaulate the query.
     * @param queryStmt The query statement.
     * @return Result of the query
     */
    public Object evaluate(QueryExpression queryStmt)
    {
        QueryResult qr = null;

        boolean useUpdateLock = RDBMSQueryUtils.useUpdateLockForQuery(query);
        try
        {
            RDBMSStoreManager storeMgr = (RDBMSStoreManager)om.getStoreManager();
            ManagedConnection mconn = storeMgr.getConnection(om);
            SQLController sqlControl = storeMgr.getSQLController();
            try
            {
                StatementText stmtText = null;
                if (query.getType() == Query.SELECT)
                {
                    stmtText = queryStmt.toStatementText(useUpdateLock);
                }
                else if (query.getType() == Query.BULK_UPDATE)
                {
                    stmtText = queryStmt.toUpdateStatementText();
                    // TODO Enable the update
                    throw new NucleusException("DataNucleus doesnt currently support bulk update statements");
                }
                else if (query.getType() == Query.BULK_DELETE)
                {
                    stmtText = queryStmt.toDeleteStatementText();
                    // TODO Enable the delete
                    throw new NucleusException("DataNucleus doesnt currently support bulk delete statements");
                }

                PreparedStatement ps = RDBMSQueryUtils.getPreparedStatementForQuery(mconn, stmtText, query);
                try
                {
                    // Apply timeouts, result set constraints, max rows etc
                    RDBMSQueryUtils.prepareStatementForExecution(ps, query, true);

                    if (query.getType() == Query.SELECT)
                    {
                        ResultSet rs = sqlControl.executeStatementQuery(mconn, stmtText.toString(), ps);
                        try
                        {
                            // Check the type of result set needed
                            String resultSetType = RDBMSQueryUtils.getResultSetTypeForQuery(query);
                            if (resultSetType.equals("scroll-insensitive") ||
                                resultSetType.equals("scroll-sensitive"))
                            {
                                qr = new LegacyScrollableQueryResult(query, rof, rs, 
                                    query.getResultDistinct() ? null : candidateCollection);
                            }
                            else
                            {
                                qr = new LegacyForwardQueryResult(query, rof, rs, 
                                    query.getResultDistinct() ? null : candidateCollection);
                            }

                            final QueryResult qr1 = qr;
                            final ManagedConnection mconn1 = mconn;
                            ManagedConnectionResourceListener listener = new ManagedConnectionResourceListener()
                            {
                                public void managedConnectionPreClose(){}
                                public void managedConnectionPostClose(){}
                                public void managedConnectionFlushed()
                                {
                                    // Disconnect the query from this ManagedConnection (read in unread rows etc)
                                    qr1.disconnect();
                                }
                                public void resourcePostClose()
                                {
                                    mconn1.removeListener(this);
                                }
                            };
                            mconn.addListener(listener);
                            ((AbstractRDBMSQueryResult)qr).addConnectionListener(listener);
                        }
                        finally
                        {
                            if (qr == null)
                            {
                                rs.close();
                            }
                        }
                    }
                    else
                    {
                        // UPDATE/DELETE query
                        int[] rcs = sqlControl.executeStatementUpdate(mconn, stmtText.toString(), ps, true);
                        NucleusLogger.QUERY.info(">> Update statement returned " + rcs[0]);
                        // TODO Return the number of affected records
                    }
                }
                finally
                {
                    if (qr == null)
                    {
                        sqlControl.closeStatement(mconn, ps);
                    }
                }
            }
            finally
            {
                mconn.release();
            }
        }
        catch (SQLException e)
        {
            throw new NucleusDataStoreException(
                LOCALISER.msg("042007", queryStmt.toStatementText(useUpdateLock), e.getMessage()), e);
        }
        return qr;
    }
}