/**********************************************************************
Copyright (c) 2003 Mike Martin (TJDO) and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Erik Bengtson - removed unused import
2003 Andy Jefferson - commented and localised
2004 Andy Jefferson - converted to use Logger
2004 Andy Jefferson - added support for uniqueness of query
2004 Andy Jefferson - added Single-String query support
2005 Andy Jefferson - added timeout support
2005 Andy Jefferson - added implicit variable/parameter support
2008 Andy Jefferson - refactored compiler to be stand-alone. Major changes
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.query.legacy;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.datanucleus.ObjectManager;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.store.Extent;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.Queryable;
import org.datanucleus.store.query.AbstractJDOQLQuery;
import org.datanucleus.store.query.NoQueryResultsException;
import org.datanucleus.store.query.Query;
import org.datanucleus.store.query.QueryNotUniqueException;
import org.datanucleus.store.query.QueryResult;
import org.datanucleus.store.query.ResultObjectFactory;
import org.datanucleus.store.rdbms.RDBMSStoreManager;
import org.datanucleus.store.rdbms.adapter.RDBMSAdapter;
import org.datanucleus.store.rdbms.query.PersistentClassROF;
import org.datanucleus.util.Imports;
import org.datanucleus.util.NucleusLogger;

/**
 * RDBMS representation of a JDOQL query.
 * The query can be specified via method calls, or via a single-string form.
 * @see Query
 */
public class JDOQLQuery extends AbstractJDOQLQuery
{
    /** The Query Statement. */
    protected transient QueryExpression queryStmt = null;

    /** Candidates for this query. */
    protected transient Queryable candidates = null;

    /** Factory for obtaining the results from the query result set. */
    protected transient ResultObjectFactory rof = null;

    /** State variable for the compilation state */
    protected transient boolean isCompiled = false;

    /** Result metadata (extension, allowing access to more info about results). **/
    protected transient QueryResultsMetaData resultMetaData;

    /**
     * Constructs a new query instance that uses the given persistence manager.
     * @param om the associated ObjectManager for this query.
     */
    public JDOQLQuery(ObjectManager om)
    {
        this(om, (JDOQLQuery)null);
    }

    /**
     * Constructs a new query instance having the same criteria as the given query.
     * @param om The ObjectManager
     * @param q The query from which to copy criteria.
     */
    public JDOQLQuery(ObjectManager om, JDOQLQuery q)
    {
        super(om, q);
    }

    /**
     * Constructor for a JDOQL query where the query is specified using the "Single-String" format.
     * @param om The persistence manager
     * @param query The query string
     */
    public JDOQLQuery(ObjectManager om, String query)
    {
        super(om, query);
    }

    /**
     * Accessor for the candidates for the query.
     * This is only valid after compiling the query.
     * @return Candidates for the query
     */
    public Queryable getCandidates()
    {
        return candidates;
    }

    /**
     * Equality operator for JDOQL.
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
    
        if (!(obj instanceof JDOQLQuery) || !super.equals(obj))
        {
            return false;
        }
    
        return true;
    }

    /**
     * Method to discard our current compiled query due to changes.
     * @see org.datanucleus.store.query.Query#discardCompiled()
     */
    protected void discardCompiled()
    {
        isCompiled = false;
        rof = null;
        queryStmt = null;
        super.discardCompiled();
    }

    /**
     * Method to return if the query is compiled.
     * @return Whether it is compiled
     */
    protected boolean isCompiled()
    {
        return isCompiled;
    }

    /**
     * Method to compile the query.
     * @param forExecute Whether to compile ready for execution
     */
    protected void compileInternal(boolean forExecute, Map parameterValues)
    {
        if (isCompiled)
        {
            return;
        }

        try
        {
            if (forExecute)
            {
                // Compile for execution
                if (NucleusLogger.QUERY.isDebugEnabled())
                {
                    NucleusLogger.QUERY.debug(LOCALISER.msg("021044", getLanguage(), getSingleStringQuery()));
                }

                // Compile the query for execution
                JDOQLQueryCompiler c = new JDOQLQueryCompiler(this, getParsedImports(), parameterValues);
                queryStmt = (QueryExpression)c.compile(QueryCompiler.COMPILE_EXECUTION);
                if (resultDistinct)
                {
                    queryStmt.setDistinctResults(true);
                }
                resultMetaData = c.getResultMetaData();
                candidateClass = c.getCandidateClass();
                candidates = c.getCandidates();
                resultClass = c.getResultClass();
                fromInclNo = c.getRangeFromIncl();
                toExclNo = c.getRangeToExcl();
                c.close();
                c = null;
            }
            else
            {
                // Use the compiler to perform a "pre-compile" check to validate the expressions
                if (NucleusLogger.QUERY.isDebugEnabled())
                {
                    NucleusLogger.QUERY.debug(LOCALISER.msg("021044", getLanguage(), getSingleStringQuery()));
                }

                // Compile the query syntactically
                JDOQLQueryCompiler c = new JDOQLQueryCompiler(this, getParsedImports(), null);
                c.compile(QueryCompiler.COMPILE_SYNTAX);
                resultMetaData = c.getResultMetaData();
                candidateClass = c.getCandidateClass();
                candidates = c.getCandidates();
                resultClass = c.getResultClass();
                fromInclNo = c.getRangeFromIncl();
                toExclNo = c.getRangeToExcl();
                c.close();
                c = null;
            }

            isCompiled = true;
        }
        catch (NucleusException jpe)
        {
            // Compile failed
            discardCompiled();
            isCompiled = false;
            throw jpe;
        }
    }

    /**
     * Retrieve the metadata for the results.
     * @return the ResultSetMetaData
     */
    public QueryResultsMetaData getResultSetMetaData()
    {
        if (resultMetaData == null)
        {
            throw new NucleusUserException("You must compile the query before calling this method.");
        }
        return resultMetaData;
    }

    /**
     * Method to execute the actual query.
     * Discards any existing compilation forcing a recompile, then calls the superclass variant.
     * @param parameters Map of parameter values keyed by parameter name
     * @return Result
     * @throws NoQueryResultsException Thrown if no results were returned from the query.
     * @throws QueryNotUniqueException Thrown if multiple results, yet expected one
     */
    protected Object executeQuery(Map parameters)
    {
        discardCompiled();
        return super.executeQuery(parameters);
    }

    /**
     * Execute the query and return the filtered QueryResult.
     * @param parameters Map containing all of the parameters.
     * @return the filtered QueryResult.
     */
    protected Object performExecute(Map parameters)
    {
        if (candidates.isEmpty())
        {
            // if the candidates are empty, we don't have to go to the database
            // to find out that we have no elements to return
            return new ArrayList();
        }

        // Apply the Query FetchPlan to the query
        // Note : we could have added getFetchPlan() to Queryable, but this would affect *many* classes
        // so leaving like this for now
        if (candidates instanceof CollectionCandidates)
        {
            ((CollectionCandidates)candidates).getFetchPlan().setGroups(getFetchPlan().getGroups());
        }
        else if (candidates instanceof Extent)
        {
            ((Extent)candidates).getFetchPlan().setGroups(getFetchPlan().getGroups());
        }

        // Create a processor for extracting the objects from the ResultSet
        rof = candidates.newResultObjectFactory(queryStmt, getIgnoreCache(), resultClass, useFetchPlan());
        if (rof instanceof PersistentClassROF)
        {
            // Allow for the user specifying an Extent of a base class but restricting this query to a subclass
            ((PersistentClassROF)rof).setPersistentClass(candidateClass);
        }

        // Execute the query
        if (NucleusLogger.QUERY.isDebugEnabled())
        {
            NucleusLogger.QUERY.debug(LOCALISER.msg("021046", getLanguage(), getSingleStringQuery()));
        }
        SQLEvaluator eval = new SQLEvaluator(this, rof, candidateCollection);
        QueryResult qr = (QueryResult)eval.evaluate(queryStmt);
        queryResults.add(qr);

        return qr;
    }

    /**
     * Execute the query to delete persistent objects.
     * @param parameters the Map containing all of the parameters.
     * @return the number of deleted objects.
     */
    protected long performDeletePersistentAll(Map parameters)
    {
        // Discard compilation before deleting - will be recompiled in the superclass implementation
        discardCompiled();
        return super.performDeletePersistentAll(parameters);
    }

    /**
     * Method to return if the query results should have the range checked and unnecessary rows
     * discarded. This is for where the query language has specified a range but the datastore doesnt
     * allow removal of unnecessary results in the query itself (so has to be done in post-processing).
     * This implementation returns false and so should be overridden by query languages to match their
     * capabilities.
     * @return Whether to apply range checks in post-processing of results.
     */
    protected boolean applyRangeChecks()
    {
        if (range == null)
        {
            // No range specified so don't apply checks!
            return false;
        }

        RDBMSStoreManager storeMgr = (RDBMSStoreManager)om.getStoreManager();
        RDBMSAdapter dba = (RDBMSAdapter)storeMgr.getDatastoreAdapter();
        boolean using_limit_select_clause = (dba.getRangeByLimitSelectClause(fromInclNo, toExclNo).length() > 0);
        boolean using_limit_where_clause = (dba.getRangeByLimitWhereClause(fromInclNo, toExclNo).length() > 0);
        boolean using_rownum = (dba.getRangeByRowNumberColumn().length() > 0);

        return (range != null && !using_limit_select_clause && !using_limit_where_clause && !using_rownum);
    }

    /**
     * Method to return the names of the extensions supported by this query.
     * To be overridden by subclasses where they support additional extensions.
     * @return The supported extension names
     */
    public Set<String> getSupportedExtensions()
    {
        Set<String> supported = super.getSupportedExtensions();
        supported.add("datanucleus.rdbms.query.resultSetType");
        supported.add("datanucleus.rdbms.query.resultSetConcurrency");
        supported.add("datanucleus.rdbms.query.useUpdateLock");
        supported.add("datanucleus.rdbms.query.fetchDirection");
        supported.add("datanucleus.rdbms.jdoql.joinType");
        supported.add("datanucleus.rdbms.jdoql.existsIncludesConstraints");
        supported.add("datanucleus.rdbms.query.containsUsesExistsAlways");
        return supported;
    }

    /**
     * Accessor for the parsed imports.
     * If no imports are set then adds candidate class, user imports, and any user-defined expression packages.
     * @return Parsed imports
     */
    public Imports getParsedImports()
    {
        if (parsedImports == null)
        {
            super.getParsedImports();

            // Add any imports for expressions etc
            Iterator it = QueryCompiler.getUserDefinedScalarExpressions().keySet().iterator();
            while (it.hasNext())
            {
                parsedImports.importClass((String)it.next());
            }
        }
        return parsedImports;
    }
}