/**********************************************************************
Copyright (c) 2003 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2004 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.query.legacy;

import java.util.Collection;
import java.util.Iterator;

import org.datanucleus.FetchPlan;
import org.datanucleus.ObjectManager;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.MappedStoreManager;
import org.datanucleus.store.mapped.expression.BooleanExpression;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.Queryable;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.query.ResultObjectFactory;
import org.datanucleus.util.Localiser;

/**
 * Collection for candidates passed to the query by setCandidates(collection).
 */
public class CollectionCandidates implements Queryable
{
    protected static final Localiser LOCALISER=Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    /** User-specified collection of possible candidates. */
    private Collection userCandidates;

    /** Extent for the candidate class. */
    private org.datanucleus.store.Extent extent;

    /**
     * Constructor.
     * @param om Object Manager
     * @param candidateClass the Class candidate
     * @param candidates The candidates
     */
    public CollectionCandidates(ObjectManager om, Class candidateClass, Collection candidates)
    {
        if (candidates == null)
        {
            throw new NucleusException(LOCALISER.msg("021072")).setFatal();
        }

        this.userCandidates = candidates;
        this.extent = om.getExtent(candidateClass, true);
    }

    /**
     * @return Returns the userCandidates.
     */
    public Collection getUserCandidates()
    {
        return userCandidates;
    }

    /**
     * Accessor for the fetch plan
     * @return The fetch plan
     */
    public FetchPlan getFetchPlan()
    {
        return extent.getFetchPlan();
    }

    /**
     * Accessor for a new Query statement.
     * @return The Query Statement
     */
    public QueryExpression newQueryStatement()
    {
        return ((Queryable) extent).newQueryStatement();
    }

	/**
     * Creates a QueryStatement. The elements that are ALLOWED to be returned
     * after quering the database are the set of elements contained in the candidate collection.
     * @param candidateClass Candidate class
     * @param candidateAlias Alias to use for the candidate in the query (if the native query supports it)
     * @return Query Statement
     */
    public QueryExpression newQueryStatement(Class candidateClass, DatastoreIdentifier candidateAlias)
    {
        QueryExpression stmt = ((Queryable)extent).newQueryStatement(candidateClass, candidateAlias);
        ObjectManager om = extent.getObjectManager();
        MappedStoreManager storeMgr = (MappedStoreManager)om.getStoreManager();
        JavaTypeMapping m = storeMgr.getDatastoreClass(candidateClass.getName(), 
            om.getClassLoaderResolver()).getIdMapping();

        /*
         * creates a query like WHERE ... AND (CANDIDATE_ID = ?1 OR CANDIDATE_ID =
         * ?2) or for classes with composite primary keys WHERE ... AND (
         * (CANDIDATE_IDa = ?1a AND CANDIDATE_IDb = ?1b) OR (CANDIDATE_IDa = ?2a
         * AND CANDIDATE_IDb = ?2b) )
         */
        BooleanExpression elementsExpr = null;
        for (Iterator it = userCandidates.iterator(); it.hasNext();)
        {
            Object candidateValue = it.next();
            ScalarExpression expr = m.newScalarExpression(stmt, stmt.getMainTableExpression());
            BooleanExpression keyExpr = expr.eq(m.newLiteral(stmt, candidateValue));
            if (elementsExpr == null)
            {
                elementsExpr = keyExpr;
            }
            else
            {
                elementsExpr = elementsExpr.ior(keyExpr);
            }
        }
        if (elementsExpr != null)
        {
            stmt.andCondition(elementsExpr, true);
        }
        return stmt;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.query.Queryable#newResultObjectFactory(org.datanucleus.store.QueryStatement, boolean)
     */
    public ResultObjectFactory newResultObjectFactory(QueryExpression stmt, boolean ignoreCache, Class resultClass,
            boolean useFetchPlan)
    {
        return ((Queryable) extent).newResultObjectFactory(stmt, ignoreCache, resultClass, useFetchPlan);
    }

    /**
     * Returns <tt>true</tt> if this collection contains no elements.<p>
     * @return <tt>true</tt> if this collection contains no elements.
     */
    public boolean isEmpty()
    {
        return userCandidates.isEmpty();
    }
}