/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Andy Jefferson - commented and javadocs
2003 Andy Jefferson - added localiser
2003 Andy Jefferson - added sequence methods
2004 Erik Bengtson - added auto increment
2004 Erik Bengtson - added query operators, sql expressions
2004 Andy Jefferson - removed IndexMapping/OptimisticMapping
2008 Andy Jefferson - option Strings
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.adapter;

import java.math.BigInteger;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.OMFContext;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.plugin.PluginManager;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.IdentifierFactory;
import org.datanucleus.store.mapped.IdentifierType;
import org.datanucleus.store.mapped.MappedStoreManager;
import org.datanucleus.store.mapped.expression.BooleanExpression;
import org.datanucleus.store.mapped.expression.CharacterExpression;
import org.datanucleus.store.mapped.expression.CharacterLiteral;
import org.datanucleus.store.mapped.expression.ConcatOperatorExpression;
import org.datanucleus.store.mapped.expression.Literal;
import org.datanucleus.store.mapped.expression.LogicSetExpression;
import org.datanucleus.store.mapped.expression.NumericExpression;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.store.mapped.expression.SqlTemporalExpression;
import org.datanucleus.store.mapped.expression.StringExpression;
import org.datanucleus.store.mapped.expression.StringLiteral;
import org.datanucleus.store.mapped.expression.SubstringExpression;
import org.datanucleus.store.mapped.expression.TableExprAsJoins;
import org.datanucleus.store.mapped.expression.ScalarExpression.IllegalArgumentTypeException;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.mapped.mapping.MappingManager;
import org.datanucleus.store.rdbms.JDBCUtils;
import org.datanucleus.store.rdbms.RDBMSStoreManager;
import org.datanucleus.store.rdbms.key.CandidateKey;
import org.datanucleus.store.rdbms.key.ForeignKey;
import org.datanucleus.store.rdbms.key.Index;
import org.datanucleus.store.rdbms.key.PrimaryKey;
import org.datanucleus.store.rdbms.mapping.RDBMSMappingManager;
import org.datanucleus.store.rdbms.query.legacy.QueryStatement;
import org.datanucleus.store.rdbms.schema.ForeignKeyInfo;
import org.datanucleus.store.rdbms.schema.RDBMSColumnInfo;
import org.datanucleus.store.rdbms.schema.RDBMSTypesInfo;
import org.datanucleus.store.rdbms.schema.SQLTypeInfo;
import org.datanucleus.store.rdbms.sql.expression.SQLExpression;
import org.datanucleus.store.rdbms.table.Column;
import org.datanucleus.store.rdbms.table.Table;
import org.datanucleus.store.rdbms.table.TableImpl;
import org.datanucleus.store.rdbms.table.ViewImpl;
import org.datanucleus.store.schema.StoreSchemaHandler;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.Localiser;

/**
 * Provides methods for adapting SQL language elements to a specific vendor's
 * database.  A database adapter is primarily used to map generic JDBC data
 * types and SQL identifiers to specific types/identifiers suitable for the
 * database in use.
 *
 * <p>Each database adapter corresponds to a particular combination of database,
 * database version, driver, and driver version, as provided by the driver's
 * own metadata.  Database adapters cannot be constructed directly, but must be
 * obtained using the {@link org.datanucleus.store.rdbms.adapter.RDBMSAdapterFactory} class.</p>
 *
 * @see RDBMSAdapterFactory
 * @see java.sql.DatabaseMetaData
 */
public class DatabaseAdapter implements RDBMSAdapter
{
    protected static final Localiser LOCALISER_BASE = Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    protected static final Localiser LOCALISER=Localiser.getInstance("org.datanucleus.store.rdbms.Localisation",
        RDBMSStoreManager.class.getClassLoader());

    /** The set of reserved keywords for this datastore. */
    protected final HashSet<String> reservedKeywords = new HashSet();

    /** The product name of the underlying datastore. */
    protected String datastoreProductName;

    /** The version number of the underlying datastore as a string. */
    protected String datastoreProductVersion;

    /** The major version number of the underlying datastore. */
    protected int datastoreMajorVersion;

    /** The minor version number of the underlying datastore. */
    protected int datastoreMinorVersion;

    /** The revision version number of the underlying datastore. */
    protected int datastoreRevisionVersion = 0;

    /** The String used to quote identifiers. */
    protected String identifierQuoteString;

    /** Supported option names. */
    protected Collection<String> supportedOptions = new HashSet();

    /** the JDBC driver name **/
    protected String driverName;

    /** the JDBC driver version **/
    protected String driverVersion;

    /** The major version number of the underlying driver. */
    protected int driverMajorVersion;

    /** The minor version number of the underlying driver. */
    protected int driverMinorVersion;

    /** The maximum length to be used for a table name. */
    protected int maxTableNameLength;

    /** The maximum length to be used for a table constraint name. */
    protected int maxConstraintNameLength;

    /** The maximum length to be used for an index name. */
    protected int maxIndexNameLength;

    /** The maximum length to be used for a column name. */
    protected int maxColumnNameLength;

    /** The String used to separate catalog and table name. */
    protected String catalogSeparator;

    /** Optional properties controlling the configuration. */
    protected Map<String, Object> properties = null;

    /**
     * Constructs a database adapter based on the given JDBC metadata.
     * @param metadata the database metadata.
     */
    protected DatabaseAdapter(DatabaseMetaData metadata)
    {
        super();

        reservedKeywords.addAll(parseKeywordList(SQLConstants.SQL92_RESERVED_WORDS));
        reservedKeywords.addAll(parseKeywordList(SQLConstants.SQL99_RESERVED_WORDS));
        reservedKeywords.addAll(parseKeywordList(SQLConstants.SQL2003_RESERVED_WORDS));
        reservedKeywords.addAll(parseKeywordList(SQLConstants.NONRESERVED_WORDS));

        try
        {
            reservedKeywords.addAll(parseKeywordList(metadata.getSQLKeywords()));

            driverMinorVersion = metadata.getDriverMinorVersion();
            driverMajorVersion = metadata.getDriverMajorVersion();
            driverName = metadata.getDriverName();
            driverVersion = metadata.getDriverVersion();
            datastoreProductName = metadata.getDatabaseProductName();
            datastoreProductVersion = metadata.getDatabaseProductVersion();

            // Try to convert the "version" string into a W.X.Y.Z version string
            StringBuffer strippedProductVersion=new StringBuffer();
            char previousChar = ' ';
            for (int i=0; i<datastoreProductVersion.length(); ++i)
            {
                char c = datastoreProductVersion.charAt(i);
                if (Character.isDigit(c) || c == '.')
                {
                    // Only update the stripped version when we have "X."
                    if (previousChar != ' ')
                    {
                        if (strippedProductVersion.length() == 0)
                        {
                            strippedProductVersion.append(previousChar);
                        }
                        strippedProductVersion.append(c);
                    }
                    previousChar = c;
                }
                else
                {
                    previousChar = ' ';
                }
            }

            try
            {
                Class mdc = metadata.getClass();

                datastoreMajorVersion = ((Integer)mdc.getMethod("getDatabaseMajorVersion").invoke(metadata)).intValue();
                datastoreMinorVersion = ((Integer)mdc.getMethod("getDatabaseMinorVersion").invoke(metadata)).intValue();

                boolean noDBVersion = false;
                if (datastoreMajorVersion <= 0 && datastoreMinorVersion <= 0)
                {
                    // Check for the crap that they package with Websphere returning major/minor as 0.0
                    noDBVersion = true;
                }

                // Retrieve the Revision version if it is accessible
                StringTokenizer parts = new StringTokenizer(strippedProductVersion.toString(), ".");
                if (parts.hasMoreTokens())
                {
                    // Major version
                    if (noDBVersion)
                    {
                        try
                        {
                            datastoreMajorVersion = Integer.parseInt(parts.nextToken());
                        }
                        catch (Exception e)
                        {
                            datastoreMajorVersion = -1; //unknown
                        }
                    }
                    else
                    {
                        // already have it, so ignore this
                        parts.nextToken();
                    }
                }
                if (parts.hasMoreTokens())
                {
                    // Minor Version
                    if (noDBVersion)
                    {
                        try
                        {
                            datastoreMinorVersion = Integer.parseInt(parts.nextToken());
                        }
                        catch (Exception e)
                        {
                            datastoreMajorVersion = -1; //unknown
                        }
                    }
                    else
                    {
                        // already have it, so ignore this
                        parts.nextToken();
                    }
                }
                if (parts.hasMoreTokens())
                {
                    // Revision Version
                    try
                    {
                        datastoreRevisionVersion = Integer.parseInt(parts.nextToken());
                    }
                    catch (Exception e)
                    {
                        datastoreRevisionVersion = -1; //unknown
                    }
                }
            }
            catch (Throwable t)
            {
                /*
                 * The driver doesn't support JDBC 3.  Try to parse major and
                 * minor version numbers out of the product version string.
                 * We do this by stripping out everything but digits and periods
                 * and hoping we get something that looks like <major>.<minor>.
                 */
                StringTokenizer parts = new StringTokenizer(strippedProductVersion.toString(), ".");
                if (parts.hasMoreTokens())
                {
                    try
                    {
                        datastoreMajorVersion = Integer.parseInt(parts.nextToken());
                    }
                    catch (Exception e)
                    {
                        datastoreMajorVersion = -1; //unknown
                    }
                }
                if (parts.hasMoreTokens())
                {
                    try
                    {
                        datastoreMinorVersion = Integer.parseInt(parts.nextToken());
                    }
                    catch (Exception e)
                    {
                        datastoreMajorVersion = -1; //unknown
                    }
                }
                if (parts.hasMoreTokens())
                {
                    try
                    {
                        datastoreRevisionVersion = Integer.parseInt(parts.nextToken());
                    }
                    catch (Exception e)
                    {
                        datastoreRevisionVersion = -1; //unknown
                    }
                }
            }

            // Extract attributes of the Database adapter
            maxTableNameLength = metadata.getMaxTableNameLength();
            maxConstraintNameLength = metadata.getMaxTableNameLength();
            maxIndexNameLength = metadata.getMaxTableNameLength();
            maxColumnNameLength = metadata.getMaxColumnNameLength();
            if (metadata.supportsCatalogsInTableDefinitions())
            {
                supportedOptions.add(CATALOGS_IN_TABLE_DEFINITIONS);
            }
            if (metadata.supportsSchemasInTableDefinitions())
            {
                supportedOptions.add(SCHEMAS_IN_TABLE_DEFINITIONS);
            }
            if (metadata.supportsBatchUpdates())
            {
                supportedOptions.add(STATEMENT_BATCHING);
            }

            // Save the identifier cases available
            if (metadata.storesLowerCaseIdentifiers())
            {
                supportedOptions.add(IDENTIFIERS_LOWERCASE);
            }
            if (metadata.storesMixedCaseIdentifiers())
            {
                supportedOptions.add(IDENTIFIERS_MIXEDCASE);
            }
            if (metadata.storesUpperCaseIdentifiers())
            {
                supportedOptions.add(IDENTIFIERS_UPPERCASE);
            }
            if (metadata.storesLowerCaseQuotedIdentifiers())
            {
                supportedOptions.add(IDENTIFIERS_LOWERCASE_QUOTED);
            }
            if (metadata.storesMixedCaseQuotedIdentifiers())
            {
                supportedOptions.add(IDENTIFIERS_MIXEDCASE_QUOTED);
            }
            if (metadata.storesUpperCaseQuotedIdentifiers())
            {
                supportedOptions.add(IDENTIFIERS_UPPERCASE_QUOTED);
            }
            if (metadata.supportsMixedCaseIdentifiers())
            {
                supportedOptions.add(IDENTIFIERS_MIXEDCASE_SENSITIVE);
            }
            if (metadata.supportsMixedCaseQuotedIdentifiers())
            {
                supportedOptions.add(IDENTIFIERS_MIXEDCASE_QUOTED_SENSITIVE);
            }

            // Retrieve the catalog separator string (default = ".")
            catalogSeparator = metadata.getCatalogSeparator();
            catalogSeparator =
                ((catalogSeparator == null) || (catalogSeparator.trim().length() < 1)) ? "." : catalogSeparator;

            // Retrieve the identifier quote string (default = "")
            identifierQuoteString = metadata.getIdentifierQuoteString();
            identifierQuoteString =
                ((null == identifierQuoteString) || (identifierQuoteString.trim().length() < 1)) ?
                "\"" : identifierQuoteString;
        }
        catch (SQLException e)
        {
            throw new NucleusDataStoreException(LOCALISER.msg("051004"), e);
        }

        supportedOptions.add(VIEWS);
        supportedOptions.add(DATETIME_STORES_MILLISECS);
        supportedOptions.add(ESCAPE_EXPRESSION_IN_LIKE_PREDICATE);
        supportedOptions.add(UNION_SYNTAX);
        supportedOptions.add(EXISTS_SYNTAX);
        supportedOptions.add(ALTER_TABLE_DROP_CONSTRAINT_SYNTAX);
        supportedOptions.add(DEFERRED_CONSTRAINTS);
        supportedOptions.add(DISTINCT_WITH_SELECT_FOR_UPDATE);
        supportedOptions.add(PERSIST_OF_UNASSIGNED_CHAR);
        supportedOptions.add(CHECK_IN_CREATE_STATEMENTS);
        supportedOptions.add(GET_GENERATED_KEYS_STATEMENT);
        supportedOptions.add(BOOLEAN_COMPARISON);
        supportedOptions.add(NULLS_IN_CANDIDATE_KEYS);
        supportedOptions.add(NULLS_KEYWORD_IN_COLUMN_OPTIONS);
        supportedOptions.add(DEFAULT_KEYWORD_IN_COLUMN_OPTIONS);
        supportedOptions.add(DEFAULT_KEYWORD_WITH_NOT_NULL_IN_COLUMN_OPTIONS);
        supportedOptions.add(DEFAULT_BEFORE_NULL_IN_COLUMN_OPTIONS);
        supportedOptions.add(ANSI_JOIN_SYNTAX);
        supportedOptions.add(AUTO_INCREMENT_KEYS_NULL_SPECIFICATION);
        supportedOptions.add(AUTO_INCREMENT_COLUMN_TYPE_SPECIFICATION);
        supportedOptions.add(INCLUDE_ORDERBY_COLS_IN_SELECT);
        supportedOptions.add(ACCESS_PARENTQUERY_IN_SUBQUERY_JOINED);

        supportedOptions.add(FK_DELETE_ACTION_CASCADE);
        supportedOptions.add(FK_DELETE_ACTION_RESTRICT);
        supportedOptions.add(FK_DELETE_ACTION_DEFAULT);
        supportedOptions.add(FK_DELETE_ACTION_NULL);
        supportedOptions.add(FK_UPDATE_ACTION_CASCADE);
        supportedOptions.add(FK_UPDATE_ACTION_RESTRICT);
        supportedOptions.add(FK_UPDATE_ACTION_DEFAULT);
        supportedOptions.add(FK_UPDATE_ACTION_NULL);

        supportedOptions.add(TX_ISOLATION_READ_COMMITTED);
        supportedOptions.add(TX_ISOLATION_READ_UNCOMMITTED);
        supportedOptions.add(TX_ISOLATION_REPEATABLE_READ);
        supportedOptions.add(TX_ISOLATION_SERIALIZABLE);
    }

    /**
     * Creates the auxiliary functions/procedures in the schema 
     * @param conn the connection to the datastore
     */
    public void initialiseDatastore(Object conn)
    {
    }

    /**
     * Initialise the types for this datastore.
     * @param handler SchemaHandler that we initialise the types for
     * @param mconn Managed connection to use
     */
    public void initialiseTypes(StoreSchemaHandler handler, ManagedConnection mconn)
    {
        // Initialise the mappings available. Load all possible, and remove unsupported for this datastore
        RDBMSStoreManager storeMgr = (RDBMSStoreManager)handler.getStoreManager();
        ClassLoaderResolver clr = storeMgr.getOMFContext().getClassLoaderResolver(null);
        PluginManager pluginMgr = storeMgr.getOMFContext().getPluginManager();
        MappingManager mapMgr = storeMgr.getMappingManager();
        mapMgr.loadDatastoreMapping(pluginMgr, clr, getVendorID());

        // Load the types from plugin(s)
        handler.getSchemaData(mconn.getConnection(), "types", null);
    }

    /**
     * Set any properties controlling how the adapter is configured.
     * @param props The properties
     */
    public void setProperties(Map<String, Object> props)
    {
        if (props != null)
        {
            properties = new HashMap<String, Object>();
        }
        properties.putAll(props);
    }

    /**
     * Accessor for a property. Null imples not defined
     * @param name Name of the property
     * @return Its value
     */
    public Object getValueForProperty(String name)
    {
        return (properties != null ? properties.get(name) : null);
    }

    /**
     * Remove all mappings from the mapping manager that don't have a datastore type initialised.
     * @param handler Schema handler
     * @param mconn Managed connection to use
     */
    public void removeUnsupportedMappings(StoreSchemaHandler handler, ManagedConnection mconn)
    {
        RDBMSStoreManager storeMgr = (RDBMSStoreManager)handler.getStoreManager();
        RDBMSMappingManager mapMgr = (RDBMSMappingManager)storeMgr.getMappingManager();
        RDBMSTypesInfo types = (RDBMSTypesInfo)handler.getSchemaData(mconn.getConnection(), "types", null);

        int[] jdbcTypes = JDBCUtils.getJDBCTypes();
        for (int i=0;i<jdbcTypes.length;i++)
        {
            if (types.getChild("" + jdbcTypes[i]) == null)
            {
                // JDBC type not supported by adapter so deregister the mapping
                // Means that we don't need to add "excludes" definitions to plugin.xml
                mapMgr.deregisterDatastoreMappingsForJDBCType(JDBCUtils.getNameForJDBCType(jdbcTypes[i]));
            }
        }
    }

    /**
     * Accessor for the SQLType info for the specified JDBC type and the SQL type name.
     * @param handler Schema handler
     * @param mconn Connection
     * @param jdbcTypeNumber JDBC type
     * @return The SQL type info
     */
    protected Collection<SQLTypeInfo> getSQLTypeInfoForJdbcType(StoreSchemaHandler handler, ManagedConnection mconn,
            short jdbcTypeNumber)
    {
        RDBMSTypesInfo types = (RDBMSTypesInfo)handler.getSchemaData(mconn.getConnection(), "types", null);

        String key = "" + jdbcTypeNumber;
        org.datanucleus.store.rdbms.schema.JDBCTypeInfo jdbcType =
            (org.datanucleus.store.rdbms.schema.JDBCTypeInfo)types.getChild(key);
        if (jdbcType == null)
        {
            return null;
        }
        return jdbcType.getChildren().values();
    }

    /**
     * Convenience method for use by adapters to add their own fake JDBC/SQL types in where the 
     * JDBC driver doesn't provide some type.
     * @param handler the schema handler managing the types
     * @param mconn Connection to use
     * @param jdbcTypeNumber The JDBC type
     * @param sqlType The type info to use
     * @param addIfNotPresent whether to add only if JDBC type not present
     */
    protected void addSQLTypeForJDBCType(StoreSchemaHandler handler, ManagedConnection mconn,
            short jdbcTypeNumber, SQLTypeInfo sqlType, boolean addIfNotPresent)
    {
        RDBMSTypesInfo types = (RDBMSTypesInfo)handler.getSchemaData(mconn.getConnection(), "types", null);
        String key = "" + jdbcTypeNumber;
        org.datanucleus.store.rdbms.schema.JDBCTypeInfo jdbcType =
            (org.datanucleus.store.rdbms.schema.JDBCTypeInfo)types.getChild(key);
        if (jdbcType != null && !addIfNotPresent)
        {
            // Already have this JDBC type so ignore
            return;
        }
        else if (jdbcType == null)
        {
            // New JDBC type
            jdbcType = new org.datanucleus.store.rdbms.schema.JDBCTypeInfo(jdbcTypeNumber);
            types.addChild(jdbcType);
            jdbcType.addChild(sqlType);
        }
        else
        {
            // Existing JDBC type so add SQL type
            jdbcType.addChild(sqlType);
        }
    }

    /**
     * Accessor for whether this database adapter supports the specified transaction isolation.
     * @param level The isolation level (as defined by Connection enums).
     * @return Whether it is supported.
     */
    public boolean supportsTransactionIsolation(int level)
    {
        if ((level == Connection.TRANSACTION_NONE && supportsOption(RDBMSAdapter.TX_ISOLATION_NONE)) ||
            (level == Connection.TRANSACTION_READ_COMMITTED && supportsOption(RDBMSAdapter.TX_ISOLATION_READ_COMMITTED)) ||
            (level == Connection.TRANSACTION_READ_UNCOMMITTED && supportsOption(RDBMSAdapter.TX_ISOLATION_READ_UNCOMMITTED)) ||
            (level == Connection.TRANSACTION_REPEATABLE_READ && supportsOption(RDBMSAdapter.TX_ISOLATION_REPEATABLE_READ)) ||
            (level == Connection.TRANSACTION_SERIALIZABLE && supportsOption(RDBMSAdapter.TX_ISOLATION_SERIALIZABLE)))
        {
            return true;
        }
        return false;
    }

    /**
     * Convenience method to log the configuration of this datastore adapter.
     */
    public void logConfiguration()
    {
        NucleusLogger.DATASTORE.debug("Datastore Adapter : " + this.getClass().getName());
        NucleusLogger.DATASTORE.debug("Datastore Details : name=\"" + datastoreProductName + "\" version=\"" + datastoreProductVersion + 
            "\" (major=" + datastoreMajorVersion + ", minor=" + datastoreMinorVersion + ", revision=" + datastoreRevisionVersion + ")");
        NucleusLogger.DATASTORE.debug("Datastore Driver : name=\"" + driverName + "\" version=\"" + driverVersion + 
            "\" (major=" + driverMajorVersion + ", minor=" + driverMinorVersion + ")");
        NucleusLogger.DATASTORE.debug("Supported Identifier Cases : " +
            (supportsOption(IDENTIFIERS_LOWERCASE) ? "lowercase " : "") +
            (supportsOption(IDENTIFIERS_LOWERCASE_QUOTED) ? "\"lowercase\" " : "") +
            (supportsOption(IDENTIFIERS_MIXEDCASE) ? "MixedCase " : "") +
            (supportsOption(IDENTIFIERS_MIXEDCASE_QUOTED) ? "\"MixedCase\" " : "") +
            (supportsOption(IDENTIFIERS_UPPERCASE) ? "UPPERCASE " : "") +
            (supportsOption(IDENTIFIERS_UPPERCASE_QUOTED) ? "\"UPPERCASE\" " : "") +
            (supportsOption(IDENTIFIERS_MIXEDCASE_SENSITIVE) ? "MixedCase-Sensitive " : "") +
            (supportsOption(IDENTIFIERS_MIXEDCASE_QUOTED_SENSITIVE) ? "\"MixedCase-Sensitive\" " : ""));
       NucleusLogger.DATASTORE.debug("Supported Identifier Lengths (max) :" +
            " Table=" + getDatastoreIdentifierMaxLength(IdentifierType.TABLE) +
            " Column=" + getDatastoreIdentifierMaxLength(IdentifierType.COLUMN) +
            " Constraint=" + getDatastoreIdentifierMaxLength(IdentifierType.CANDIDATE_KEY) +
            " Index=" + getDatastoreIdentifierMaxLength(IdentifierType.INDEX) +
            " Delimiter=" + getIdentifierQuoteString());
        NucleusLogger.DATASTORE.debug("Support for Identifiers in DDL :" +
            " catalog=" + supportsOption(CATALOGS_IN_TABLE_DEFINITIONS) +
            " schema=" + supportsOption(SCHEMAS_IN_TABLE_DEFINITIONS));
        NucleusLogger.DATASTORE.debug("Support Statement Batching : " + (supportedOptions.contains(STATEMENT_BATCHING) ? "yes" : "no"));
    }

    /**
     * Accessor for the options that are supported by this datastore adapter and the underlying datastore.
     * @return The options (Collection<String>)
     */
    public Collection<String> getSupportedOptions()
    {
        return supportedOptions;
    }

    /**
     * Accessor for whether the supplied option is supported.
     * @param option The option
     * @return Whether supported.
     */
    public boolean supportsOption(String option)
    {
        return supportedOptions.contains(option);
    }

    /**
     * Accessor for a MappingManager suitable for use with this datastore adapter.
     * @param storeMgr The StoreManager
     * @return the MappingManager
     */
    public MappingManager getMappingManager(MappedStoreManager storeMgr)
    {
        return new RDBMSMappingManager(storeMgr);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.DatastoreAdapter#getAdapterTime(java.sql.Timestamp)
     */
    public long getAdapterTime(Timestamp time)
    {
        long timestamp = getTime(time.getTime(), time.getNanos());
        int ms = getMiliseconds(time.getNanos());

        return timestamp + ms;
    }

    protected long getTime(long time, long nanos)
    {
        if (nanos < 0)
        {
            return (((time / 1000) - 1) * 1000);
        }
        return (time / 1000) * 1000;
    }

    protected int getMiliseconds(long nanos)
    {
        return (int) (nanos / 1000000);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.DatastoreAdapter#getDatastoreMajorVersion()
     */
    public int getDatastoreMajorVersion()
    {
        return datastoreMajorVersion;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.DatastoreAdapter#getDatastoreMinorVersion()
     */
    public int getDatastoreMinorVersion()
    {
        return datastoreMinorVersion;
    }

    /**
     * Whether the datastore will support setting the query fetch size to the supplied value.
     * @param size The value to set to
     * @return Whether it is supported.
     */
    public boolean supportsQueryFetchSize(int size)
    {
        // Default to supported all possible values
        return true;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.DatastoreAdapter#getVendorID()
     */
    public String getVendorID()
    {
        return null;
    }

    /**
     * Tests if a given string is a SQL keyword.
     * <p>
     * The list of key words tested against is defined to contain all SQL/92 keywords, plus any additional 
     * key words reported by the JDBC driver for this adapter via <code>DatabaseMetaData.getSQLKeywords()</code>.
     * <p>
     * In general, use of a SQL key word as an identifier should be avoided.
     * SQL/92 key words are divided into reserved and non-reserved words. If a reserved word is used as
     * an identifier it must be quoted with double quotes. Strictly speaking, the same is not true of 
     * non-reserved words. However, as C.J. Date writes in <u>A Guide To The SQL Standard </u>:
     * <blockquote>The rule by which it is determined within the standard that one key word needs to be 
     * reserved while another need not is not clear to this writer. In practice, it is probably wise to 
     * treat all key words as reserved.</blockquote>
     * @param word The word to test.
     * @return <code>true</code> if <var>word </var> is a SQL key word for this DBMS. 
     *     The comparison is case-insensitive.
     * @see SQLConstants
     */
    public boolean isReservedKeyword(String word)
    {
        return reservedKeywords.contains(word.toUpperCase());
    }

    /**
     * Accessor for an identifier quote string.
     * @return Identifier quote string.
     **/
    public String getIdentifierQuoteString()
    {
        return identifierQuoteString;
    }

    /**
     * Accessor for the JDBC driver major version
     * @return The driver major version
     */
    public int getDriverMajorVersion()
    {
        return driverMajorVersion;
    }

    /**
     * Accessor for the JDBC driver minor version
     * @return The driver minor version
     */
    public int getDriverMinorVersion()
    {
        return driverMinorVersion;
    }

    /**
     * Method to return the maximum length of a datastore identifier of the specified type.
     * If no limit exists then returns -1
     * @param identifierType Type of identifier (see IdentifierFactory.TABLE, etc)
     * @return The max permitted length of this type of identifier
     */
    public int getDatastoreIdentifierMaxLength(IdentifierType identifierType)
    {
        if (identifierType == IdentifierType.TABLE)
        {
            return maxTableNameLength;
        }
        else if (identifierType == IdentifierType.COLUMN)
        {
            return maxColumnNameLength;
        }
        else if (identifierType == IdentifierType.CANDIDATE_KEY)
        {
            return maxConstraintNameLength;
        }
        else if (identifierType == IdentifierType.FOREIGN_KEY)
        {
            return maxConstraintNameLength;
        }
        else if (identifierType == IdentifierType.INDEX)
        {
            return maxIndexNameLength;
        }
        else if (identifierType == IdentifierType.PRIMARY_KEY)
        {
            return maxConstraintNameLength;
        }
        else if (identifierType == IdentifierType.SEQUENCE)
        {
            return maxTableNameLength;
        }
        else
        {
            return -1;
        }
    }

    /**
     * Accessor for the maximum foreign keys by table permitted for this datastore.
     * @return Max number of FKs for a table
     **/
    public int getMaxForeignKeys()
    {
        // TODO This is arbitrary. Should be relative to the RDBMS in use
        return 9999;
    }

    /**
     * Accessor for the maximum indexes by schema permitted for this datastore.
     * @return Max number of indexes for a table
     **/
    public int getMaxIndexes()
    {
        // TODO This is arbitrary. Should be relative to the RDBMS in use
        return 9999;
    }

    /**
     * Iterator for the reserved words constructed from the method
     * DataBaseMetaData.getSQLKeywords + standard SQL reserved words
     * @return an Iterator with a set of reserved words
     */
    public Iterator iteratorReservedWords()
    {
        return reservedKeywords.iterator();
    }

    public RDBMSColumnInfo newRDBMSColumnInfo(ResultSet rs)
    {
        return new RDBMSColumnInfo(rs);
    }

    public SQLTypeInfo newSQLTypeInfo(ResultSet rs)
    {
        return new SQLTypeInfo(rs);
    }

    /**
     * Method to return ForeignKeyInfo for the current row of the ResultSet which will have been
     * obtained from a call to DatabaseMetaData.getImportedKeys() or DatabaseMetaData.getExportedKeys().
     * @param rs The result set returned from DatabaseMetaData.get??portedKeys()
     * @return The foreign key info 
     */
    public ForeignKeyInfo newFKInfo(ResultSet rs)
    {
        return new ForeignKeyInfo(rs);
    }

    /**
     * Utility method to parse a list of keywords and split them out into
     * words.
     * @param list The comma-separated list of keywords.
     * @return Set of keywords.
     **/
    protected Set parseKeywordList(String list)
    {
        StringTokenizer tokens = new StringTokenizer(list, ",");
        HashSet words = new HashSet();

        while (tokens.hasMoreTokens())
        {
            words.add(tokens.nextToken().trim().toUpperCase());
        }

        return words;
    }

    /**
     * Tests if a given string is a SQL key word.
     * <p>
     * The list of key words tested against is defined to contain all SQL/92 key
     * words, plus any additional key words reported by the JDBC driver for this
     * adapter via <code>DatabaseMetaData.getSQLKeywords()</code>.
     * <p>
     * In general, use of a SQL key word as an identifier should be avoided.
     * SQL/92 key words are divided into reserved and non-reserved words. If a
     * reserved word is used as an identifier it must be quoted with double
     * quotes. Strictly speaking, the same is not true of non-reserved words.
     * However, as C.J. Date writes in <u>A Guide To The SQL Standard </u>:
     * <blockquote>The rule by which it is determined within the standard that
     * one key word needs to be reserved while another need not is not clear to
     * this writer. In practice, it is probably wise to treat all key words as
     * reserved. </blockquote>
     * @param word The word to test.
     * @return <code>true</code> if <var>word </var> is a SQL key word for
     * this DBMS. The comparison is case-insensitive.
     * @see SQLConstants
     */
    public boolean isKeyword(String word)
    {
        return isReservedKeyword(word.toUpperCase());
    }

    /**
     * Returns the precision value to be used when creating string columns of "unlimited" length.
     * Usually, if this value is needed it is provided in.  However, for some types in some databases 
     * the value must be computed.
     * @param typeInfo the typeInfo object for which the precision value is needed.
     * @return the precision value to be used when creating the column, or -1 if no value should be used.
     */
    public int getUnlimitedLengthPrecisionValue(SQLTypeInfo typeInfo)
    {
        if (typeInfo.getCreateParams() != null && typeInfo.getCreateParams().length() > 0)
        {
            return typeInfo.getPrecision();
        }
        else
        {
            return -1;
        }
    }

    /**
     * Method to return whether the specified JDBC type is valid for use in a PrimaryKey.
     * @param datatype The JDBC type.
     * @return Whether it is valid for use in the PK
     */
    public boolean isValidPrimaryKeyType(int datatype)
    {
        // This is temporary since some RDBMS allow indexing of Blob/Clob/LongVarBinary
        // TODO Transfer to individual adapters
        if (datatype == Types.BLOB || datatype == Types.CLOB || datatype == Types.LONGVARBINARY)
        {
            return false;
        }
        return true;
    }

    /**
     * Some databases, Oracle, treats an empty string (0 length) equals null
     * @return returns a surrogate to replace the empty string in the database
     * otherwise it would be treated as null
     */
    public String getSurrogateForEmptyStrings()
    {
        return null;
    }

    /**
     * Accessor for the transaction isolation level to use during schema creation.
     * @return The transaction isolation level for schema generation process
     */
    public int getTransactionIsolationForSchemaCreation()
    {
        return Connection.TRANSACTION_SERIALIZABLE;
    }

    /**
     * Accessor for the "required" transaction isolation level if it has to be a certain value
     * for this adapter.
     * @return Transaction isolation level (-1 implies no restriction)
     */
    public int getRequiredTransactionIsolationLevel()
    {
        return -1;
    }

    /**
     * Accessor for the Catalog Name for this datastore.
     * @param conn Connection to the datastore
     * @return The catalog name
     * @throws SQLException Thrown if error occurs in determining the catalog name.
     **/
    public String getCatalogName(Connection conn)
    throws SQLException
    {
        throw new UnsupportedOperationException(LOCALISER.msg("051015",datastoreProductName,datastoreProductVersion));
    }

    /**
     * Accessor for the Schema Name for this datastore.
     * @param conn Connection to the datastore
     * @return The schema name
     * @throws SQLException Thrown if error occurs in determining the schema name.
     **/
    public String getSchemaName(Connection conn)
    throws SQLException
    {
        throw new UnsupportedOperationException(LOCALISER.msg("051016",datastoreProductName,datastoreProductVersion));
    }

    /**
     * Accessor for the catalog separator.
     * @return Catalog separator string.
     **/
    public String getCatalogSeparator()
    {
        return catalogSeparator;
    }

    /**
     * The option to specify in "SELECT ... FROM TABLE ... WITH (option)" to lock instances
     * Null if not supported.
     * @return The option to specify with "SELECT ... FROM TABLE ... WITH (option)"
     **/
    public String getSelectWithLockOption()
    {
        return null;
    }

    /**
     * Method returning the text to append to the end of the SELECT to perform the equivalent
     * of "SELECT ... FOR UPDATE" (on some RDBMS). This method means that we can have different
     * text with some datastores (e.g Derby).
     * @return The "FOR UPDATE" text
     */
    public String getSelectForUpdateText()
    {
        return "FOR UPDATE";
    }

    /**
     * The function to creates a unique value of type uniqueidentifier.
     * @return The function. e.g. "SELECT NEWID()"
     **/
    public String getSelectNewUUIDStmt()
    {
        return null;
    }
    
    /**
     * The function to creates a unique value of type uniqueidentifier.
     * @return The function. e.g. "NEWID()"
     **/
    public String getNewUUIDFunction()
    {
        return null;
    }

    /**
     * Convenience method to allow adaption of an ordering string before applying it.
     * This is useful where the datastore accepts some conversion adapter around the ordering column
     * for example.
     * @param omfCtx The OMF Context
     * @param orderString The basic ordering string
     * @param sqlExpr The sql expression being represented here
     * @return The adapted ordering string
     */
    public String getOrderString(OMFContext omfCtx, String orderString, SQLExpression sqlExpr)
    {
        return orderString;
    }

    // ---------------------------- AutoIncrement Support ----------------------

    /**
     * Accessor for the autoincrementing sql statement for this datastore.
     * @param table Name of the table that the autoincrement is for
     * @param columnName Name of the column that the autoincrement is for
     * @return The statement for getting the latest autoincremented key
     **/
    public String getAutoIncrementStmt(Table table, String columnName)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("051019"));
    }

    /**
     * Accessor for the autoincrementing keyword for generating DDLs.
     * (CREATE TABLEs...).
     * @return The keyword for a column using autoincrement
     **/
    public String getAutoIncrementKeyword()
    {
        throw new UnsupportedOperationException(LOCALISER.msg("051019"));
    }

    /**
     * Verifies if the given <code>typeName</code> is auto incremented by the datastore.
     * @param typeName the datastore type name
     * @return true when the <code>typeName</code> has values auto incremented by the datastore
     **/
    public boolean isIdentityFieldDataType(String typeName)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("051019"));
    }

    /**
     * Method to return the INSERT statement to use when inserting into a table that has no
     * columns specified. This is the case when we have a single column in the table and that column
     * is autoincrement/identity (and so is assigned automatically in the datastore).
     * @param table The table
     * @return The statement for the INSERT
     */
    public String getInsertStatementForNoColumns(Table table)
    {
        return "INSERT INTO " + table.toString() + " () VALUES ()";
    }

    // ---------------------------- Sequence Support ---------------------------

    /**
     * Accessor for the sequence create statement for this datastore.
     * @param sequence_name Name of the sequence 
     * @param min Minimum value for the sequence
     * @param max Maximum value for the sequence
     * @param start Start value for the sequence
     * @param increment Increment value for the sequence
     * @param cache_size Cache size for the sequence
     * @return The statement for getting the next id from the sequence
     */
    public String getSequenceCreateStmt(String sequence_name,
                                        Integer min,Integer max,
                                        Integer start,Integer increment,
                                        Integer cache_size)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("051020"));
    }

    /**
     * Accessor for the sequence statement to get the next id for this 
     * datastore.
     * @param sequence_name Name of the sequence 
     * @return The statement for getting the next id for the sequence
     **/
    public String getSequenceNextStmt(String sequence_name)
    {
        throw new UnsupportedOperationException(LOCALISER.msg("051020"));
    }

    /**
     * Provide the existing indexes in the database for the table.
     * This is implemented if and only if the datastore has its own way of getting indexes.
     * Otherwise we will use DatabaseMetaData.getIndexInfo().
     * The implementation here returns null.
     * @param conn the JDBC connection
     * @param catalog the catalog name
     * @param schema the schema name
     * @param table the table name
     * @return a ResultSet with the format @see DatabaseMetaData#getIndexInfo(java.lang.String, java.lang.String, java.lang.String, boolean, boolean)
     * @throws SQLException
     */
    public ResultSet getExistingIndexes(Connection conn, String catalog, String schema, String table) 
    throws SQLException
    {
        return null;
    }

    /**
     * Returns the appropriate SQL to create the given table having the given
     * columns. No column constraints or key definitions should be included.
     * It should return something like:
     * <p>
     * <pre>
     * CREATE TABLE FOO ( BAR VARCHAR(30), BAZ INTEGER )
     * </pre>
     *
     * @param table The table to create.
     * @param columns The columns of the table.
     * @param props Properties for controlling the table creation
     * @return The text of the SQL statement.
     */
    public String getCreateTableStatement(TableImpl table, Column[] columns, Properties props)
    {
        StringBuffer createStmt = new StringBuffer();

        // CREATE TABLE with column specifiers
        createStmt.append("CREATE TABLE ").append(table.toString())
                  .append(getContinuationString())
                  .append("(")
                  .append(getContinuationString());
        for (int i = 0; i < columns.length; ++i)
        {
            if (i > 0)
            {
                createStmt.append(",").append(getContinuationString());
            }

            createStmt.append("    ").append(columns[i].getSQLDefinition());
        }

        // PRIMARY KEY(col[,col])
        if (supportsOption(PRIMARYKEY_IN_CREATE_STATEMENTS))
        {
            PrimaryKey pk = table.getPrimaryKey();
            if (pk != null && pk.size() > 0)
            {
                createStmt.append(",").append(getContinuationString());
                createStmt.append("    ").append(pk.toString());
            }
        }

        // UNIQUE( col [,col] )
        if (supportsOption(UNIQUE_IN_END_CREATE_STATEMENTS))
        {
            StringBuffer uniqueConstraintStmt = new StringBuffer();
            for (int i = 0; i < columns.length; ++i)
            {
                if (columns[i].isUnique())
                {
                    if (uniqueConstraintStmt.length() < 1)
                    {
                        uniqueConstraintStmt.append(",").append(getContinuationString());
                        uniqueConstraintStmt.append(" UNIQUE (");
                    }
                    else
                    {
                        uniqueConstraintStmt.append(",");
                    }
                    uniqueConstraintStmt.append(columns[i].getIdentifier().toString());
                }
            }       
            if (uniqueConstraintStmt.length() > 1)
            {
                uniqueConstraintStmt.append(")");
                createStmt.append(uniqueConstraintStmt.toString());
            }
        }
        
        // CHECK (column_identifier IN (literal[,literal]))
        if (supportsOption(CHECK_IN_END_CREATE_STATEMENTS))
        {
            StringBuffer checkConstraintStmt = new StringBuffer();
	        for (int i = 0; i < columns.length; ++i)
	        {
	            if (columns[i].getConstraints() != null)
	            {
	                checkConstraintStmt.append(",").append(getContinuationString());
	                checkConstraintStmt.append(columns[i].getConstraints());
	            }
	        }
	        if (checkConstraintStmt.length() > 1)
	        {
	            createStmt.append(checkConstraintStmt.toString());
	        }
        }

        createStmt.append(getContinuationString()).append(")");

        return createStmt.toString();
    }

    /**
     * Returns the appropriate SQL to add a primary key to its table.
     * It should return something like:
     * <p>
     * <pre>
     * ALTER TABLE FOO ADD CONSTRAINT FOO_PK PRIMARY KEY (BAR)
     * ALTER TABLE FOO ADD PRIMARY KEY (BAR)
     * </pre>
     *
     * @param pk An object describing the primary key.
     * @param factory Identifier factory
     * @return The text of the SQL statement.
     */
    public String getAddPrimaryKeyStatement(PrimaryKey pk, IdentifierFactory factory)
    {
        if (pk.getName() != null)
        {
            String identifier = factory.getIdentifierInAdapterCase(pk.getName());
            return "ALTER TABLE " + pk.getDatastoreContainerObject().toString() + " ADD CONSTRAINT " + identifier + ' ' + pk;
        }
        else
        {
            return "ALTER TABLE " + pk.getDatastoreContainerObject().toString() + " ADD " + pk;
        }
    }

    /**
     * Returns the appropriate SQL to add a candidate key to its table.
     * It should return something like:
     * <p>
     * <pre>
     * ALTER TABLE FOO ADD CONSTRAINT FOO_CK UNIQUE (BAZ)
     * ALTER TABLE FOO ADD UNIQUE (BAZ)
     * </pre>
     *
     * @param ck An object describing the candidate key.
     * @param factory Identifier factory
     * @return The text of the SQL statement.
     */
    public String getAddCandidateKeyStatement(CandidateKey ck, IdentifierFactory factory)
    {
        if (ck.getName() != null)
        {
            String identifier = factory.getIdentifierInAdapterCase(ck.getName());
            return "ALTER TABLE " + ck.getDatastoreContainerObject().toString() + " ADD CONSTRAINT " + identifier + ' ' + ck;
        }
        else
        {
            return "ALTER TABLE " + ck.getDatastoreContainerObject().toString() + " ADD " + ck;
        }
    }

    /**
     * Returns the appropriate SQL to add a foreign key to its table.
     * It should return something like:
     * <p>
     * <pre>
     * ALTER TABLE FOO ADD CONSTRAINT FOO_FK1 FOREIGN KEY (BAR, BAZ) REFERENCES ABC (COL1, COL2)
     * ALTER TABLE FOO ADD FOREIGN KEY (BAR, BAZ) REFERENCES ABC (COL1, COL2)
     * </pre>
     * @param fk An object describing the foreign key.
     * @param factory Identifier factory
     * @return  The text of the SQL statement.
     */
    public String getAddForeignKeyStatement(ForeignKey fk, IdentifierFactory factory)
    {
        if (fk.getName() != null)
        {
            String identifier = factory.getIdentifierInAdapterCase(fk.getName());
            return "ALTER TABLE " + fk.getDatastoreContainerObject().toString() + " ADD CONSTRAINT " + identifier + ' ' + fk;
        }
        else
        {
            return "ALTER TABLE " + fk.getDatastoreContainerObject().toString() + " ADD " + fk;
        }
    }

    /**
     * Accessor for the SQL statement to add a column to a table.
     * @param table The table
     * @param col The column
     * @return The SQL necessary to add the column
     */
    public String getAddColumnStatement(DatastoreContainerObject table, Column col)
    {
        return "ALTER TABLE " + table.toString() + " ADD " + col.getSQLDefinition();
    }

    /**
     * Returns the appropriate SQL to add an index to its table.
     * It should return something like:
     * <p>
     * <pre>
     * CREATE INDEX FOO_N1 ON FOO (BAR,BAZ) [Extended Settings]
     * CREATE UNIQUE INDEX FOO_U1 ON FOO (BAR,BAZ) [Extended Settings]
     * </pre>
     *
     * @param idx An object describing the index.
     * @param factory Identifier factory
     * @return The text of the SQL statement.
     */
    public String getCreateIndexStatement(Index idx, IdentifierFactory factory)
    {
        String indexIdentifier = factory.getIdentifierInAdapterCase(idx.getName());
        return "CREATE " + (idx.getUnique() ? "UNIQUE " : "") + "INDEX " + indexIdentifier + " ON " + idx.getDatastoreContainerObject().toString() + ' ' + idx + (idx.getExtendedIndexSettings() == null ? "" : " "+idx.getExtendedIndexSettings());
    }

    /**
     * Creates a CHECK constraint definition based on the given values
     * e.g. <pre>CHECK ("COLUMN" IN ('VAL1','VAL2') OR "COLUMN" IS NULL)</pre>
     * @param identifier Column identifier
     * @param values Valid values
     * @param nullable whether the datastore identifier is null
     * @return The check constraint
     */
    public String getCheckConstraintForValues(DatastoreIdentifier identifier, Object[] values, boolean nullable)
    {
        StringBuffer constraints = new StringBuffer("CHECK (");
        constraints.append(identifier);
        constraints.append(" IN (");
        for (int i=0;i<values.length;i++)
        {
            if (i > 0)
            {
                constraints.append(",");
            }
            if (values[i] instanceof String)
            {
                constraints.append("'").append(values[i]).append("'");
            }
            else
            {
                constraints.append(values[i]);
            }
        }
        constraints.append(")");
        if (nullable)
        {
            constraints.append(" OR " + identifier + " IS NULL");
        }
        constraints.append(")");
        return constraints.toString();
    }

    /**
     * Returns the appropriate SQL to drop the given table.
     * It should return something like:
     * <p>
     * <pre>
     * DROP TABLE FOO CASCADE
     * </pre>
     *
     * @param table The table to drop.
     * @return The text of the SQL statement.
     */
    public String getDropTableStatement(DatastoreContainerObject table)
    {
        return "DROP TABLE " + table.toString() + " CASCADE";
    }

    /**
     * Returns the appropriate SQL to drop the given view.
     * It should return something like:
     * <p>
     * <pre>
     * DROP VIEW FOO
     * </pre>
     *
     * @param view The view to drop.
     * @return The text of the SQL statement.
     */
    public String getDropViewStatement(ViewImpl view)
    {
        return "DROP VIEW " + view.toString();
    }

    /**
     * Method to return the SQL to append to the SELECT clause of a SELECT statement to handle
     * restriction of ranges using the LIMIT keyword. Defaults to an empty
     * string (not supported).
     * SELECT {LIMIT} params
     * @param offset The offset to return from
     * @param count The number of items to return
     * @return The SQL to append to allow for ranges using LIMIT.
     */
    public String getRangeByLimitSelectClause(long offset, long count)
    {
        return "";
    }

    /**
     * Method to return the SQL to append to the WHERE clause of a SELECT statement to handle
     * restriction of ranges using the LIMIT keyword. Defaults to an empty
     * string (not supported).
     * SELECT param ... WHERE {LIMIT}
     * @param offset The offset to return from
     * @param count The number of items to return
     * @return The SQL to append to allow for ranges using LIMIT.
     */
    public String getRangeByLimitWhereClause(long offset, long count)
    {
        return "";
    }

    /**
     * Method to return the column name to use when handling ranges via
     * a rownumber on the select. Defaults to an empty string (not supported).
     * @return The row number column.
     */
    public String getRangeByRowNumberColumn()
    {
        return "";
    }

    /**
     * Accessor for table and column information for a catalog/schema in this datastore.
     * @param conn Connection to use
     * @param catalog The catalog (null if none)
     * @param schema The schema (null if none)
     * @param table The table (null if all)
     * @param columnNamePattern The column name (null if all)
     * @return ResultSet containing the table/column information
     * @throws SQLException Thrown if an error occurs
     */
    public ResultSet getColumns(Connection conn, String catalog, String schema, String table, String columnNamePattern)
    throws SQLException
    {
        DatabaseMetaData dmd = conn.getMetaData();
        return dmd.getColumns(catalog, schema, table, columnNamePattern);
    }

    public String toString()
    {
        StringBuffer sb = new StringBuffer();
        sb.append("================ DatabaseAdapter ==================");
        sb.append("\n");
        sb.append("Adapter : " + this.getClass().getName());
        sb.append("\n");
        sb.append("Datastore : name=\"" + datastoreProductName + "\" version=\"" + datastoreProductVersion + 
            "\" (major=" + datastoreMajorVersion + ", minor=" + datastoreMinorVersion + ", revision=" + datastoreRevisionVersion + ")");
        sb.append("\n");
        sb.append("Driver : name=\"" + driverName + "\" version=\"" + driverVersion + 
            "\" (major=" + driverMajorVersion + ", minor=" + driverMinorVersion + ")");
        sb.append("\n");
        sb.append("===================================================");
        return sb.toString();
    }

    /**
     * Accessor for a statement that will return the statement to use to get the datastore date.
     * @return SQL statement to get the datastore date
     */
    public String getDatastoreDateStatement()
    {
        return "SELECT CURRENT_TIMESTAMP";
    }

    // ------------------------------ Query Expression methods ----------------------------------
    // TODO Remove all of these when we ditch the original JDOQL architecture

    /**
     * Method to return an expression for the current date in the datastore.
     * @param qs QueryExpression
     * @return Current date expression for this datastore
     */
    public ScalarExpression getCurrentDateMethod(QueryExpression qs)
    {
        return new SqlTemporalExpression("CURRENT_DATE", qs);
    }

    /**
     * Method to return an expression for the current time in the datastore.
     * @param qs QueryExpression
     * @return Current time expression for this datastore
     */
    public ScalarExpression getCurrentTimeMethod(QueryExpression qs)
    {
        return new SqlTemporalExpression("CURRENT_TIME", qs);
    }

    /**
     * Method to return an expression for the current timestamp in the datastore.
     * @param qs QueryExpression
     * @return Current timestamp expression for this datastore
     */
    public ScalarExpression getCurrentTimestampMethod(QueryExpression qs)
    {
        return new SqlTemporalExpression("CURRENT_TIMESTAMP", qs);
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.DatastoreAdapter#modOperator(org.datanucleus.store.expression.ScalarExpression, org.datanucleus.store.expression.ScalarExpression)
     */
    public NumericExpression modOperator(ScalarExpression operand1, ScalarExpression operand2)
    {
        return new NumericExpression(operand1, ScalarExpression.OP_MOD, operand2);
    }

    /**
     * Convenience accessor for the mapping for the specified class.
     * Provides a wrapper to the method on the MappingManager.
     * @param c Java type
     * @param expr the ScalarExpression
     * @return The mapping for the class.
     */
    protected final JavaTypeMapping getMapping(Class c, ScalarExpression expr)
    {
        return expr.getQueryExpression().getStoreManager().getMappingManager().getMappingWithDatastoreMapping(c, false, false, 
            expr.getQueryExpression().getClassLoaderResolver());
    }

    /**
     * The pattern string for representing one character.
     * Most of databases will use the underscore character.
     * @param patternExpression The expression that represents one character for a matcher/parser in the database
     * @return the pattern string.
     **/
    public ScalarExpression getEscapedPatternExpression(ScalarExpression patternExpression)
    {
        if (patternExpression instanceof StringLiteral)
        {
            String value = (String) ((StringLiteral)patternExpression).getValue();
            value = value.replace("\\","\\\\");
            value = value.replace("%","\\%");
            value = value.replace("_","\\_");
            JavaTypeMapping m = getMapping(String.class, patternExpression);

            return m.newLiteral(patternExpression.getQueryExpression(),value);
        }
        else
        {
            return patternExpression;
        }
    }

    /**
     * The pattern string for representing one character that is expanded in word searches.
     * Most of databases will use the underscore character.
     * @return the pattern string.
     **/
    public String getPatternExpressionAnyCharacter()
    {
        return "_";
    }
    
    /**
     * The pattern string for representing zero or more characters that is expanded in word searches.
     * Most of databases will use the percent sign character.
     * @return the pattern string.
     **/
    public String getPatternExpressionZeroMoreCharacters()
    {
        return "%";
    }
    
    /**
     * The character for escaping characters in pattern expressions.
     * @return the character.
     **/
    public String getEscapePatternExpression()
    {
        return "ESCAPE '\\'";
    }
    
    /**
     * The character for escaping characters in pattern expressions.
     * @return the character.
     **/
    public String getEscapeCharacter()
    {
        return "\\";
    }
    
    /**
     * Continuation string to use where the SQL statement goes over more than 1
     * line. Some JDBC adapters (e.g DB2) don't do conversion.
     * @return Continuation string.
     **/
    public String getContinuationString()
    {
        return "\n";
    }

    /**
     * Generates a expression that represents the cartesian product of two sets: <code>X</code> and <code>Y</code>.
     * Actually, <code>X</code> is not generated to the expression.
     * @param Y right hand set
     * @return the cartesion product expression. if the cartesian product expression is "X x Y", the returned expression
     * is " x Y". Note that the left hand set was not introduced to the expression
     **/
    public String cartersianProduct(LogicSetExpression Y)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(" , ");
        sb.append(Y.toString());
        return sb.toString();
    }

    /**
     * Accessor for a new query statement.
     * @param table The table to query 
     * @param rangeVar A range variable for the query
     * @param clr The ClassLoaderResolver
     * @return The Query Statement
     **/
    public QueryExpression newQueryStatement(DatastoreContainerObject table, DatastoreIdentifier rangeVar, ClassLoaderResolver clr)
    {
        return new QueryStatement(table, rangeVar, clr);
    }

    /**
     * Returns a new TableExpression object appropriate for this DBMS.
     * This should be an instance of one of the three built-in styles of table
     * expression:
     * <ul>
     *   <li>TableExprAsJoins</li>
     *   <li>TableExprAsSubjoins</li>
     *   <li>TableExprAsSubquery</li>
     * </ul>
     * TableExprAsSubjoins is the default, which arguably produces the most
     * readable SQL but doesn't work on all DBMS's.  TableExprAsSubjoins
     * should work anywhere, but may be less efficient.
     *
     * @param qs The query statement in which the table expression will be included.
     * @param table  The main table in the expression.
     * @param rangeVar The SQL alias, or "range variable", to assign to the expression or to the main table.
     * @return The expression
     */
    public LogicSetExpression newTableExpression(QueryExpression qs, DatastoreContainerObject table, DatastoreIdentifier rangeVar)
    {
        return new TableExprAsJoins(qs, table, rangeVar);
    }

    /**
     * Accessor for the function to use for converting to numeric.
     * @return The numeric conversion function for this datastore.
     */
    public String getNumericConversionFunction()
    {
        return "ASCII";
    }

    /**
     * Returns the appropriate expression for the <code>(int)'A'</code> expression.
     * In SQL, it should compile something like:
     * <p>
     * <blockquote><pre>
     * ASCII('A')
     * </pre></blockquote>
     * </p>
     * @param expr The CharacterExpression
     * @return The NumericExpression
     */
    public NumericExpression toNumericExpression(CharacterExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            char c = ((Character)((CharacterLiteral)expr).getValue()).charValue();
            BigInteger value = new BigInteger(""+(int)c);
            return (NumericExpression) getMapping(value.getClass(), expr).newLiteral(expr.getQueryExpression(),value);
        }        
        else if (expr instanceof Literal)
        {
            BigInteger value = new BigInteger((String)((Literal)expr).getValue());
            return (NumericExpression) getMapping(value.getClass(), expr).newLiteral(expr.getQueryExpression(),value);
        }
        ArrayList args = new ArrayList();
        args.add(expr);

        return new NumericExpression("ASCII", args);
    }
    
    /**
     * Method to translate all chars in this expression to the <code>fromExpr</code> which 
     * corresponds to <code>toExpr</code>.
     * @return The expression.
     **/
    public StringExpression translateMethod(ScalarExpression expr, ScalarExpression toExpr, ScalarExpression fromExpr)
    {
        ArrayList args = new ArrayList();
        args.add(expr);
        args.add(fromExpr);
        args.add(toExpr);

        return new StringExpression("TRANSLATE", args);
    }

    /**
     * Accessor for a numeric expression to represent the method call, with passed argument.
     * @param method The method (case insensitive)
     * @param expr The argument to the method
     * @return The numeric expression that results
     */
    public NumericExpression getNumericExpressionForMethod(String method, ScalarExpression expr)
    {
        ArrayList args = new ArrayList();
        args.add(expr);

        if (method.equalsIgnoreCase("abs"))
        {
            return new NumericExpression("ABS", args);
        }
        else if (method.equalsIgnoreCase("sqrt"))
        {
            return new NumericExpression("SQRT", args);
        }
        else if (method.equalsIgnoreCase("cos"))
        {
            return new NumericExpression("COS", args);
        }
        else if (method.equalsIgnoreCase("sin"))
        {
            return new NumericExpression("SIN", args);
        }
        else if (method.equalsIgnoreCase("tan"))
        {
            return new NumericExpression("TAN", args);
        }
        else if (method.equalsIgnoreCase("acos"))
        {
            return new NumericExpression("ACOS", args);
        }
        else if (method.equalsIgnoreCase("asin"))
        {
            return new NumericExpression("ASIN", args);
        }
        else if (method.equalsIgnoreCase("atan"))
        {
            return new NumericExpression("ATAN", args);
        }
        else if (method.equalsIgnoreCase("exp"))
        {
            return new NumericExpression("EXP", args);
        }
        else if (method.equalsIgnoreCase("log"))
        {
            return new NumericExpression("LOG", args);
        }
        else if (method.equalsIgnoreCase("floor"))
        {
            return new NumericExpression("FLOOR", args);
        }
        else if (method.equalsIgnoreCase("ceil"))
        {
            return new NumericExpression("CEIL", args);
        }
        else if (method.equalsIgnoreCase("length"))
        {
            return new NumericExpression("CHAR_LENGTH", args);
        }
        else if (method.equalsIgnoreCase("year"))
        {
            return new NumericExpression("YEAR", args);
        }
        else if (method.equalsIgnoreCase("month"))
        {
            // Delete one from the SQL "month" (origin=1) to be compatible with Java month (origin=0)
            JavaTypeMapping m = getMapping(BigInteger.class, expr);
            ScalarExpression integerLiteral = m.newLiteral(expr.getQueryExpression(), BigInteger.ONE);
            NumericExpression numExpr = new NumericExpression(new NumericExpression("MONTH", args), 
                ScalarExpression.OP_SUB, integerLiteral);
            numExpr.encloseWithInParentheses();
            return numExpr;
        }
        else if (method.equalsIgnoreCase("day"))
        {
            return new NumericExpression("DAY", args);
        }
        else if (method.equalsIgnoreCase("hour"))
        {
            return new NumericExpression("HOUR", args);
        }
        else if (method.equalsIgnoreCase("minute"))
        {
            return new NumericExpression("MINUTE", args);
        }
        else if (method.equalsIgnoreCase("second"))
        {
            return new NumericExpression("SECOND", args);
        }
        else
        {
            throw new NucleusUserException("Attempt to get numeric expression for " + method + 
                " yet this is not supported");
        }
    }

    /**
     * Returns whether this string ends with the specified string.
     * @param leftOperand the source string
     * @param rightOperand The string to compare against.
     * @return Whether it ends with the string.
     **/
    public BooleanExpression endsWithMethod(ScalarExpression leftOperand, ScalarExpression rightOperand)
    {
        if (!(rightOperand instanceof StringExpression))
        {
            throw new IllegalArgumentTypeException(rightOperand);
        }

        StringLiteral pct = new StringLiteral(leftOperand.getQueryExpression(), leftOperand.getMapping(), '%');
        return new BooleanExpression(leftOperand, ScalarExpression.OP_LIKE,
            pct.add(getEscapedPatternExpression(rightOperand)));
    }

    /**
     * Matches this to the argument expression pattern. Use "." to find any
     * character and ".*" for wildcard matches. A global case-insensitive flag
     * "(?i)" can be set for the pattern. If used, the global case-insensitive
     * flag must prefix the pattern. The pattern passed to matches must be a
     * literal or parameter.
     * @param text   The argument to the length() method.
     * @param pattern The literal expression with the pattern.
     * @return the match expression.
     */
    public BooleanExpression matchesMethod(StringExpression text, StringExpression pattern)
    {
        return new BooleanExpression(text, ScalarExpression.OP_LIKE, pattern);
    }
    
    /**
     * A String conversion that converts a numeric expression to string.
     * If the expr argument represents a Literal value, converts to a Literal string.
     * In SQL, it should compile something like:
     * <p>
     * <blockquote>
     * <pre>
     * CAST(999999 AS VARCHAR(4000))
     * </pre>
     * </blockquote>
     * </p>
     * @param expr The NumericExpression
     * @return the StringExpression
     */
    public StringExpression toStringExpression(NumericExpression expr)
    {
        if( expr instanceof Literal )
        {
            JavaTypeMapping m = getMapping(String.class, expr);
            return new StringLiteral(expr.getQueryExpression(),m,((Literal)expr).getValue().toString());            
        }
        List args = new ArrayList();
        args.add(expr);        
        List types = new ArrayList();
        types.add("VARCHAR(4000)");        
        return new StringExpression("CAST", args, types);
    }
    
    /**
     * A String conversion that converts a String literal to String expression. It will allow
     * the String to only be evaluated at runtime.
     * In SQL, it should compile something like:
     * <p>
     * <blockquote>
     * <pre>
     * CAST(999999 AS VARCHAR(4000))
     * </pre>
     * </blockquote>
     * </p>
     * @param expr The NumericExpression
     * @return the StringExpression
     */
    public StringExpression toStringExpression(StringLiteral expr)
    {
        return expr;
    }

    /**
     * Returns the appropriate SQL expression for the JDOQL
     * String.substring(str,begin) method.
     * It should return something like:
     * <p>
     * <pre>
     * SUBSTRING(str FROM begin)
     * </pre>
     * Note that the value of <var>begin</var> is base 0(Java-style), while most
     * SQL string functions use base 1.
     *
     * @param str The first argument to the substring() method.
     * @param begin The second argument to the substring() method.
     * @return  The text of the SQL expression.
     */
    public StringExpression substringMethod(StringExpression str,
                                            NumericExpression begin)
    {
        return new SubstringExpression(str, begin);
    }

    public StringExpression lowerMethod(StringExpression str)
    {
        ArrayList args = new ArrayList();
        args.add(str);
        return new StringExpression("LOWER", args);
    }

    public StringExpression upperMethod(StringExpression str)
    {
        ArrayList args = new ArrayList();
        args.add(str);
        return new StringExpression("UPPER", args);
    }

    /**
     * Returns the appropriate SQL expression for a query "trim" method.
     * It should return something like:
     * <pre>TRIM(str)</pre>
     * @param str The first argument to the trim() method.
     * @param leading Whether to trim leading spaces
     * @param trailing Whether to trim trailing spaces
     * @return  The text of the SQL expression.
     */
    public StringExpression trimMethod(StringExpression str, boolean leading, boolean trailing)
    {
        ArrayList args = new ArrayList();
        args.add(str);

        if (leading && trailing)
        {
            return new StringExpression("TRIM", args);
        }
        else if (leading)
        {
            return new StringExpression("LTRIM", args);
        }
        else if (trailing)
        {
            return new StringExpression("RTRIM", args);
        }
        return str; // Nothing to trim
    }

    /**
     * Returns the appropriate SQL expression for the JDOQL
     * String.substring(str,begin,end) method.
     * It should return something like:
     * <p>
     * <blockquote><pre>
     * SUBSTRING(str FROM begin FOR len)
     * </pre></blockquote>
     * Note that the value of <var>begin</var> is base 0 (Java-style), while most
     * SQL string functions use base 1.
     * Note also that an end position is given, while most SQL substring
     * functions take a length.
     *
     * @param str   The first argument to the substring() method.
     * @param begin The second argument to the substring() method.
     * @param end   The third argument to the substring() method.
     * @return  The text of the SQL expression.
     */
    public StringExpression substringMethod(StringExpression str,
                                            NumericExpression begin,
                                            NumericExpression end)
    {
        return new SubstringExpression(str, begin, end);
    }

    /**
     * Method to handle the starts with operation.
     * Will return something like
     * <PRE>
     * source LIKE str%
     * </PRE>
     * @param source The expression with the searched string
     * @param str The expression for the search string 
     * @return The expression.
     **/
    public BooleanExpression startsWithMethod(ScalarExpression source, ScalarExpression str)
    {
        ScalarExpression pct = getMapping(String.class, source).newLiteral(source.getQueryExpression(), "%");

        return new BooleanExpression(source, ScalarExpression.OP_LIKE, 
            getEscapedPatternExpression(str).add(pct));
    }

    /**
     * Returns the appropriate SQL expression for the JDOQL String.indexOf() method.
     * It should return something like:
     * <p>
     * <blockquote><pre>
     * LOCATE(str, substr [,pos])-1
     * </pre></blockquote>
     * since LOCATE returns the first character as position 1. Similarly the "pos" is based on the first
     * position being 1.
     * </p>
     * @param source The expression we want to search.
     * @param str The argument to the indexOf() method.
     * @param from The from position
     * @return The text of the SQL expression.
     */
    public NumericExpression indexOfMethod(ScalarExpression source, ScalarExpression str, NumericExpression from)
    {
        JavaTypeMapping m = getMapping(BigInteger.class, source);
        ScalarExpression integerLiteral = m.newLiteral(source.getQueryExpression(), BigInteger.ONE);

        ArrayList args = new ArrayList();
        args.add(str);
        args.add(source);
        if (from != null)
        {
            // Add 1 to the passed in value so that it is of origin 1 to be compatible with LOCATE
            args.add(from.add(integerLiteral));
        }
        NumericExpression locateExpr = new NumericExpression("LOCATE", args);

        // Subtract 1 from the result of LOCATE to be consistent with Java strings
        // TODO Would be nice to put this in parentheses
        return new NumericExpression(locateExpr, ScalarExpression.OP_SUB, integerLiteral);
    }

    /**
     * An operator in a string expression that concatenates two or more
     * character or binary strings, columns, or a combination of strings and
     * column names into one expression (a string operator).
     * 
     * @return the operator SQL String
     */
    public String getOperatorConcat()
    {
        return "||";
    }

    /**
     * <p>
     * If only one operand expression is of type String, then string conversion
     * is performed on the other operand to produce a string at run time. The
     * result is a reference to a String object (newly created, unless the
     * expression is a compile-time constant expression (�15.28))that is the
     * concatenation of the two operand strings. The characters of the left-hand
     * operand precede the characters of the right-hand operand in the newly
     * created string. If an operand of type String is null, then the string
     * "null" is used instead of that operand. "null" is used instead of that
     * operand.
     * </p>
     * <p>
     * Concatenates two or more character or binary strings, columns, or a
     * combination of strings and column names into one expression (a string
     * operator).
     * </p>
     * @param operand1 the left expression
     * @param operand2 the right expression
     * @return The Expression for concatenation
     */
    public ScalarExpression concatOperator(ScalarExpression operand1, ScalarExpression operand2)
    {
        return new ConcatOperatorExpression(operand1, operand2);
    }
}