/**********************************************************************
Copyright (c) 2003 Mike Martin (TJDO) and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2003 Kelly Grizzle (TJDO)
2003 Erik Bengtson  - removed exist() operation
2003 Erik Bengtson  - refactored the persistent id generator System property
2003 Andy Jefferson - added localiser
2003 Andy Jefferson - updated exception handling with SchemaTable
2003 Andy Jefferson - restructured to remove SchemaTable, and add StoreManagerHelper
2003 Andy Jefferson - updated getSubClassesForClass to recurse
2004 Erik Bengtson  - removed unused method and variables 
2004 Erik Bengtson  - fixed problem with getObjectById for App ID in getClassForOID
2004 Andy Jefferson - re-emergence of SchemaTable. Addition of addClass().
2004 Andy Jefferson - Addition of AutoStartMechanism interface
2004 Andy Jefferson - Update to use Logger
2004 Andy Jefferson - Addition of Catalog name to accompany Schema name
2004 Marco Schulze  - replaced catch(NotPersistenceCapableException ...)
                  by advance-check via TypeManager.isSupportedType(...)
2004 Andy Jefferson - split StoreData into superclass.
2004 Andy Jefferson - added support for other inheritance types
2004 Andy Jefferson - added capability to dynamically add columns
2005 Marco Schulze - prevent closing starter during recursion of ClassAdder.addClassTables(...)
    ...
**********************************************************************/
package org.datanucleus.store.rdbms;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.FetchPlan;
import org.datanucleus.OMFContext;
import org.datanucleus.ObjectManager;
import org.datanucleus.PersistenceConfiguration;
import org.datanucleus.StateManager;
import org.datanucleus.UserTransaction;
import org.datanucleus.api.ApiAdapter;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.identity.OID;
import org.datanucleus.identity.SCOID;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ClassMetaData;
import org.datanucleus.metadata.ClassPersistenceModifier;
import org.datanucleus.metadata.ExtensionMetaData;
import org.datanucleus.metadata.IdentityMetaData;
import org.datanucleus.metadata.IdentityStrategy;
import org.datanucleus.metadata.IdentityType;
import org.datanucleus.metadata.InheritanceMetaData;
import org.datanucleus.metadata.InheritanceStrategy;
import org.datanucleus.metadata.MapMetaData;
import org.datanucleus.metadata.SequenceMetaData;
import org.datanucleus.metadata.TableGeneratorMetaData;
import org.datanucleus.store.Extent;
import org.datanucleus.store.NucleusConnection;
import org.datanucleus.store.NucleusSequence;
import org.datanucleus.store.StoreData;
import org.datanucleus.store.connection.ConnectionFactory;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.exceptions.NoExtentException;
import org.datanucleus.store.exceptions.UnsupportedDataTypeException;
import org.datanucleus.store.fieldmanager.FieldManager;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.IdentifierFactory;
import org.datanucleus.store.mapped.MappedStoreManager;
import org.datanucleus.store.mapped.StatementClassMapping;
import org.datanucleus.store.mapped.mapping.DatastoreMapping;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.mapped.scostore.FKArrayStore;
import org.datanucleus.store.mapped.scostore.FKListStore;
import org.datanucleus.store.mapped.scostore.FKMapStore;
import org.datanucleus.store.mapped.scostore.FKSetStore;
import org.datanucleus.store.mapped.scostore.JoinArrayStore;
import org.datanucleus.store.mapped.scostore.JoinListStore;
import org.datanucleus.store.mapped.scostore.JoinMapStore;
import org.datanucleus.store.mapped.scostore.JoinSetStore;
import org.datanucleus.store.query.ResultObjectFactory;
import org.datanucleus.store.rdbms.adapter.RDBMSAdapter;
import org.datanucleus.store.rdbms.adapter.RDBMSAdapterFactory;
import org.datanucleus.store.rdbms.fieldmanager.ParameterSetter;
import org.datanucleus.store.rdbms.fieldmanager.ResultSetGetter;
import org.datanucleus.store.rdbms.mapping.RDBMSMapping;
import org.datanucleus.store.rdbms.query.PersistentClassROF;
import org.datanucleus.store.rdbms.query.legacy.ExtentHelper;
import org.datanucleus.store.rdbms.schema.JDBCTypeInfo;
import org.datanucleus.store.rdbms.schema.RDBMSColumnInfo;
import org.datanucleus.store.rdbms.schema.RDBMSSchemaHandler;
import org.datanucleus.store.rdbms.schema.RDBMSSchemaInfo;
import org.datanucleus.store.rdbms.schema.RDBMSTableInfo;
import org.datanucleus.store.rdbms.schema.RDBMSTypesInfo;
import org.datanucleus.store.rdbms.schema.SQLTypeInfo;
import org.datanucleus.store.rdbms.scostore.RDBMSFKArrayStore;
import org.datanucleus.store.rdbms.scostore.RDBMSFKListStore;
import org.datanucleus.store.rdbms.scostore.RDBMSFKMapStore;
import org.datanucleus.store.rdbms.scostore.RDBMSFKSetStore;
import org.datanucleus.store.rdbms.scostore.RDBMSJoinArrayStore;
import org.datanucleus.store.rdbms.scostore.RDBMSJoinListStore;
import org.datanucleus.store.rdbms.scostore.RDBMSJoinMapStore;
import org.datanucleus.store.rdbms.scostore.RDBMSJoinSetStore;
import org.datanucleus.store.rdbms.sql.expression.SQLExpressionFactory;
import org.datanucleus.store.rdbms.table.ArrayTable;
import org.datanucleus.store.rdbms.table.ClassTable;
import org.datanucleus.store.rdbms.table.ClassView;
import org.datanucleus.store.rdbms.table.CollectionTable;
import org.datanucleus.store.rdbms.table.Column;
import org.datanucleus.store.rdbms.table.JoinTable;
import org.datanucleus.store.rdbms.table.MapTable;
import org.datanucleus.store.rdbms.table.ProbeTable;
import org.datanucleus.store.rdbms.table.Table;
import org.datanucleus.store.rdbms.table.TableImpl;
import org.datanucleus.store.rdbms.table.ViewImpl;
import org.datanucleus.store.valuegenerator.AbstractDatastoreGenerator;
import org.datanucleus.store.valuegenerator.ValueGenerationConnectionProvider;
import org.datanucleus.store.valuegenerator.ValueGenerator;
import org.datanucleus.transaction.TransactionUtils;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.MacroString;
import org.datanucleus.util.MultiMap;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

import java.io.IOException;
import java.io.PrintStream;
import java.io.Writer;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeSet;

/**
 * StoreManager for RDBMS datastores. 
 * Provided by the "store-manager" extension key "rdbms" and accepts datastore URLs valid for JDBC.
 * <p>
 * The RDBMS manager's responsibilities extend those for StoreManager to add :
 * <ul>
 * <li>creates and controls access to the data sources required for this datastore instance</li>
 * <li>implements insert(), fetch(), update(), delete() in the interface to the StateManager.</li>
 * <li>Providing cached access to JDBC database metadata (in particular column information).</li>
 * <li>Resolving SQL identifier macros to actual SQL identifiers.</li>
 * </ul>
 * TODO Change RDBMSManager to share schema information (DatabaseMetaData) with other RDBMSManager.
 */
public class RDBMSStoreManager extends MappedStoreManager
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER_RDBMS = Localiser.getInstance(
        "org.datanucleus.store.rdbms.Localisation", RDBMSStoreManager.class.getClassLoader());

    /** Catalog name for the RDBMS database */
    private String catalogName = null;

    /** Schema name for the RDBMS database */
    private String schemaName = null;

    /** The connectionProvider for this RDBMS **/
    private ConnectionProvider connProvider;

    /** Controller for SQL executed on this store. */
    private SQLController sqlController = null;

    /** Factory for expressions using the generic query SQL mechanism. */
    protected SQLExpressionFactory expressionFactory;

    /**
     * The active class adder transaction, if any. Some RDBMSManager methods are
     * called recursively in the course of adding new classes. This field allows
     * such methods to coordinate with the active ClassAdder transaction.
     * Recursive methods include:
     * <ul>
     * <li>addClasses()</li>
     * <li>addSetTable()</li>
     * <li>addMapTable()</li>
     * <li>addListTable()</li>
     * <li>addArrayTable()</li>
     * </ul>
     * Access is synchronized on the RDBMSManager itself. 
     * Invariant: classAdder == null if RDBMSManager is unlocked.
     */
    private ClassAdder classAdder = null;

    /** 
     * Object to use for locking the ClassAdder process. This could be revised with CORE-3409
     * and the above ClassAdder field to find a better locking strategy.
     * TODO This locks all RDBMSManager ClassAdder but should really only lock one
     */
    private final Object CLASSADDER_MUTEX = new Object();

    /** Writer for use when this RDBMSManager is configured to write DDL. */
    private Writer ddlWriter = null;

    /** 
     * Flag for use when this RDBMSManager is configured to write DDL, whether we should generate 
     * complete DDL or upgrade DDL 
     */
    private boolean completeDDL = false;

    /** 
     * DDL statements already written when in DDL mode. 
     * This is used to eliminate duplicate statements from bidirectional relations.
     */
    private Set<String> writtenDdlStatements = null;

    /** State variable for schema generation of the callback information to be processed. */
    private MultiMap schemaCallbacks = new MultiMap();

    /**
     * Constructs a new RDBMSManager. On successful return the new RDBMSManager
     * will have successfully connected to the database with the given
     * credentials and determined the schema name, but will not have inspected
     * the schema contents any further. The contents (tables, views, etc.) will
     * be subsequently created and/or validated on-demand as the application
     * accesses persistent classes.
     * 
     * @param clr the ClassLoaderResolver
     * @param omfContext The corresponding PersistenceManagerFactory. This factory's non-tx data source will be 
     *     used to get database connections as needed to perform management functions.
     * @exception NucleusDataStoreException If the database could not be accessed or the name of the
     *                schema could not be determined.
     */
    public RDBMSStoreManager(ClassLoaderResolver clr, OMFContext omfContext)
    {
        super("rdbms", clr, omfContext);

        PersistenceConfiguration conf = omfContext.getPersistenceConfiguration();
        try
        {
            connProvider = (ConnectionProvider) omfContext.getPluginManager().createExecutableExtension(
                    "org.datanucleus.store.rdbms.connectionprovider", "name", 
                    conf.getStringProperty("datanucleus.rdbms.connectionProviderName"), 
                    "class-name", null, null);
            if (connProvider == null)
            {
                // No provider with this name (missing plugin ?)
                throw new NucleusException(LOCALISER_RDBMS.msg("050000",
                    conf.getStringProperty("datanucleus.rdbms.connectionProviderName"))).setFatal();
            }
            connProvider.setFailOnError(conf.getBooleanProperty("datanucleus.rdbms.connectionProviderFailOnError"));
        }
        catch (Exception e)
        {
            // Error creating provider
            throw new NucleusException(LOCALISER.msg("050001",
                conf.getStringProperty("datanucleus.rdbms.connectionProviderName"), e.getMessage()), e).setFatal();
        }

        // Handler for persistence operations
        persistenceHandler = new RDBMSPersistenceHandler(this);

        // Handler for schema operations
        schemaHandler = new RDBMSSchemaHandler(this);

        // Factory for SQL expressions
        expressionFactory = new SQLExpressionFactory(this);

        // Retrieve the Database Adapter for this datastore
        try
        {
            ManagedConnection mc = getConnection(-1);
            Connection conn = (Connection)mc.getConnection();
            if (conn == null)
            {
                //somehow we haven't got an exception from the JDBC driver
                //to troubleshoot the user should telnet to ip/port of database and check if he can open a connection
                //this may be due to security / firewall things.
                throw new NucleusDataStoreException(LOCALISER_RDBMS.msg("050007"));
            }

            try
            {
                dba = RDBMSAdapterFactory.getInstance().getDatastoreAdapter(clr, conn, 
                    conf.getStringProperty("datanucleus.rdbms.datastoreAdapterClassName"), 
                    omfContext.getPluginManager());
                dba.initialiseTypes(schemaHandler, mc);
                dba.removeUnsupportedMappings(schemaHandler, mc);

                // User specified default catalog/schema name - check for validity, and store
                if (conf.hasProperty("datanucleus.mapping.Catalog"))
                {
                    if (!((RDBMSAdapter)dba).supportsOption(RDBMSAdapter.CATALOGS_IN_TABLE_DEFINITIONS))
                    {
                        NucleusLogger.DATASTORE.warn(LOCALISER_RDBMS.msg("050002",
                            conf.getStringProperty("datanucleus.mapping.Catalog")));
                    }
                    else
                    {
                        catalogName = conf.getStringProperty("datanucleus.mapping.Catalog");
                    }
                }
                if (conf.hasProperty("datanucleus.mapping.Schema"))
                {
                    if (!((RDBMSAdapter)dba).supportsOption(DatastoreAdapter.SCHEMAS_IN_TABLE_DEFINITIONS))
                    {
                        NucleusLogger.DATASTORE.warn(LOCALISER_RDBMS.msg("050002",
                            conf.getStringProperty("datanucleus.mapping.Schema")));
                    }
                    else
                    {
                        schemaName = conf.getStringProperty("datanucleus.mapping.Schema");
                    }
                }

                // Create an identifier factory - needs the database adapter to exist first
                initialiseIdentifierFactory(omfContext);

                // Now that we have the identifier factory, make sure any user-provided names were valid!
                if (schemaName != null)
                {
                    String validSchemaName = identifierFactory.getIdentifierInAdapterCase(schemaName);
                    if (!validSchemaName.equals(schemaName))
                    {
                        NucleusLogger.DATASTORE_SCHEMA.warn(LOCALISER_RDBMS.msg("020192", "schema", schemaName, validSchemaName));
                        schemaName = validSchemaName;
                    }
                }
                if (catalogName != null)
                {
                    String validCatalogName = identifierFactory.getIdentifierInAdapterCase(catalogName);
                    if (!validCatalogName.equals(catalogName))
                    {
                        NucleusLogger.DATASTORE_SCHEMA.warn(LOCALISER_RDBMS.msg("020192", "catalog", catalogName, validCatalogName));
                        catalogName = validCatalogName;
                    }
                }

                // Create the SQL controller
                sqlController = new SQLController(((RDBMSAdapter)dba).supportsOption(RDBMSAdapter.STATEMENT_BATCHING), 
                    conf.getIntProperty("datanucleus.rdbms.statementBatchLimit"),
                    conf.getIntProperty("datanucleus.datastoreReadTimeout"),
                    conf.getBooleanProperty("datanucleus.rdbms.sqlParamValuesInBrackets"));

                // Initialise any properties controlling the adapter
                // Just use properties matching the pattern "datanucleus.rdbms.adapter.*"
                Map<String, Object> props = new HashMap();
                Map<String, Object> omfProps = omfContext.getPersistenceConfiguration().getPersistenceProperties();
                Iterator<String> propIter = omfProps.keySet().iterator();
                while (propIter.hasNext())
                {
                    String prop = propIter.next();
                    if (prop.startsWith("datanucleus.rdbms.adapter."))
                    {
                        props.put(prop, omfProps.get(prop));
                    }
                }
                if (props.size() > 0)
                {
                    dba.setProperties(props);
                }

                // Initialise the Schema
                initialiseSchema(conn, clr);
            }
            finally
            {
                mc.close();
            }
        }
        catch (NucleusException jpex)
        {
            NucleusLogger.DATASTORE_SCHEMA.error(LOCALISER_RDBMS.msg("050004"), jpex);
            throw jpex.setFatal();
        }
        catch (Exception e1)
        {
            // Unknown type of exception so wrap it in a NucleusUserException for later handling
            String msg = LOCALISER_RDBMS.msg("050004") + ' ' + 
                LOCALISER_RDBMS.msg("050006") + ' ' + LOCALISER_RDBMS.msg("048000",e1);
            NucleusLogger.DATASTORE_SCHEMA.error(msg, e1);
            throw new NucleusUserException(msg, e1).setFatal();
        }
        finally
        {
            logConfiguration();
        }
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.StoreManager#getQueryCacheKey()
     */
    public String getQueryCacheKey()
    {
        // Return "rdbms-hsqldb", etc
        return getStoreManagerKey() + "-" + getDatastoreAdapter().getVendorID();
    }

    public MultiMap getSchemaCallbacks()
    {
        return schemaCallbacks;
    }

    /**
     * Convenience method to log the configuration of this store manager.
     */
    protected void logConfiguration()
    {
        if (NucleusLogger.DATASTORE.isDebugEnabled())
        {
            PersistenceConfiguration conf = getOMFContext().getPersistenceConfiguration();

            String classNames = conf.getStringProperty("datanucleus.autoStartClassNames");
            NucleusLogger.DATASTORE.debug("======================= Datastore =========================");
            NucleusLogger.DATASTORE.debug("StoreManager : \"" + storeManagerKey + "\" (" + getClass().getName() + ")");
            NucleusLogger.DATASTORE.debug("AutoStart : mechanism=" + autoStartMechanism + 
                ", mode=" + conf.getStringProperty("datanucleus.autoStartMechanismMode") +
                ((classNames != null) ? (", classes=" + classNames) : ""));

            // Connection Pooling
            ConnectionFactoryImpl connFactory = (ConnectionFactoryImpl)connectionMgr.lookupConnectionFactory(nontxConnectionFactoryName);
            String poolingType = connFactory.getPoolingType();
            NucleusLogger.DATASTORE.debug("Connection Pooling : " + (poolingType.equals("default") ? "None" : poolingType));

            if (identifierFactory != null)
            {
                NucleusLogger.DATASTORE.debug("Datastore Identifiers :" +
                    " factory=\"" + conf.getStringProperty("datanucleus.identifierFactory") + "\"" +
                    " case=" + identifierFactory.getIdentifierCase().toString() +
                    (catalogName != null ? (" catalog=" + catalogName) : "") +
                    (schemaName != null ? (" schema=" + schemaName) : ""));
            }
            NucleusLogger.DATASTORE.debug("Datastore : " + (readOnlyDatastore ? "read-only" : "read-write") +
                (fixedDatastore ? ", fixed" : "") +
                (conf.getBooleanProperty("datanucleus.rdbms.useUpdateLock") ? ", useUpdateLock" : "") +
                (conf.getBooleanProperty("datanucleus.rdbms.checkExistTablesOrViews") ? ", checkTableViewExistence" : "") +
                ", rdbmsConstraintCreateMode=" + conf.getStringProperty("datanucleus.rdbms.constraintCreateMode") +
                ", initialiseColumnInfo=" + conf.getStringProperty("datanucleus.rdbms.initializeColumnInfo"));

            // Auto-Create
            StringBuffer autoCreateOptions = null;
            if (autoCreateTables || autoCreateColumns || autoCreateConstraints)
            {
                autoCreateOptions = new StringBuffer();
                boolean first = true;
                if (autoCreateTables)
                {
                    if (!first)
                    {
                        autoCreateOptions.append(",");
                    }
                    autoCreateOptions.append("Tables");
                    first = false;
                }
                if (autoCreateColumns)
                {
                    if (!first)
                    {
                        autoCreateOptions.append(",");
                    }
                    autoCreateOptions.append("Columns");
                    first = false;
                }
                if (autoCreateConstraints)
                {
                    if (!first)
                    {
                        autoCreateOptions.append(",");
                    }
                    autoCreateOptions.append("Constraints");
                    first = false;
                }
            }

            // Validate
            StringBuffer validateOptions = null;
            if (validateTables || validateColumns || validateConstraints)
            {
                validateOptions = new StringBuffer();
                boolean first = true;
                if (validateTables)
                {
                    validateOptions.append("Tables");
                    first = false;
                }
                if (validateColumns)
                {
                    if (!first)
                    {
                        validateOptions.append(",");
                    }
                    validateOptions.append("Columns");
                    first = false;
                }
                if (validateConstraints)
                {
                    if (!first)
                    {
                        validateOptions.append(",");
                    }
                    validateOptions.append("Constraints");
                    first = false;
                }
            }

            NucleusLogger.DATASTORE.debug("Schema Control : " +
                "AutoCreate(" + (autoCreateOptions != null ? autoCreateOptions.toString() : "None") + ")" +
                ", Validate(" + (validateOptions != null ? validateOptions.toString() : "None") + ")");

            int batchLimit = conf.getIntProperty("datanucleus.rdbms.statementBatchLimit");
            NucleusLogger.DATASTORE.debug("Statement Batching : max-batch-size=" +
                (batchLimit == -1 ? "UNLIMITED" : "" + batchLimit));
            String[] queryLanguages = getOMFContext().getPluginManager().getAttributeValuesForExtension("org.datanucleus.store_query_query", 
                "datastore", storeManagerKey, "name");
            NucleusLogger.DATASTORE.debug("Query Languages : " + StringUtils.objectArrayToString(queryLanguages));
            NucleusLogger.DATASTORE.debug("Queries : Timeout=" +
                conf.getIntProperty("datanucleus.datastoreReadTimeout"));
            NucleusLogger.DATASTORE.debug("Queries : Results " +
                "direction=" + conf.getStringProperty("datanucleus.rdbms.query.fetchDirection") + 
                ", type=" + conf.getStringProperty("datanucleus.rdbms.query.resultSetType") +
                ", concurrency=" + conf.getStringProperty("datanucleus.rdbms.query.resultSetConcurrency"));

            // JDBC Types
            NucleusLogger.DATASTORE.debug("Java-Types : string-default-length=" + conf.getIntProperty("datanucleus.rdbms.stringDefaultLength"));
            RDBMSTypesInfo typesInfo = (RDBMSTypesInfo)schemaHandler.getSchemaData(null, "types", null);
            if (typesInfo != null && typesInfo.getNumberOfChildren() > 0)
            {
                StringBuffer typeStr = new StringBuffer();
                Iterator jdbcTypesIter = typesInfo.getChildren().keySet().iterator();
                while (jdbcTypesIter.hasNext())
                {
                    String jdbcTypeStr = (String)jdbcTypesIter.next();
                    int jdbcTypeNumber = 0;
                    try
                    {
                        jdbcTypeNumber = Short.valueOf(jdbcTypeStr).shortValue();
                    }
                    catch (NumberFormatException nfe) { };

                    typeStr.append(JDBCUtils.getNameForJDBCType(jdbcTypeNumber));
                    if (jdbcTypesIter.hasNext())
                    {
                        typeStr.append(", ");
                    }
                }
                NucleusLogger.DATASTORE.debug("JDBC-Types : " + typeStr);
            }

            // Log the datastore adapter configuration
            if (dba != null)
            {
                ((RDBMSAdapter)dba).logConfiguration();
            }

            NucleusLogger.DATASTORE.debug("===========================================================");
        }
    }

    /**
     * Method to create the IdentifierFactory to be used by this store.
     * Relies on the datastore adapter existing before creation
     * @param omfContext ObjectManagerFactory context
     */
    protected void initialiseIdentifierFactory(OMFContext omfContext)
    {
        PersistenceConfiguration conf = omfContext.getPersistenceConfiguration();
        String idFactoryName = conf.getStringProperty("datanucleus.identifierFactory");
        String idFactoryClassName = omfContext.getPluginManager().getAttributeValueForExtension("org.datanucleus.store_identifierfactory", 
            "name", idFactoryName, "class-name");
        if (idFactoryClassName == null)
        {
            throw new NucleusUserException(LOCALISER_RDBMS.msg("039003", idFactoryName)).setFatal();
        }

        try
        {
            // Create the control properties for identifier generation
            Map props = new HashMap();
            if (catalogName != null)
            {
                props.put("DefaultCatalog", catalogName);
            }
            if (schemaName != null)
            {
                props.put("DefaultSchema", schemaName);
            }
            String val = conf.getStringProperty("datanucleus.identifier.case");
            if (val != null)
            {
                props.put("RequiredCase", val);
            }
            val = conf.getStringProperty("datanucleus.identifier.wordSeparator");
            if (val != null)
            {
                props.put("WordSeparator", val);
            }
            val = conf.getStringProperty("datanucleus.identifier.tablePrefix");
            if (val != null)
            {
                props.put("TablePrefix", val);
            }
            val = conf.getStringProperty("datanucleus.identifier.tableSuffix");
            if (val != null)
            {
                props.put("TableSuffix", val);
            }

            // Create the IdentifierFactory
            Class[] argTypes = new Class[] 
                {DatastoreAdapter.class, ClassLoaderResolver.class, Map.class};
            Object[] args = new Object[]
                {
                    dba, omfContext.getClassLoaderResolver(null), props
                };
            identifierFactory = (IdentifierFactory)omfContext.getPluginManager().createExecutableExtension(
                "org.datanucleus.store_identifierfactory", "name", idFactoryName, "class-name",
                argTypes, args);
        }
        catch (ClassNotFoundException cnfe)
        {
            throw new NucleusUserException(LOCALISER_RDBMS.msg("039004", idFactoryName, idFactoryClassName), cnfe).setFatal();
        }
        catch (Exception e)
        {
            NucleusLogger.PERSISTENCE.error(e);
            throw new NucleusException(LOCALISER_RDBMS.msg("039005", idFactoryClassName), e).setFatal();
        }
    }

    /**
     * Release of resources
     */
    public void close()
    {
        super.close();
        classAdder = null;
    }

    /**
     * Method to return a datastore sequence for this datastore matching the passed sequence MetaData.
     * @param om The ObjectManager
     * @param seqmd SequenceMetaData
     * @return The Sequence
     */
    public NucleusSequence getNucleusSequence(ObjectManager om, SequenceMetaData seqmd)
    {
        return new NucleusSequenceImpl(om, this, seqmd);
    }

    /**
     * Method to return a NucleusConnection for the ObjectManager.
     * @param om ObjectManager
     * @return The NucleusConnection
     */
    public NucleusConnection getNucleusConnection(final ObjectManager om)
    {
        final ManagedConnection mc;
        final boolean enlisted;
        if (!om.getTransaction().isActive())
        {
            // no active transaction so dont enlist
            enlisted = false;
        }
        else
        {
            enlisted = true;
        }
        ConnectionFactory cf = null;
        if (enlisted)
        {
            cf = connectionMgr.lookupConnectionFactory(txConnectionFactoryName);
        }
        else
        {
            cf = connectionMgr.lookupConnectionFactory(nontxConnectionFactoryName);
        }
        mc = cf.getConnection(enlisted?om:null, om.getTransaction(), null); // Will throw exception if already locked

        // Lock the connection now that it is in use by the user
        mc.lock();

        return new JDOConnectionImpl(mc.getConnection(), new Runnable()
        {
            public void run()
            {
                // Unlock the connection now that the user has finished with it
                mc.unlock();
                if (!enlisted)
                {
                    // Close the (unenlisted) connection (committing its statements)
                    try
                    {
                        ((Connection)mc.getConnection()).close();
                    }
                    catch (SQLException sqle)
                    {
                        throw new NucleusDataStoreException(sqle.getMessage());
                    }
                }
            }
        });
    }

    /**
     * Accessor for the SQL controller.
     * @return The SQL controller
     */
    public SQLController getSQLController()
    {
        return sqlController;
    }

    /**
     * Accessor for the SQL expression factory to use when generating SQL statements.
     * @return SQL expression factory
     */
    public SQLExpressionFactory getSQLExpressionFactory()
    {
        return expressionFactory;
    }

    /**
     * Initialises the schema name for the datastore, and (optionally) the schema table 
     * (and associated initial schema data).
     * @param conn A connection to the database
     * @param clr ClassLoader resolver
     */
    private void initialiseSchema(Connection conn, ClassLoaderResolver clr)
    throws Exception
    {
        // Initialise the Catalog/Schema names
        RDBMSAdapter rdba = (RDBMSAdapter)dba;
        if (schemaName == null && catalogName == null)
        {
            // User didn't provide catalog/schema so determine the defaults from the datastore
            // TODO Should we bother with this if the RDBMS doesn't support catalog/schema in the table identifiers ?
            try
            {
                try
                {
                    catalogName = rdba.getCatalogName(conn);
                    schemaName = rdba.getSchemaName(conn);
                }
                catch (UnsupportedOperationException e)
                {
                    if (!readOnlyDatastore && !fixedDatastore)
                    {
                        // If we aren't a read-only datastore, try to create a table and then 
                        // retrieve its details, so as to obtain the catalog, schema. 
                        ProbeTable pt = new ProbeTable(this);
                        pt.initialize(clr);
                        pt.create(conn);
                        try
                        {
                            String[] schema_details = pt.findSchemaDetails(conn);
                            if (schema_details != null)
                            {
                                catalogName = schema_details[0];
                                schemaName = schema_details[1];
                            }
                        }
                        finally
                        {
                            pt.drop(conn);
                        }
                    }
                }
            }
            catch (SQLException e)
            {
                String msg = LOCALISER_RDBMS.msg("050005", e.getMessage()) + ' ' + 
                    LOCALISER_RDBMS.msg("050006");
                NucleusLogger.DATASTORE_SCHEMA.warn(msg);
                // This is only logged as a warning since if the JDBC driver has some issue creating the ProbeTable we would be stuck
                // We need to allow SchemaTool "dbinfo" mode to work in all circumstances.
            }
        }
        // TODO If catalogName/schemaName are set convert them to the adapter case

        if (!readOnlyDatastore)
        {
            // Provide any add-ons for the datastore that may be needed later
            dba.initialiseDatastore(conn);
        }

        // AutoStarter - Load up any startup class names
        if ((readOnlyDatastore || fixedDatastore) && autoStartMechanism != null && autoStartMechanism.equals("SchemaTable"))
        {
            // Schema fixed and user requires an auto-starter needing schema content so turn it off
            autoStartMechanism = "None";
        }
        if (NucleusLogger.DATASTORE_SCHEMA.isInfoEnabled())
        {
            NucleusLogger.DATASTORE_SCHEMA.info(LOCALISER_RDBMS.msg("050008", catalogName, schemaName, autoStartMechanism));
        }
        initialiseAutoStart(clr);

        if (NucleusLogger.DATASTORE_SCHEMA.isInfoEnabled())
        {
            if (readOnlyDatastore)
            {
                NucleusLogger.DATASTORE_SCHEMA.info(LOCALISER_RDBMS.msg("050010",
                    catalogName, schemaName, "" + storeDataMgr.size()));
            }
            else if (fixedDatastore)
            {
                NucleusLogger.DATASTORE_SCHEMA.info(LOCALISER_RDBMS.msg("050011",
                        catalogName, schemaName, "" + storeDataMgr.size()));
            }
            else
            {
                NucleusLogger.DATASTORE_SCHEMA.info(LOCALISER_RDBMS.msg("050009",
                        catalogName, schemaName, "" + storeDataMgr.size()));
            }
        }
    }

    /**
     * Clears all knowledge of tables, cached requests, metadata, etc and resets
     * the store manager to its initial state.
     */
    private void clearSchemaData()
    {
        deregisterAllStoreData();

        schemaHandler.clear();
        ((RDBMSPersistenceHandler)persistenceHandler).removeAllRequests();
    }

    /**
     * Accessor for the (default) RDBMS catalog name.
     * 
     * @return The catalog name.
     */
    public String getCatalogName()
    {
        return catalogName;
    }

    /**
     * Accessor for the (default) RDBMS schema name.
     * 
     * @return The schema name.
     */
    public String getSchemaName()
    {
        return schemaName;
    }

    /**
     * Accessor to the ConnectionProvider
     * @return the ConnectionProvider
     */
    public ConnectionProvider getConnectionProvider()
    {
        return connProvider;
    }

    /**
     * Get the date/time of the datastore.
     * @return Date/time of the datastore
     */
    public Date getDatastoreDate()
    {
        Date serverDate = null;

        String dateStmt = ((RDBMSAdapter)dba).getDatastoreDateStatement();
        ManagedConnection mconn = null;
        try
        {
            mconn = getConnection(UserTransaction.TRANSACTION_NONE);

            PreparedStatement ps = null;
            ResultSet rs = null;
            try
            {
                ps = getSQLController().getStatementForQuery(mconn, dateStmt);
                rs = getSQLController().executeStatementQuery(mconn, dateStmt, ps);
                if (rs.next())
                {
                    // Retrieve the timestamp for the server date/time using the server TimeZone from OMF
                    // Assume that the dateStmt returns 1 column and is Timestamp
                    Timestamp time = rs.getTimestamp(1,
                        getOMFContext().getPersistenceConfiguration().getCalendarForDateTimezone());
                    serverDate = new Date(time.getTime());
                }
                else
                {
                    return null;
                }
            }
            catch (SQLException sqle)
            {
                String msg = LOCALISER_RDBMS.msg("050052", sqle.getMessage());
                NucleusLogger.DATASTORE.warn(msg, sqle);
                throw new NucleusUserException(msg, sqle).setFatal();
            }
            finally
            {
                if (rs != null)
                {
                    rs.close();
                }
                if (ps != null)
                {
                    getSQLController().closeStatement(mconn, ps);
                }
            }
        }
        catch (SQLException sqle)
        {
            String msg = LOCALISER_RDBMS.msg("050052", sqle.getMessage());
            NucleusLogger.DATASTORE.warn(msg, sqle);
            throw new NucleusException(msg, sqle).setFatal();
        }
        finally
        {
            mconn.close();
        }

        return serverDate;
    }

    // ----------------------------- Class Management -------------------------------

    /**
     * Method to add several (PersistenceCapable) classes to the store manager's set of supported classes. 
     * This will create any necessary database objects (tables, views, constraints, indexes etc). 
     * This will also cause the addition of any related classes.
     * @param classNames Name of the class(es) to be added.
     * @param clr The ClassLoaderResolver
     */
    public void addClasses(String[] classNames, ClassLoaderResolver clr)
    {
        addClasses(classNames, clr, (Writer)null, false);
    }

    /**
     * Method to add several (PersistenceCapable) classes to the store manager's set of supported classes.
     * This will create any necessary database objects (tables, views, constraints, indexes etc). 
     * This will cause the addition of any related classes.
     * <b>This is used by SchemaTool and is specific to RDBMSManager</b>
     * @param classNames Name of the class(es) to be added.
     * @param clr The ClassLoaderResolver
     * @param writer Optional writer when you just want the DDL for persisting the specified classes
     * @param completeDdl whether complete DDL will be created when writing DDL to a file, or only for missing elements
     */
    public void addClasses(String[] classNames, ClassLoaderResolver clr, Writer writer, boolean completeDdl)
    {
        if (writer != null)
        {
            this.ddlWriter = writer;
            this.completeDDL = completeDdl;
            this.writtenDdlStatements = new HashSet();
        }

        synchronized (CLASSADDER_MUTEX)
        {
            if (classAdder != null)
            {
                // addClasses() has been recursively re-entered: just add table
                // objects for the requested classes and return.
                classAdder.addClasses(classNames, clr);
                return;
            }
        }
        if (classNames != null && classNames.length > 0)
        {
            new ClassAdder(classNames, writer).execute(clr);
        }
        if (writer != null)
        {
            this.ddlWriter = null;
            this.completeDDL = false;
            this.writtenDdlStatements.clear();
            this.writtenDdlStatements = null;
        }
    }

    /**
     * Utility to remove all classes that we are managing.
     * @param clr The ClassLoaderResolver
     */
    public void removeAllClasses(ClassLoaderResolver clr)
    {
        DeleteTablesSchemaTransaction deleteTablesTxn = new DeleteTablesSchemaTransaction(this,
            Connection.TRANSACTION_READ_COMMITTED, storeDataMgr);
        boolean success = true;
        try
        {
            deleteTablesTxn.execute(clr);
        }
        catch (NucleusException ne)
        {
            success = false;
            throw ne;
        }
        finally
        {
            if (success)
            {
                clearSchemaData();
            }
        }
    }

    /**
     * Accessor for the writer for DDL (if set).
     * @return DDL writer
     */
    public Writer getDdlWriter()
    {
        return ddlWriter;
    }

    /**
     * Accessor for whether we should generate complete DDL when in that mode.
     * Otherwise will generate "upgrade DDL".
     * @return Generate complete DDL ?
     */
    public boolean getCompleteDDL()
    {
        return completeDDL;
    }

    /**
     * When we are in SchemaTool DDL mode, return if the supplied statement is already present.
     * This is used to eliminate duplicate statements from a bidirectional relation.
     * @param stmt The statement
     * @return Whether we have that statement already
     */
    public boolean hasWrittenDdlStatement(String stmt)
    {
        return (writtenDdlStatements != null && writtenDdlStatements.contains(stmt));
    }

    /**
     * When we are in SchemaTool DDL mode, add a new DDL statement.
     * @param stmt The statement
     */
    public void addWrittenDdlStatement(String stmt)
    {
        if (writtenDdlStatements != null)
        {
            writtenDdlStatements.add(stmt);
        }
    }

    /**
     * Utility to validate the specified table.
     * This is useful where we have made an update to the columns in a table and want to
     * apply the updates to the datastore.
     * @param clr The ClassLoaderResolver
     */
    public void validateTable(final TableImpl table, ClassLoaderResolver clr)
    {
        ValidateTableSchemaTransaction validateTblTxn = new ValidateTableSchemaTransaction(this, 
            Connection.TRANSACTION_READ_COMMITTED, table);
        validateTblTxn.execute(clr);
    }

    // ---------------------------------------------------------------------------------------

    /**
     * Returns the class corresponding to the given object JDO ID. If the object
     * is an OID, return the PersistenceCapable class. If the object is a SCOID,
     * return the SCO class. If the object is an AppID PK, return the PersistenceCapable
     * class that the id relates to. If SingleFieldIdentity return the associated PC class.
     * If the object id is an application id and the user supplies the "pm" argument then
     * a check can be performed in the datastore where necessary.
     * @param id The JDO identity of some object.
     * @param clr ClassLoader resolver
     * @param om Object Manager (optional - to allow check in the datastore)
     * @return For datastore identity, return the class of the corresponding
     * object. For application identity, return the class of the corresponding
     * object or null if object does not exist.
     * @exception ClassCastException If the type of ID is not recognized (
     * {@link OID}or {@link SCOID}).
     */
    public String getClassNameForObjectID(Object id, ClassLoaderResolver clr, ObjectManager om)
    {
        // Object is a SCOID
        if (id instanceof SCOID)
        {
            return ((SCOID) id).getSCOClass();
        }
        // Multiple subclasses so we need to find which one using a query
        // Only include classes which have a table.
        List<StoreData> classTree = new ArrayList();

        String targetClassName = null;
        AbstractClassMetaData cmd = null;
        ApiAdapter api = getApiAdapter();

        if (id instanceof OID)
        {
            // Object is an OID
            OID oid = (OID) id;
            cmd = getMetaDataManager().getMetaDataForClass(oid.getPcClass(), clr);
            if (!cmd.hasDiscriminatorStrategy())
            {
                // Using SingleFieldIdentity so can assume that object is of the target class or a subclass
                List subclasses = new ArrayList(this.getSubClassesForClass(oid.getPcClass(), true, clr));
                if (subclasses.size() < 1)
                {
                    NucleusLogger.PERSISTENCE.debug("1) Id \""+id+"\" has been determined to be the id of class "+oid.getPcClass());
                    // No subclasses so must be an instance of the OID class
                    return oid.getPcClass();
                }
                targetClassName = oid.getPcClass();
            }
        }
        else if (api.isSingleFieldIdentity(id))
        {
            // Using SingleFieldIdentity so can assume that object is of the target class or a subclass
            targetClassName = api.getTargetClassNameForSingleFieldIdentity(id);

            cmd = getMetaDataManager().getMetaDataForClass(targetClassName, clr);
            if (cmd.getIdentityType() != IdentityType.APPLICATION ||
                !cmd.getObjectidClass().equals(id.getClass().getName()))
            {
                throw new NucleusUserException(LOCALISER_RDBMS.msg("050022", id, cmd.getFullClassName()));
            }
            if (cmd.hasDiscriminatorStrategy())
            {
                targetClassName = null;
            }
        }
        else
        {
            // Find all of the application identity PK classes of this type
            Collection<AbstractClassMetaData> pkCmds =
                getMetaDataManager().getClassMetaDataWithApplicationId(id.getClass().getName());
            if (pkCmds != null && pkCmds.size() > 0)
            {
                Iterator<AbstractClassMetaData> iter = pkCmds.iterator();
                while (iter.hasNext())
                {
                    AbstractClassMetaData pkCmd = iter.next();
                    RDBMSStoreData storeData = (RDBMSStoreData)storeDataMgr.get(pkCmd.getFullClassName());
                    if (storeData != null && storeData.hasTable() && storeData.isFCO())
                    {
                        classTree.add(storeData);
                    }
                }
            }
        }

        //OID or SingleFieldIdentity
        if (targetClassName != null)
        {
            RDBMSStoreData data = (RDBMSStoreData)storeDataMgr.get(targetClassName);
            if (data != null)
            {
                if (data.hasTable())
                {
                    classTree.add(data);
                }
                HashSet subclasses = getSubClassesForClass(targetClassName, true, clr);
                Iterator subclassesIter = subclasses.iterator();
                while (subclassesIter.hasNext())
                {
                    String subclassName = (String)subclassesIter.next();
                    RDBMSStoreData subclassData = (RDBMSStoreData)storeDataMgr.get(subclassName);
                    if (subclassData.hasTable())
                    {
                        classTree.add(subclassData);
                    }
                }
            }
        }

        if (classTree.size() == 0 && (cmd == null || !cmd.hasDiscriminatorStrategy()))
        {
            NucleusLogger.PERSISTENCE.debug("2) Object id of \""+id+
                "\" doesnt define a known persistable class in this persistence factory." +
                " Make sure that the class is registered (e.g pm.getExtent)");
            return null;
        }
        else if (classTree.size() == 1)
        {
            cmd = getMetaDataManager().getMetaDataForClass(((RDBMSStoreData)classTree.get(0)).getName(), clr);
            if (cmd == null || !cmd.hasDiscriminatorStrategy() )
            {
                NucleusLogger.PERSISTENCE.debug("3) Id \""+id+"\" has been determined to be the id of class "+
                    classTree.get(0).getName());
                return classTree.get(0).getName();
            }
        }
        if (om != null)
        {
            // Perform a check on the exact object inheritance level with this key (uses SQL query)
            String className = null;
            if (cmd != null && cmd.hasDiscriminatorStrategy())
            {
                // Class has discriminator so do a discriminator query allowing for all possible subclasses
                DatastoreClass primaryTable = this.getDatastoreClass(cmd.getFullClassName(), om.getClassLoaderResolver());
                if (primaryTable != null)
                {
                    classTree.add(storeDataMgr.get(cmd.getFullClassName()));
                    HashSet<String> subclasses = getSubClassesForClass(cmd.getFullClassName(), true, clr);
                    Iterator<String> subclassesIter = subclasses.iterator();
                    while (subclassesIter.hasNext())
                    {
                        String subclassName = subclassesIter.next();
                        RDBMSStoreData subclassData = (RDBMSStoreData)storeDataMgr.get(subclassName);
                        if (subclassData.hasTable())
                        {
                            classTree.add(subclassData);
                        }
                    }
                    className = RDBMSStoreHelper.getClassNameForIdKeyUsingDiscriminator(om, id, classTree);
                    if (className != null)
                    {
                        NucleusLogger.PERSISTENCE.debug("4) Id \""+id+"\" has been determined to be the id of class "+className);
                        return className;
                    }
                    return null;
                }
            }
            else
            {
                // No discriminator so use a UNION
                className = RDBMSStoreHelper.getClassNameForIdKeyUsingUnion(om, id, classTree);
            }
            if (className != null)
            {
                NucleusLogger.PERSISTENCE.debug("5) Id \""+id+"\" has been determined to be the id of class "+className);
                return className;
            }

            if (id instanceof OID)
            {
                NucleusLogger.PERSISTENCE.debug("6) Id \""+id+"\" has been determined to be the id of class "+((OID)id).getPcClass());
                return ((OID)id).getPcClass();
            }
        }
        else
        {
            // Check not possible so just return the first
            NucleusLogger.PERSISTENCE.debug("7) Id \""+id+"\" has been determined to be the id of class "+
                classTree.get(0).getName());
            return classTree.get(0).getName();
        }

        return null;
    }

    /**
     * Method to return a FieldManager for extracting information from the supplied results.
     * @param sm StateManager for the object
     * @param resultSet The results
     * @param resultMappings Mappings for the results for this class
     * @return FieldManager to use
     */
    public FieldManager getFieldManagerForResultProcessing(StateManager sm, Object resultSet,
            StatementClassMapping resultMappings)
    {
        return new ResultSetGetter(sm, resultSet, resultMappings);
    }

    /**
     * Method to return a FieldManager for populating information in statements.
     * @param sm The state manager for the object.
     * @param stmt The Prepared Statement to set values on.
     * @param stmtMappings the index of parameters/mappings
     * @param checkNonNullable Whether to check for nullability
     * @return The FieldManager to use
     */
    public FieldManager getFieldManagerForStatementGeneration(StateManager sm, Object stmt,
        StatementClassMapping stmtMappings, boolean checkNonNullable)
    {
        return new ParameterSetter(sm, stmt, stmtMappings, true);
    }

    /**
     * Method to return the value from the results for the mapping at the specified position.
     * @param resultSet The results
     * @param mapping The mapping
     * @param position The position in the results
     * @return The value at that position
     * @throws NucleusDataStoreException if an error occurs accessing the results
     */
    public Object getResultValueAtPosition(Object resultSet, JavaTypeMapping mapping, int position)
    {
        try
        {
            return ((ResultSet)resultSet).getObject(position);
        }
        catch (SQLException sqle)
        {
            throw new NucleusDataStoreException(sqle.getMessage(), sqle);
        }
    }

    /**
     * Accessor for an Extent for a class, and its subclasses.
     * @param om The ObjectManager
     * @param c The (candidate) class to use for the Extent
     * @param subclasses Whether to include subclasses of 'c'
     * @return The Extent
     * @exception NoExtentException if an extent is not managed for the specified class
     */
    public Extent getExtent(ObjectManager om, Class c, boolean subclasses)
    {
        // TODO Remove this when we swap out the legacy implementation of JDOQL
        String jdoqlImplName = omfContext.getPersistenceConfiguration().getStringProperty(
            "datanucleus.query.JDOQL.implementation");
        if (jdoqlImplName != null && jdoqlImplName.equalsIgnoreCase("JDOQL2"))
        {
            return super.getExtent(om, c, subclasses);
        }
        else
        {
            return ExtentHelper.getExtent(this, c, subclasses, om);
        }
    }

    /**
     * Accessor for the next value from the specified generator.
     * This implementation caters for datastore-specific generators and provides synchronisation
     * on the connection to the datastore.
     * @param generator The generator
     * @return The next value.
     */
    protected Object getStrategyValueForGenerator(ValueGenerator generator, final ObjectManager om)
    {
        Object oid = null;
        synchronized (generator)
        {
            // Get the next value for this generator for this ObjectManager
            // Note : this is synchronised since we dont want to risk handing out this generator
            // while its connectionProvider is set to that of a different ObjectManager
            // It maybe would be good to change ValueGenerator to have a next taking the connectionProvider
            if (generator instanceof AbstractDatastoreGenerator)
            {
                // RDBMS-based generator so set the connection provider
                final RDBMSStoreManager thisStoreMgr = this;
                ValueGenerationConnectionProvider connProvider = new ValueGenerationConnectionProvider()
                {
                    ManagedConnection mconn;
                    public ManagedConnection retrieveConnection()
                    {
                        PersistenceConfiguration conf = om.getOMFContext().getPersistenceConfiguration();
                        if (conf.getStringProperty("datanucleus.valuegeneration.transactionAttribute").equalsIgnoreCase("UsePM"))
                        {
                            this.mconn = thisStoreMgr.getConnection(om);
                        }
                        else
                        {
                            int isolationLevel = TransactionUtils.getTransactionIsolationLevelForName(
                                conf.getStringProperty("datanucleus.valuegeneration.transactionIsolation"));
                            this.mconn = thisStoreMgr.getConnection(isolationLevel);
                        }
                        return mconn;
                    }

                    public void releaseConnection()
                    {
                        try
                        {
                            PersistenceConfiguration conf = om.getOMFContext().getPersistenceConfiguration();
                            if (conf.getStringProperty("datanucleus.valuegeneration.transactionAttribute").equalsIgnoreCase("UsePM"))
                            {
                                mconn.release();
                            }
                            else
                            {
                                mconn.close();
                            }
                            mconn = null;
                        }
                        catch (NucleusException e)
                        {
                            String msg = LOCALISER_RDBMS.msg("050025", e);
                            NucleusLogger.VALUEGENERATION.error(msg);
                            throw new NucleusDataStoreException(msg, e);
                        }
                    }
                };
                ((AbstractDatastoreGenerator)generator).setConnectionProvider(connProvider);
            }
            oid = generator.next();
        }
        return oid;
    }

    /**
     * Method to return the properties to pass to the generator for the specified field.
     * @param cmd MetaData for the class
     * @param absoluteFieldNumber Number of the field (-1 = datastore identity)
     * @param om Object Manager
     * @param seqmd Any sequence metadata
     * @param tablegenmd Any table generator metadata
     * @return The properties to use for this field
     */
    protected Properties getPropertiesForGenerator(AbstractClassMetaData cmd, int absoluteFieldNumber,
            ObjectManager om, SequenceMetaData seqmd, TableGeneratorMetaData tablegenmd)
    {
        AbstractMemberMetaData mmd = null;
        IdentityStrategy strategy = null;
        String sequence = null;
        ExtensionMetaData[] extensions = null;
        if (absoluteFieldNumber >= 0)
        {
            // real field
            mmd = cmd.getMetaDataForManagedMemberAtAbsolutePosition(absoluteFieldNumber);
            strategy = mmd.getValueStrategy();
            sequence = mmd.getSequence();
            extensions = mmd.getExtensions();
        }
        else
        {
            // datastore-identity surrogate field
            // always use the root IdentityMetaData since the root class defines the identity
            IdentityMetaData idmd = cmd.getBaseIdentityMetaData();
            strategy = idmd.getValueStrategy();
            sequence = idmd.getSequence();
            extensions = idmd.getExtensions();
        }

        // Get base table with the required field
        DatastoreClass tbl = getDatastoreClass(cmd.getBaseAbstractClassMetaData().getFullClassName(),
            om.getClassLoaderResolver());
        if (tbl == null)
        {
            tbl = getTableForStrategy(cmd,absoluteFieldNumber,om.getClassLoaderResolver());
        }
        JavaTypeMapping m = null;
        if (mmd != null)
        {
            m = tbl.getMemberMapping(mmd);
            if (m == null)
            {
                // Field not mapped in root table so use passed-in table
                tbl = getTableForStrategy(cmd,absoluteFieldNumber,om.getClassLoaderResolver());
                m = tbl.getMemberMapping(mmd);
            }
        }
        else
        {
            m = tbl.getIdMapping();
        }
        StringBuffer columnsName = new StringBuffer();
        for (int i = 0; i < m.getNumberOfDatastoreMappings(); i++)
        {
            if (i > 0)
            {
                columnsName.append(",");
            }
            columnsName.append(m.getDatastoreMapping(i).getDatastoreField().getIdentifier().toString());
        }

        Properties properties = new Properties();
        properties.setProperty("class-name", cmd.getFullClassName());
        properties.put("root-class-name", cmd.getBaseAbstractClassMetaData().getFullClassName());
        if (mmd != null)
        {
            properties.setProperty("field-name", mmd.getFullFieldName());
        }
        if (cmd.getCatalog() != null)
        {
            properties.setProperty("catalog-name", cmd.getCatalog());
        }
        if (cmd.getSchema() != null)
        {
            properties.setProperty("schema-name", cmd.getSchema());
        }
        properties.setProperty("table-name", tbl.getIdentifier().toString());
        properties.setProperty("column-name", columnsName.toString());

        if (sequence != null)
        {
            properties.setProperty("sequence-name", sequence);
        }

        // Add any extension properties
        if (extensions != null)
        {
            for (int i=0;i<extensions.length;i++)
            {
                properties.put(extensions[i].getKey(), extensions[i].getValue());
            }
        }

        if (strategy == IdentityStrategy.INCREMENT && tablegenmd != null)
        {
            // User has specified a TableGenerator (JPA)
            properties.put("key-initial-value", "" + tablegenmd.getInitialValue());
            properties.put("key-cache-size", "" + tablegenmd.getAllocationSize());
            if (tablegenmd.getTableName() != null)
            {
                properties.put("sequence-table-name", tablegenmd.getTableName());
            }
            if (tablegenmd.getCatalogName() != null)
            {
                properties.put("sequence-catalog-name", tablegenmd.getCatalogName());
            }
            if (tablegenmd.getSchemaName() != null)
            {
                properties.put("sequence-schema-name", tablegenmd.getSchemaName());
            }
            if (tablegenmd.getPKColumnName() != null)
            {
                properties.put("sequence-name-column-name", tablegenmd.getPKColumnName());
            }
            if (tablegenmd.getPKColumnName() != null)
            {
                properties.put("sequence-nextval-column-name", tablegenmd.getValueColumnName());
            }
            if (tablegenmd.getPKColumnValue() != null)
            {
                properties.put("sequence-name", tablegenmd.getPKColumnValue());
            }

            // Using JPA generator so don't enable initial value detection
            properties.remove("table-name");
            properties.remove("column-name");
        }
        else if (strategy == IdentityStrategy.SEQUENCE && seqmd != null)
        {
            // User has specified a SequenceGenerator (JDO/JPA)
            if (seqmd.getDatastoreSequence() != null)
            {
                if (seqmd.getInitialValue() >= 0)
                {
                    properties.put("key-start-with", "" + seqmd.getInitialValue());
                }
                if (seqmd.getAllocationSize() > 0)
                {
                    // Set both variants
                    properties.put("key-increment-by", "" + seqmd.getAllocationSize());
                    properties.put("key-cache-size", "" + seqmd.getAllocationSize());
                }
                properties.put("sequence-name", "" + seqmd.getDatastoreSequence());

                // Add on any extensions specified on the sequence
                ExtensionMetaData[] seqExtensions = seqmd.getExtensions();
                if (seqExtensions != null)
                {
                    for (int i=0;i<seqExtensions.length;i++)
                    {
                        properties.put(seqExtensions[i].getKey(), seqExtensions[i].getValue());
                    }
                }
            }
            else
            {
                // JDO Factory-based sequence generation
                // TODO Support this
            }
        }
        return properties;
    }

    private DatastoreClass getTableForStrategy(AbstractClassMetaData cmd, int fieldNumber, ClassLoaderResolver clr)
    {
        DatastoreClass t = getDatastoreClass(cmd.getFullClassName(), clr);
        if (t == null && cmd.getInheritanceMetaData().getStrategy() == InheritanceStrategy.SUBCLASS_TABLE)
        {
            throw new NucleusUserException(LOCALISER.msg("032013", cmd.getFullClassName()));
        }

        if (fieldNumber>=0)
        {
            AbstractMemberMetaData fmd = cmd.getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber);
            t = t.getBaseDatastoreClassWithMember(fmd);
        }
        else if (t!=null)
        {
            // Go up to overall superclass to find id for that class.
            boolean has_superclass = true;
            while (has_superclass)
            {
                DatastoreClass supert = t.getSuperDatastoreClass();
                if (supert != null)
                {
                    t = supert;
                }
                else
                {
                    has_superclass = false;
                }
            }
        }
        return t;
    }

    /**
     * Method defining which value-strategy to use when the user specifies "native".
     * @return Just returns "increment". Should be overridden by all store managers that have other behaviour.
     */
    protected String getStrategyForNative(AbstractClassMetaData cmd, int absFieldNumber)
    {
        // Using "native" generator so pick the most suitable
        String sequence = null;
        if (absFieldNumber >= 0)
        {
            // real field
            sequence = cmd.getMetaDataForManagedMemberAtAbsolutePosition(absFieldNumber).getSequence();
        }
        else
        {
            // datastore-identity surrogate field
            sequence = cmd.getIdentityMetaData().getSequence();
        }

        if (dba.supportsOption(DatastoreAdapter.SEQUENCES) && sequence != null)
        {
            return "sequence";
        }
        else
        {
            return "table-sequence"; // Maybe ought to use "increment"
        }
    }

    /**
     * Accessor for the (default) SQL type info for the specified JDBC type
     * @param jdbcType JDBC type
     * @return (default) SQL type
     * @throws UnsupportedDataTypeException If the JDBC type is not found
     */
    public SQLTypeInfo getSQLTypeInfoForJDBCType(int jdbcType)
    throws UnsupportedDataTypeException
    {
        // NB The connection first arg is not required since will be cached from initialisation stage
        RDBMSTypesInfo typesInfo = (RDBMSTypesInfo)schemaHandler.getSchemaData(null, "types", null);
        JDBCTypeInfo jdbcTypeInfo = (JDBCTypeInfo)typesInfo.getChild("" + jdbcType);
        if (jdbcTypeInfo.getNumberOfChildren() == 0)
        {
            throw new UnsupportedDataTypeException(LOCALISER.msg("051005",
                JDBCUtils.getNameForJDBCType(jdbcType)));
        }

        // Use default type
        return (SQLTypeInfo)jdbcTypeInfo.getChild("DEFAULT");
    }

    /**
     * Returns the column info for a column name. This should be used instead
     * of making direct calls to DatabaseMetaData.getColumns().
     * <p>
     * Where possible, this method loads and caches column info for more than
     * just the table being requested, improving performance by reducing the
     * overall number of calls made to DatabaseMetaData.getColumns() (each of
     * which usually results in one or more database queries).
     * </p>
     * @param table The table/view
     * @param conn JDBC connection to the database.
     * @param column the column
     * @return The ColumnInfo objects describing the column.
     * @throws SQLException
     */
    public RDBMSColumnInfo getColumnInfoForColumnName(Table table, Connection conn, DatastoreIdentifier column)
    throws SQLException
    {
        RDBMSColumnInfo colInfo = (RDBMSColumnInfo)schemaHandler.getSchemaData(
            conn, "column", new Object[] {table, column.getIdentifierName()});
        return colInfo;
    }

    /**
     * Returns the column info for a database table. This should be used instead
     * of making direct calls to DatabaseMetaData.getColumns().
     * <p>
     * Where possible, this method loads and caches column info for more than
     * just the table being requested, improving performance by reducing the
     * overall number of calls made to DatabaseMetaData.getColumns() (each of
     * which usually results in one or more database queries).
     * </p>
     * @param table The table/view
     * @param conn JDBC connection to the database.
     * @return A list of ColumnInfo objects describing the columns of the table.
     * The list is in the same order as was supplied by getColumns(). If no
     * column info is found for the given table, an empty list is returned.
     * @throws SQLException
     */
    public List getColumnInfoForTable(Table table, Connection conn)
    throws SQLException
    {
        RDBMSTableInfo tableInfo = (RDBMSTableInfo)schemaHandler.getSchemaData(conn, "columns",
            new Object[] {table});
        if (tableInfo == null)
        {
            return Collections.EMPTY_LIST;
        }

        List cols = new ArrayList(tableInfo.getNumberOfChildren());
        cols.addAll(tableInfo.getChildren());
        return cols;
    }

    /**
     * Method to invalidate the cached column info for a table.
     * This is called when we have just added columns to the table in the schema
     * has the effect of a reload of the tables information the next time it is needed.
     * @param table The table
     */
    public void invalidateColumnInfoForTable(Table table)
    {
        RDBMSSchemaInfo schemaInfo = (RDBMSSchemaInfo)schemaHandler.getSchemaData(null, "tables", null);
        if (schemaInfo != null && schemaInfo.getNumberOfChildren() > 0)
        {
            schemaInfo.getChildren().remove(table.getIdentifier().getFullyQualifiedName(true));
        }
    }

    /**
     * Convenience accessor of the Table objects managed in this datastore at this point.
     * @param catalog Name of the catalog to restrict the collection by
     * @param schema Name of the schema to restrict the collection by
     * @return Collection<Table>
     */
    public Collection getManagedTables(String catalog, String schema)
    {
        if (storeDataMgr == null)
        {
            return Collections.EMPTY_SET;
        }

        Collection tables = new HashSet();
        for (Iterator<StoreData> i = storeDataMgr.getManagedStoreData().iterator(); i.hasNext();)
        {
            RDBMSStoreData sd = (RDBMSStoreData) i.next();
            if (sd.getDatastoreContainerObject() != null)
            {
                // Catalog/Schema match if either managed table not set, or input requirements not set
                DatastoreIdentifier identifier = sd.getDatastoreContainerObject().getIdentifier();
                boolean catalogMatches = true;
                boolean schemaMatches = true;
                if (catalog != null && identifier.getCatalogName() != null &&
                    !catalog.equals(identifier.getCatalogName()))
                {
                    catalogMatches = false;
                }
                if (schema != null && identifier.getSchemaName() != null &&
                    !schema.equals(identifier.getSchemaName()))
                {
                    schemaMatches = false;
                }
                if (catalogMatches && schemaMatches)
                {
                    tables.add(sd.getDatastoreContainerObject());
                }
            }
        }
        return tables;
    }

    /**
     * Resolves an identifier macro. The public fields <var>className</var>, <var>fieldName </var>,
     * and <var>subfieldName </var> of the given macro are taken as inputs, and the public
     * <var>value </var> field is set to the SQL identifier of the corresponding database table or column.
     * @param im The macro to resolve.
     * @param clr The ClassLoaderResolver
     */
    public void resolveIdentifierMacro(MacroString.IdentifierMacro im, ClassLoaderResolver clr)
    {
        DatastoreClass ct = getDatastoreClass(im.className, clr);
        if (im.fieldName == null)
        {
            im.value = ct.getIdentifier().toString();
            return;
        }

        JavaTypeMapping m;
        if (im.fieldName.equals("this")) // TODO This should be candidate alias or something, not hardcoded "this"
        {
            if (!(ct instanceof ClassTable))
            {
                throw new NucleusUserException(LOCALISER_RDBMS.msg("050034", im.className));
            }

            if (im.subfieldName != null)
            {
                throw new NucleusUserException(LOCALISER_RDBMS.msg("050035", im.className, im.fieldName, im.subfieldName));
            }
            m = ((DatastoreContainerObject) ct).getIdMapping();
        }
        else
        {
            AbstractMemberMetaData fmd = getMetaDataManager().getMetaDataForMember(im.className, im.fieldName, clr);
            m = ct.getMemberMapping(fmd);
            DatastoreContainerObject t = getDatastoreContainerObject(fmd);
            if (im.subfieldName == null)
            {
                if (t != null)
                {
                    im.value = t.getIdentifier().toString();
                    return;
                }
            }
            else
            {
                if (t instanceof CollectionTable)
                {
                    CollectionTable collTable = (CollectionTable) t;
                    if (im.subfieldName.equals("owner"))
                    {
                        m = collTable.getOwnerMapping();
                    }
                    else if (im.subfieldName.equals("element"))
                    {
                        m = collTable.getElementMapping();
                    }
                    else if (im.subfieldName.equals("index"))
                    {
                        m = collTable.getOrderMapping();
                    }
                    else
                    {
                        throw new NucleusUserException(LOCALISER_RDBMS.msg(
                            "050036", im.subfieldName, im));
                    }
                }
                else if (t instanceof MapTable)
                {
                    MapTable mt = (MapTable) t;
                    if (im.subfieldName.equals("owner"))
                    {
                        m = mt.getOwnerMapping();
                    }
                    else if (im.subfieldName.equals("key"))
                    {
                        m = mt.getKeyMapping();
                    }
                    else if (im.subfieldName.equals("value"))
                    {
                        m = mt.getValueMapping();
                    }
                    else
                    {
                        throw new NucleusUserException(LOCALISER_RDBMS.msg(
                                "050037",
                                im.subfieldName, im));
                    }
                }
                else
                {
                    throw new NucleusUserException(LOCALISER_RDBMS.msg(
                            "050035", im.className,
                            im.fieldName, im.subfieldName));
                }
            }
        }
        im.value = ((Column)m.getDatastoreMapping(0).getDatastoreField()).getIdentifier().toString();
    }

    /**
     * Method to output particular information owned by this datastore.
     * Supports "DATASTORE" and "SCHEMA" categories.
     * @param category Category of information
     * @param ps PrintStream
     * @throws Exception Thrown if an error occurs in the output process
     */
    public void printInformation(String category, PrintStream ps)
    throws Exception
    {
        RDBMSAdapter dba = (RDBMSAdapter) getDatastoreAdapter();

        super.printInformation(category, ps);

        if (category.equalsIgnoreCase("DATASTORE"))
        {
            ps.println(dba.toString());
            ps.println();
            ps.println("Database TypeInfo");

            RDBMSTypesInfo typesInfo = (RDBMSTypesInfo)schemaHandler.getSchemaData(null, "types", null);
            if (typesInfo != null)
            {
                Iterator iter = typesInfo.getChildren().keySet().iterator();
                while (iter.hasNext())
                {
                    String jdbcTypeStr = (String)iter.next();
                    short jdbcTypeNumber = 0;
                    try
                    {
                        jdbcTypeNumber = Short.valueOf(jdbcTypeStr).shortValue();
                    }
                    catch (NumberFormatException nfe) { }
                    JDBCTypeInfo jdbcType = (JDBCTypeInfo)typesInfo.getChild(jdbcTypeStr);
                    Collection sqlTypeNames = jdbcType.getChildren().keySet();

                    // SQL type names for JDBC type
                    String typeStr = "JDBC Type=" + JDBCUtils.getNameForJDBCType(jdbcTypeNumber) +
                        " sqlTypes=" + StringUtils.collectionToString(sqlTypeNames);
                    ps.println(typeStr);

                    // Default SQL type details
                    SQLTypeInfo sqlType = (SQLTypeInfo)jdbcType.getChild("DEFAULT");
                    ps.println(sqlType);
                }
            }
            ps.println("");

            // Print out the keywords info
            ps.println("Database Keywords");

            Iterator reservedWordsIter = dba.iteratorReservedWords();
            while (reservedWordsIter.hasNext())
            {
                Object words = reservedWordsIter.next();
                ps.println(words);
            }
            ps.println("");
        }
        else if (category.equalsIgnoreCase("SCHEMA"))
        {
            ps.println(dba.toString());
            ps.println();
            ps.println("TABLES");

            ManagedConnection mc = getConnection(-1);
            try
            {
                Connection conn = (Connection)mc.getConnection();
                RDBMSSchemaInfo schemaInfo = (RDBMSSchemaInfo)schemaHandler.getSchemaData(
                    conn, "tables", new Object[] {this.catalogName, this.schemaName});
                if (schemaInfo != null)
                {
                    Iterator tableIter = schemaInfo.getChildren().values().iterator();
                    while (tableIter.hasNext())
                    {
                        // Print out the table information
                        RDBMSTableInfo tableInfo = (RDBMSTableInfo)tableIter.next();
                        ps.println(tableInfo);

                        Iterator columnIter = tableInfo.getChildren().iterator();
                        while (columnIter.hasNext())
                        {
                            // Print out the column information
                            RDBMSColumnInfo colInfo = (RDBMSColumnInfo)columnIter.next();
                            ps.println(colInfo);
                        }
                    }
                }
            }
            finally
            {
                if (mc != null)
                {
                    mc.close();
                }
            }
            ps.println("");
        }
    }

    /**
     * Called by (container) Mapping objects to request the creation of a DatastoreObject (table).
     * If the specified field doesn't require a join table then this returns null.
     * If the join table already exists, then this returns it.
     * @param mmd The field metadata describing the field.
     * @param clr The ClassLoaderResolver
     * @return The container object (SetTable/ListTable/MapTable/ArrayTable)
     */
    public synchronized DatastoreContainerObject newJoinDatastoreContainerObject(AbstractMemberMetaData mmd, ClassLoaderResolver clr)
    {
        if (mmd.getJoinMetaData() == null)
        {
            AbstractMemberMetaData[] relatedMmds = mmd.getRelatedMemberMetaData(clr);
            if (relatedMmds != null && relatedMmds[0].getJoinMetaData() != null)
            {
                // Join specified at other end of a bidirectional relation so create a join table
            }
            else
            {
                Class element_class;
                if (mmd.hasCollection())
                {
                    element_class = clr.classForName(mmd.getCollection().getElementType());
                }
                else if (mmd.hasMap())
                {
                    MapMetaData mapmd = (MapMetaData)mmd.getContainer();
                    if (mmd.getValueMetaData() != null && mmd.getValueMetaData().getMappedBy() != null)
                    {
                        // value stored in the key table
                        element_class = clr.classForName(mapmd.getKeyType());
                    }
                    else if (mmd.getKeyMetaData() != null && mmd.getKeyMetaData().getMappedBy() != null)
                    {
                        // key stored in the value table
                        element_class = clr.classForName(mapmd.getValueType());
                    }
                    else
                    {
                        // No information given for what is stored in what, so throw it back to the user to fix the input :-)
                        throw new NucleusUserException(LOCALISER_RDBMS.msg("050050", mmd.getFullFieldName()));
                    }
                }
                else if (mmd.hasArray())
                {
                    element_class = clr.classForName(mmd.getTypeName()).getComponentType();
                }
                else
                {
                    // what is this? should not happen
                    return null;
                }

                // Check that the element class has MetaData
                if (getMetaDataManager().getMetaDataForClass(element_class, clr) != null)
                {
                    // FK relationship, so no join table
                    return null;
                }
                else if (ClassUtils.isReferenceType(element_class))
                {
                    // reference type using FK relationship so no join table
                    return null;
                }

                // Trap all non-PC elements that haven't had a join table specified but need one
                throw new NucleusUserException(LOCALISER_RDBMS.msg("050049",
                    mmd.getFullFieldName(), mmd.toString()));
            }
        }

        // Check if the join table already exists
        DatastoreContainerObject joinTable = getDatastoreContainerObject(mmd);
        if (joinTable != null)
        {
            return joinTable;
        }

        // Create a new join table for the container
        if (classAdder == null)
        {
            throw new IllegalStateException(LOCALISER_RDBMS.msg("050016"));
        }

        if (mmd.getType().isArray())
        {
            // Use Array table for array types
            return classAdder.addJoinTableForContainer(mmd, clr, ClassAdder.JOIN_TABLE_ARRAY);
        }
        else if (Map.class.isAssignableFrom(mmd.getType()))
        {
            // Use Map join table for supported map types
            return classAdder.addJoinTableForContainer(mmd, clr, ClassAdder.JOIN_TABLE_MAP);
        }
        else
        {
            // Use Set join table for collection/set types
            return classAdder.addJoinTableForContainer(mmd, clr, ClassAdder.JOIN_TABLE_COLLECTION);
        }
    }

    /**
     * A schema transaction that adds a set of classes to the RDBMSManager,
     * making them usable for persistence.
     * <p>
     * This class embodies the work necessary to activate a persistent class and
     * ready it for storage management. It is the primary mutator of a RDBMSManager.
     * </p>
     * <p>
     * Adding classes is an involved process that includes the creation and/or
     * validation in the database of tables, views, and table constraints, and
     * their corresponding Java objects maintained by the RDBMSManager. Since
     * it's a management transaction, the entire process is subject to retry on
     * SQLExceptions. It is responsible for ensuring that the procedure either
     * adds <i>all </i> of the requested classes successfully, or adds none of
     * them and preserves the previous state of the RDBMSManager exactly as it was.
     * </p>
     */
    private class ClassAdder extends AbstractSchemaTransaction
    {
        /** join table for Collection. **/
        public static final int JOIN_TABLE_COLLECTION = 1;
        /** join table for Map. **/
        public static final int JOIN_TABLE_MAP = 2;
        /** join table for Array. **/
        public static final int JOIN_TABLE_ARRAY = 3;

        /** Optional writer to dump the DDL for any classes being added. */
        private Writer ddlWriter = null;

        /** Whether to check if table/view exists */
        protected final boolean checkExistTablesOrViews;

        /** tracks the SchemaData currrently being added - used to rollback the AutoStart added classes **/
        private HashSet<RDBMSStoreData> schemaDataAdded = new HashSet();

        private final String[] classNames;

        /**
         * Constructs a new class adder transaction that will add the given classes to the RDBMSManager.
         * @param classNames Names of the (initial) class(es) to be added.
         * @param writer Optional writer for DDL when we want the DDL outputting to file instead of creating the tables
         */
        private ClassAdder(String[] classNames, Writer writer)
        {
            super(RDBMSStoreManager.this, ((RDBMSAdapter)dba).getTransactionIsolationForSchemaCreation());
            this.ddlWriter = writer;
            this.classNames = getOMFContext().getTypeManager().filterOutSupportedSecondClassNames(classNames);

            PersistenceConfiguration conf = omfContext.getPersistenceConfiguration();
            checkExistTablesOrViews = conf.getBooleanProperty("datanucleus.rdbms.checkExistTablesOrViews");
        }

        /**
         * Method to give a string version of this object.
         * @return The String version of this object.
         */
        public String toString()
        {
            return LOCALISER_RDBMS.msg("050038", catalogName, schemaName);
        }

        /**
         * Method to perform the action using the specified connection to the datastore.
         * @param clr the ClassLoaderResolver
         * @throws SQLException Thrown if an error occurs in execution.
         */
        protected void run(ClassLoaderResolver clr)
        throws SQLException
        {
            if (classNames.length == 0)
            {
                return;
            }

            synchronized (CLASSADDER_MUTEX)
            {
                classAdder = this;
                try
                {
                    addClassTablesAndValidate(classNames, clr);
                }
                finally
                {
                    classAdder = null;
                }
            }
        }

        /**
         * Called by RDBMSManager.addClasses() when it has been recursively
         * re-entered. This just adds table objects for the requested additional 
         * classes and returns.
         * @param classNames Names of the (additional) class(es) to be added.
         * @param clr the ClassLoaderResolver
         */
        private void addClasses(String[] classNames, ClassLoaderResolver clr)
        {
            // Filter out any supported classes
            classNames = getOMFContext().getTypeManager().filterOutSupportedSecondClassNames(classNames);
            if (classNames.length == 0)
            {
                return;
            }

            try
            {
                if (getCurrentConnection() == null)
                {
                    throw new IllegalStateException(LOCALISER_RDBMS.msg("050039"));
                }
            }
            catch(SQLException e)
            {
                throw new NucleusDataStoreException("SQL exception: " + this, e);
            }

            // Add the tables for these additional classes
            addClassTables(classNames, clr);
        }
        
        private int addClassTablesRecursionCounter = 0;

        /**
         * Adds a new table object (ie ClassTable or ClassView) for every
         * class in the given list. These classes
         * <ol>
         * <li>require a table</li>
         * <li>do not yet have a table initialized in the store manager.</li>
         * </ol>
         * <p>
         * This doesn't initialize or validate the tables, it just adds the
         * table objects to the RDBMSManager's internal data structures.
         * </p>
         *
         * @param classNames Names of class(es) whose tables are to be added.
         * @param clr the ClassLoaderResolver
         */
        private void addClassTables(String[] classNames, ClassLoaderResolver clr)
        {
            addClassTablesRecursionCounter += 1;
            try
            {
                Iterator iter = getMetaDataManager().getReferencedClasses(classNames, clr).iterator();
                try
                {
                    if (starter != null && starterInitialised && !starter.isOpen())
                    {
                        starter.open();
                    }

                    // Pass through the classes and create necessary tables
                    while (iter.hasNext())
                    {
                        ClassMetaData cmd = (ClassMetaData) iter.next();
                        addClassTable(cmd, clr);
                    }

                    // For data where the table wasn't defined, make a second pass.
                    // This is necessary where a subclass uses "superclass-table" and the superclass' table
                    // hadn't been defined at the point of adding this class
                    Iterator<RDBMSStoreData> addedIter = new HashSet(this.schemaDataAdded).iterator();
                    while (addedIter.hasNext())
                    {
                        RDBMSStoreData data = addedIter.next();
                        if (data.getDatastoreContainerObject() == null && data.isFCO())
                        {
                            AbstractClassMetaData cmd = (AbstractClassMetaData) data.getMetaData();
                            InheritanceMetaData imd = cmd.getInheritanceMetaData();
                            if (imd.getStrategy() == InheritanceStrategy.SUPERCLASS_TABLE)
                            {
                                AbstractClassMetaData[] managingCmds = getClassesManagingTableForClass(cmd, clr);
                                DatastoreClass superTable = null;
                                if (managingCmds != null && managingCmds.length == 1)
                                {
                                    RDBMSStoreData superData =
                                        (RDBMSStoreData) storeDataMgr.get(managingCmds[0].getFullClassName());

                                    // Assert that managing class is in the set of storeDataByClass
                                    if (superData == null)
                                    {
                                        this.addClassTables(new String[]{managingCmds[0].getFullClassName()}, clr);
                                        superData = (RDBMSStoreData) storeDataMgr.get(managingCmds[0].getFullClassName());
                                    }
                                    if (superData == null)
                                    {
                                        String msg = LOCALISER_RDBMS.msg("050013",
                                            cmd.getFullClassName());
                                        NucleusLogger.PERSISTENCE.error(msg);
                                        throw new NucleusUserException(msg);
                                    }
                                    superTable = (DatastoreClass) superData.getDatastoreContainerObject();
                                    data.setDatastoreContainerObject(superTable);
                                }
                            }
                        }
                    }
                }
                finally
                {
                    if (starter != null && starterInitialised && starter.isOpen() && addClassTablesRecursionCounter <= 1)
                    {
                        starter.close();
                    }
                }
            }
            finally
            {
                addClassTablesRecursionCounter -= 1;
            }
        }

        /**
         * Method to add a new table object (ie ClassTable or ClassView).
         * Doesn't initialize or validate the tables, just adding the table objects to the internal data structures.
         * @param cmd the ClassMetaData
         * @param clr the ClassLoaderResolver
         */
        private void addClassTable(ClassMetaData cmd, ClassLoaderResolver clr)
        {
            // Only add tables for "PERSISTENCE_CAPABLE" classes
            if (cmd.getPersistenceModifier() != ClassPersistenceModifier.PERSISTENCE_CAPABLE)
            {
                return;
            }
            if (cmd.getIdentityType() == IdentityType.NONDURABLE)
            {
                if (cmd.hasExtension("requires-table") && cmd.getValueForExtension("requires-table") != null && 
                    cmd.getValueForExtension("requires-table").equalsIgnoreCase("false"))
                {
                    return;
                }
            }
            RDBMSStoreData sd = (RDBMSStoreData) storeDataMgr.get(cmd.getFullClassName());
            if (sd == null)
            {
                // For application-identity classes with user-defined identities we check for use of the 
                // objectid-class in different inheritance trees. We prevent this to avoid problems later on.
                // The builtin objectid-classes are allowed to be duplicated.
                if (cmd.getIdentityType() == IdentityType.APPLICATION)
                {
                    if (!cmd.usesSingleFieldIdentityClass())
                    {
                        // Check whether this class has the same base persistable class as the others using the PK. 
                        // If not, then throw an error
                        String baseClassWithMetaData = cmd.getBaseAbstractClassMetaData().getFullClassName();
                        Collection<AbstractClassMetaData> pkCmds =
                            getMetaDataManager().getClassMetaDataWithApplicationId(cmd.getObjectidClass());
                        if (pkCmds != null && pkCmds.size() > 0)
                        {
                            // We already have at least 1 class using the same app id PK class
                            // so check if it is has the same persistable root class.
                            boolean in_same_tree = false;
                            String sample_class_in_other_tree = null;

                            Iterator<AbstractClassMetaData> iter = pkCmds.iterator();
                            while (iter.hasNext())
                            {
                                AbstractClassMetaData pkCmd = iter.next();
                                String otherClassBaseClass = 
                                    pkCmd.getBaseAbstractClassMetaData().getFullClassName();
                                if (otherClassBaseClass.equals(baseClassWithMetaData))
                                {
                                    in_same_tree = true;
                                    break;
                                }
                                sample_class_in_other_tree = pkCmd.getFullClassName();
                            }

                            if (!in_same_tree)
                            {
                                String error_msg = LOCALISER_RDBMS.msg("050021", cmd.getFullClassName(), 
                                    cmd.getObjectidClass(), sample_class_in_other_tree);
                                NucleusLogger.DATASTORE.error(error_msg);
                                throw new NucleusUserException(error_msg);
                            }
                        }
                    }
                }

                if (cmd.isEmbeddedOnly())
                {
                    // Nothing to do. Only persisted as SCO.
                    NucleusLogger.DATASTORE.info(LOCALISER.msg("032012", cmd.getFullClassName()));
                }
                else
                {
                    InheritanceMetaData imd = cmd.getInheritanceMetaData();
                    RDBMSStoreData sdNew = null;
                    if (imd.getStrategy() == InheritanceStrategy.SUBCLASS_TABLE)
                    {
                        // Table mapped into the table(s) of subclass(es)
                        // Just add the SchemaData entry with no table - managed by subclass
                        sdNew = new RDBMSStoreData(cmd, null, false);
                        registerStoreData(sdNew);
                    }
                    else if (imd.getStrategy() == InheritanceStrategy.COMPLETE_TABLE && 
                            cmd.isAbstract())
                    {
                        // Abstract class with "complete-table" so gets no table
                        sdNew = new RDBMSStoreData(cmd, null, false);
                        registerStoreData(sdNew);
                    }
                    else if (imd.getStrategy() == InheritanceStrategy.NEW_TABLE ||
                             imd.getStrategy() == InheritanceStrategy.COMPLETE_TABLE)
                    {
                        // Table managed by this class
                        // Generate an identifier for the table required
                        DatastoreIdentifier tableName = null;
                        RDBMSStoreData tmpData = (RDBMSStoreData) storeDataMgr.get(cmd.getFullClassName());
                        if (tmpData !=null && tmpData.getDatastoreIdentifier() != null)
                        {
                            tableName = tmpData.getDatastoreIdentifier();
                        }
                        else
                        {
                            tableName = identifierFactory.newDatastoreContainerIdentifier(cmd);
                        }

                        // Check that the required table isn't already in use
                        StoreData[] existingStoreData = getStoreDataForDatastoreContainerObject(tableName);
                        if (existingStoreData != null)
                        {
                            String existingClass = null;
                            for (int j=0;j<existingStoreData.length;j++)
                            {
                                if (!existingStoreData[j].getName().equals(cmd.getFullClassName()))
                                {
                                    existingClass = existingStoreData[j].getName();
                                    break;
                                }
                            }
                            // Give a warning and then create a new instance of the table (mapped to the same datastore object)
                            if (existingClass != null)
                            {
                                String msg = LOCALISER_RDBMS.msg("050015", cmd.getFullClassName(), 
                                    tableName.getIdentifierName(), existingClass);
                                NucleusLogger.DATASTORE.warn(msg);
                            }
                        }

                        // Create the table to use for this class
                        DatastoreClass t = null;
                        boolean hasViewDef = false;
                        if (dba.getVendorID() != null)
                        {
                            hasViewDef = cmd.hasExtension("view-definition" + '-' + dba.getVendorID());
                        }
                        if (!hasViewDef)
                        {
                            hasViewDef = cmd.hasExtension("view-definition");
                        }
                        if (hasViewDef)
                        {
                            t = new ClassView(tableName, RDBMSStoreManager.this, cmd);
                        }
                        else
                        {
                            t = new ClassTable(tableName, RDBMSStoreManager.this, cmd);
                        }

                        sdNew = new RDBMSStoreData(cmd, t, true);
                        registerStoreData(sdNew);

                        // must be initialized after registering, to avoid StackOverflowError
                        ((Table) t).preInitialize(clr);
                    }
                    else if (imd.getStrategy() == InheritanceStrategy.SUPERCLASS_TABLE)
                    {
                        // Table mapped into table of superclass
                        // Find the superclass - should have been created first
                        AbstractClassMetaData[] managingCmds = getClassesManagingTableForClass(cmd, clr);
                        DatastoreContainerObject superTable = null;
                        if (managingCmds != null && managingCmds.length == 1)
                        {
                            RDBMSStoreData superData = (RDBMSStoreData) storeDataMgr.get(managingCmds[0].getFullClassName());
                            if (superData != null)
                            {
                                // Specify the table if it already exists
                                superTable = superData.getDatastoreContainerObject();
                            }
                            sdNew = new RDBMSStoreData(cmd, superTable, false);
                            registerStoreData(sdNew);
                        }
                        else
                        {
                            String msg = LOCALISER_RDBMS.msg("050013", cmd.getFullClassName());
                            NucleusLogger.PERSISTENCE.error(msg);
                            throw new NucleusUserException(msg);
                        }
                    }
                    schemaDataAdded.add(sdNew);
                }
            }
        }

        /**
         * Adds a new table object (ie ClassTable or ClassView) for every class
         * in the given list that 1) requires an extent and 2) does not yet have
         * an extent (ie table) initialized in the store manager.
         * <p>
         * After all of the table objects, including any other tables they might
         * reference, have been added, each table is initialized and validated
         * in the database.
         * </p>
         * <p>
         * If any error occurs along the way, any table(s) that were created are
         * dropped and the state of the RDBMSManager is rolled back to the point
         * at which this method was called.
         * </p>
         * @param classNames The class(es) whose tables are to be added.
         * @param clr the ClassLoaderResolver
         */
        private void addClassTablesAndValidate(String[] classNames, ClassLoaderResolver clr)
        {
            storeDataMgr.begin();
            boolean completed = false;

            List tablesCreated = null;
            List tableConstraintsCreated = null;
            List viewsCreated = null;
            
            try
            {
                List autoCreateErrors = new ArrayList();

                // Add SchemaData entries and Table's for the requested classes
                addClassTables(classNames, clr);

                // Initialise all tables/views for the classes
                List[] toValidate = initializeClassTables(classNames, clr);

                if (toValidate[0] != null && toValidate[0].size() > 0)
                {
                    // Validate the tables
                    List[] result = performTablesValidation(toValidate[0], clr);
                    tablesCreated = result[0];
                    tableConstraintsCreated = result[1];
                    autoCreateErrors = result[2];
                }

                if (toValidate[1] != null && toValidate[1].size() > 0)
                {
                    // Validate the views
                    List[] result = performViewsValidation(toValidate[1]);
                    viewsCreated = result[0];
                    autoCreateErrors.addAll(result[1]);
                }

                // Process all errors from the above
                verifyErrors(autoCreateErrors);

                completed = true;
            }
            catch (SQLException sqle)
            {
                String msg = LOCALISER_RDBMS.msg("050044", sqle);
                NucleusLogger.DATASTORE_SCHEMA.error(msg);
                throw new NucleusDataStoreException(msg, sqle);
            }
            catch (Exception e)
            {
                if (NucleusException.class.isAssignableFrom(e.getClass()))
                {
                    throw (NucleusException)e;
                }
                else
                {
                    NucleusLogger.DATASTORE_SCHEMA.error(LOCALISER_RDBMS.msg("050044", e));
                }
                throw new NucleusException(e.toString(), e).setFatal();
            }
            finally
            {
                // If something went wrong, roll things back to the way they were before we started.
                // This may not restore the database 100% of the time (if DDL statements are not transactional) 
                // but it will always put the RDBMSManager's internal structures back the way they were.
                if (!completed)
                {
                    storeDataMgr.rollback();
                    rollbackSchemaCreation(viewsCreated,tableConstraintsCreated,tablesCreated);
                }
                else
                {
                    storeDataMgr.commit();
                }
                schemaDataAdded.clear();
            }
        }

        /**
         * Initialisation of tables. Updates the internal representation of the table to match what is 
         * required for the class(es). Each time a table object is initialized, it may cause other associated 
         * table objects to be added (via callbacks to addClasses()) so the loop is repeated until no more 
         * initialisation is needed.
         * @param classNames String array of class names
         * @param clr the ClassLoaderResolver
         * @return an array of List where index == 0 is list of the tables created, index == 1 is list of the views created
         */
        private List[] initializeClassTables(String[] classNames, ClassLoaderResolver clr)
        {
            List tablesToValidate = new ArrayList();
            List viewsToValidate = new ArrayList();
            boolean someNeededInitialization;
            List recentlyInitilized = new ArrayList();
            do
            {
                someNeededInitialization = false;
                RDBMSStoreData[] rdbmsStoreData =
                    storeDataMgr.getManagedStoreData().toArray(new RDBMSStoreData[storeDataMgr.size()]);
                for (int i=0; i<rdbmsStoreData.length; i++)
                {
                    if (rdbmsStoreData[i].hasTable())
                    {
                        Table t = (Table)rdbmsStoreData[i].getDatastoreContainerObject();
                        if (t instanceof DatastoreClass)
                        {
                            ((RDBMSPersistenceHandler)persistenceHandler).removeRequestsForTable((DatastoreClass)t);
                        }

                        // Any classes managed by their own table needing initialising
                        if (!t.isInitialized())
                        {
                            t.initialize(clr);
                            recentlyInitilized.add(t);
                            if (t instanceof ViewImpl)
                            {
                                viewsToValidate.add(t);
                            }
                            else
                            {
                                tablesToValidate.add(t);
                            }
                            someNeededInitialization = true;
                        }

                        // Any classes that are managed by other tables needing initialising
                        if (!rdbmsStoreData[i].isTableOwner() && !((ClassTable)t).managesClass(rdbmsStoreData[i].getName()))
                        {
                            ((ClassTable)t).manageClass((ClassMetaData)rdbmsStoreData[i].getMetaData(), clr);
                            if (!tablesToValidate.contains(t))
                            {
                                tablesToValidate.add(t);
                            }
                            someNeededInitialization = true;
                        }
                    }
                    else
                    {
                        // Nothing to do for cases without their own table ("subclass-table" strategy) since
                        // the table is initialised to contain those fields by the subclass.
                    }
                }
            }
            while (someNeededInitialization);

            // Post initialisation of tables
            for (int j=0; j<recentlyInitilized.size(); j++)
            {
                ((Table)recentlyInitilized.get(j)).postInitialize(clr);
            }

            return new List[] { tablesToValidate, viewsToValidate };
        }

        /**
         * Validate tables.
         * @param tablesToValidate list of TableImpl to validate
         * @param clr the ClassLoaderResolver
         * @return an array of List where index == 0 has a list of the tables created
         *                                index == 1 has a list of the contraints created
         *                                index == 2 has a list of the auto creation errors 
         * @throws SQLException
         */
        private List[] performTablesValidation(List tablesToValidate, ClassLoaderResolver clr) throws SQLException
        {
            List autoCreateErrors = new ArrayList();
            List tableConstraintsCreated = new ArrayList();
            List tablesCreated = new ArrayList();

            if (ddlWriter != null)
            {
                // Remove any existence of the same actual table more than once so we dont duplicate its
                // DDL for creation. Note that this will allow more than once instance of tables with the same
                // name (identifier) since when you have multiple inheritance trees each inheritance tree
                // will have its own ClassTable, and you want both of these to pass through to schema generation.
                tablesToValidate = removeDuplicateTablesFromList(tablesToValidate);
            }

            // Table existence and validation.
            // a). Check for existence of the table
            // b). If autocreate, create the table if necessary
            // c). If validate, validate the table
            Iterator i = tablesToValidate.iterator();
            while (i.hasNext())
            {
                TableImpl t = (TableImpl) i.next();

                boolean columnsValidated = false;
                if (checkExistTablesOrViews)
                {
                    if (ddlWriter != null)
                    {
                        try
                        {
                            if (t instanceof ClassTable)
                            {
                                ddlWriter.write("-- Table " + t.toString() + 
                                    " for classes " + StringUtils.objectArrayToString(((ClassTable)t).getManagedClasses()) + "\n");
                            }
                            else if (t instanceof JoinTable)
                            {
                                ddlWriter.write("-- Table " + t.toString() + " for join relationship\n");
                            }
                        }
                        catch (IOException ioe)
                        {
                            NucleusLogger.DATASTORE_SCHEMA.error("error writing DDL into file", ioe);
                        }
                    }

                    if (!tablesCreated.contains(t) && t.exists(getCurrentConnection(), autoCreateTables))
                    {
                        // Table has been created so add to our list so we dont process it multiple times
                        // Any subsequent instance of this table in the list will have the columns checked only
                        tablesCreated.add(t);
                        columnsValidated = true;
                    }
                    else
                    {
                        // Table wasn't just created, so do any autocreate of columns necessary
                        if (t.isInitializedModified() || autoCreateColumns)
                        {
                            // Check for existence of the required columns and add where required
                            t.validateColumns(getCurrentConnection(), false, autoCreateColumns, autoCreateErrors);
                            columnsValidated = true;
                        }
                    }
                }

                if (validateTables && !columnsValidated) // Table not just created and validation requested
                {
                    // Check down to the column structure where required
                    t.validate(getCurrentConnection(), validateColumns, false, autoCreateErrors);
                }
                else if (!columnsValidated)
                {
                    // Validation not requested but allow initialisation of the column information
                    String initInfo = omfContext.getPersistenceConfiguration().getStringProperty("datanucleus.rdbms.initializeColumnInfo");
                    if (initInfo.equalsIgnoreCase("PK"))
                    {
                        // Initialise the PK columns only
                        t.initializeColumnInfoForPrimaryKeyColumns(getCurrentConnection());
                    }
                    else if (initInfo.equalsIgnoreCase("ALL"))
                    {
                        // Initialise all columns
                        t.initializeColumnInfoFromDatastore(getCurrentConnection());
                    }
                }

                // Discard any cached column info used to validate the table
                invalidateColumnInfoForTable(t);
            }

            // Table constraint existence and validation
            // a). Check for existence of the constraint
            // b). If autocreate, create the constraint if necessary
            // c). If validate, validate the constraint
            // Constraint processing is done as a separate step from table processing
            // since the constraints are dependent on tables being available
            i = tablesToValidate.iterator();
            while (i.hasNext())
            {
                TableImpl t = (TableImpl) i.next();
                if (validateConstraints || autoCreateConstraints)
                {
                    if (ddlWriter != null)
                    {
                        try
                        {
                            if (t instanceof ClassTable)
                            {
                                ddlWriter.write("-- Constraints for table " + t.toString() + 
                                    " for class(es) " + StringUtils.objectArrayToString(((ClassTable)t).getManagedClasses()) + "\n");
                            }
                            else
                            {
                                ddlWriter.write("-- Constraints for table " + t.toString() + "\n");
                            }
                        }
                        catch (IOException ioe)
                        {
                            NucleusLogger.DATASTORE_SCHEMA.error("error writing DDL into file", ioe);
                        }
                    }
                    // TODO : split this method into checkExistsConstraints and validateConstraints
                    // TODO : if duplicated entries on the list, we need to validate before.
                    if (tablesCreated.contains(t) && !hasDuplicateTablesFromList(tablesToValidate))
                    {
                        if (t.createConstraints(getCurrentConnection(), autoCreateErrors, clr))
                        {
                            tableConstraintsCreated.add(t);
                        }
                    }
                    else if (t.validateConstraints(getCurrentConnection(), autoCreateConstraints, autoCreateErrors, clr))
                    {
                        tableConstraintsCreated.add(t);
                    }
                    if (ddlWriter != null)
                    {
                        try
                        {
                            ddlWriter.write("\n");
                        }
                        catch (IOException ioe)
                        {
                            NucleusLogger.DATASTORE_SCHEMA.error("error writing DDL into file", ioe);
                        }
                    }
                }
            }
            return new List[] { tablesCreated, tableConstraintsCreated, autoCreateErrors };
        }

        /**
         * Remove duplicated tables from the list.
         * Tables are only removed if they are the same table object. That is we dont remove if they have
         * the same table identifier.
         * @param newTables the list of DatastoreContainerObject
         * @return a distinct list with DatastoreContainerObject  
         */
        private List removeDuplicateTablesFromList(List newTables)
        {
            List result = new ArrayList();
            Set uniqueTables = new TreeSet(new Comparator()
            {
                public int compare(Object o1, Object o2)
                {
                    DatastoreContainerObject t1 = (DatastoreContainerObject) o1;
                    DatastoreContainerObject t2 = (DatastoreContainerObject) o2;

                    // Sometimes duplicates exist by name, but they are different references.
                    return StringUtils.toJVMIDString(t1.getIdentifier()).compareTo(
                        StringUtils.toJVMIDString(t2.getIdentifier()));
                }
            });
            uniqueTables.addAll(newTables);
            result.addAll(uniqueTables);
            return result;
        }

        /**
         * Check if duplicated tables are in the list.
         * @param newTables the list of DatastoreContainerObject
         * @return true if duplicated tables are in the list
         */
        private boolean hasDuplicateTablesFromList(List newTables)
        {
            Map map = new HashMap();
            for (int i=0; i<newTables.size(); i++)
            {
                DatastoreContainerObject t1 = (DatastoreContainerObject) newTables.get(i);
                if (map.containsKey(t1.getIdentifier().getIdentifierName()))
                {
                    return true;
                }
                map.put(t1.getIdentifier().getIdentifierName(), t1);
            }
            return false;
        }

        /**
         * Validate the supplied views.
         * @param viewsToValidate list of ViewImpl to validate
         * @return an array of List where index == 0 has a list of the views created
         *                                index == 1 has a list of the auto creation errors 
         * @throws SQLException
         */
        private List[] performViewsValidation(List viewsToValidate) throws SQLException
        {
            List viewsCreated = new ArrayList();
            List autoCreateErrors = new ArrayList();
            // View existence and validation.
            // a). Check for existence of the view
            // b). If autocreate, create the view if necessary
            // c). If validate, validate the view
            Iterator i = viewsToValidate.iterator();
            while (i.hasNext())
            {
                ViewImpl v = (ViewImpl) i.next();
                if (checkExistTablesOrViews)
                {
                    if (v.exists(getCurrentConnection(), autoCreateTables))
                    {
                        viewsCreated.add(v);
                    }
                }
                if (validateTables)
                {
                    v.validate(getCurrentConnection(), true, false, autoCreateErrors);
                }

                // Discard any cached column info used to validate the view
                invalidateColumnInfoForTable(v);
            }
            return new List[] { viewsCreated, autoCreateErrors };
        }

        /**
         * Verify the list of errors, log the errors and raise NucleusDataStoreException when fail on error is enabled.
         * @param autoCreateErrors the list of Throwables
         */
        private void verifyErrors(List autoCreateErrors)
        {
            if (autoCreateErrors.size() > 0)
            {
                // Print out all errors found during auto-creation/validation
                Iterator errorsIter = autoCreateErrors.iterator();
                while (errorsIter.hasNext())
                {
                    Throwable exc = (Throwable)errorsIter.next();
                    if (autoCreateWarnOnError)
                    {
                        NucleusLogger.DATASTORE.warn(LOCALISER_RDBMS.msg("050044", exc));
                    }
                    else
                    {
                        NucleusLogger.DATASTORE.error(LOCALISER_RDBMS.msg("050044", exc));
                    }
                }
                if (!autoCreateWarnOnError)
                {
                    throw new NucleusDataStoreException(LOCALISER_RDBMS.msg("050043"), 
                        (Throwable[])autoCreateErrors.toArray(new Throwable[autoCreateErrors.size()]));
                }
            }
        }

        /**
         * Rollback / Compensate schema creation by dropping tables, views, constraints and
         * deleting entries in the auto start mechanism.
         * @param viewsCreated the views created that must be dropped
         * @param tableConstraintsCreated the constraints created that must be dropped
         * @param tablesCreated the tables created that must be dropped
         */
        private void rollbackSchemaCreation(List viewsCreated, List tableConstraintsCreated, List tablesCreated)
        {
            if (NucleusLogger.DATASTORE_SCHEMA.isDebugEnabled())
            {
                NucleusLogger.DATASTORE_SCHEMA.debug(LOCALISER_RDBMS.msg("050040"));
            }

            // Tables, table constraints, and views get removed in the reverse order from which they were created.
            try
            {
                if (viewsCreated != null)
                {
                    ListIterator li = viewsCreated.listIterator(viewsCreated.size());
                    while (li.hasPrevious())
                    {
                        ((ViewImpl) li.previous()).drop(getCurrentConnection());
                    }
                }
                if( tableConstraintsCreated != null)
                {
                    ListIterator li = tableConstraintsCreated.listIterator(tableConstraintsCreated.size());
                    while (li.hasPrevious())
                    {
                        ((TableImpl) li.previous()).dropConstraints(getCurrentConnection());
                    }
                }
                if (tablesCreated != null)
                {
                    ListIterator li = tablesCreated.listIterator(tablesCreated.size());
                    while (li.hasPrevious())
                    {
                        ((TableImpl) li.previous()).drop(getCurrentConnection());
                    }
                }
            }
            catch (Exception e)
            {
                NucleusLogger.DATASTORE_SCHEMA.warn(LOCALISER_RDBMS.msg(
                    "050041", e));
            }

            // AutoStarter - Remove all classes from the supported list that were added in this pass.
            if (starter != null && starterInitialised)
            {
                try
                {
                    if (!starter.isOpen())
                    {
                        starter.open();
                    }
                    Iterator<RDBMSStoreData> schema_added_iter = schemaDataAdded.iterator();
                    while (schema_added_iter.hasNext())
                    {
                        RDBMSStoreData sd = schema_added_iter.next();
                        starter.deleteClass(sd.getName());
                    }                            
                }
                finally
                {
                    if (starter.isOpen())
                    {
                        starter.close();
                    }
                }
            }
        }

        /**
         * Called by Mapping objects in the midst of RDBMSManager.addClasses()
         * to request the creation of a join table to hold a containers' contents.
         * @param fmd The field metadata describing the Array field.
         * @param type The type of the join table
         */
        private DatastoreContainerObject addJoinTableForContainer(AbstractMemberMetaData fmd, ClassLoaderResolver clr, int type)
        {
            DatastoreIdentifier tableName = null;
            RDBMSStoreData sd = (RDBMSStoreData) storeDataMgr.get(fmd);
            if (sd != null && sd.getDatastoreIdentifier() != null)
            {
                tableName = sd.getDatastoreIdentifier();
            }
            else
            {
                tableName = identifierFactory.newDatastoreContainerIdentifier(fmd);
            }

            DatastoreContainerObject join = null;
            if (type == JOIN_TABLE_COLLECTION)
            {
                join = new CollectionTable(tableName, fmd, RDBMSStoreManager.this);
            }
            else if (type == JOIN_TABLE_MAP)
            {
                join = new MapTable(tableName, fmd, RDBMSStoreManager.this);
            }
            else if (type == JOIN_TABLE_ARRAY)
            {
                join = new ArrayTable(tableName, fmd, RDBMSStoreManager.this);
            }

            RDBMSStoreData data;
            try
            {
                if (starter != null && starterInitialised && !starter.isOpen())
                {
                    starter.open();
                }

                data = new RDBMSStoreData(fmd, join);
                registerStoreData(data);
            }
            finally
            {
                if (starter != null && starterInitialised && starter.isOpen())
                {
                    starter.close();
                }
            }

            schemaDataAdded.add(data);
            return join;
        }
    }

    /**
     * Accessor for the supported options in string form
     */
    public Collection getSupportedOptions()
    {
        Set set = new HashSet();
        set.add("ORM");
        set.add("BackedSCO");
        set.add("DatastoreIdentity");
        set.add("ApplicationIdentity");
        set.add("OptimisticTransaction");

        // Add isolation levels for this database adapter
        if (dba.supportsOption(RDBMSAdapter.TX_ISOLATION_READ_COMMITTED))
        {
            set.add("TransactionIsolationLevel.read-committed");
        }
        if (dba.supportsOption(RDBMSAdapter.TX_ISOLATION_READ_UNCOMMITTED))
        {
            set.add("TransactionIsolationLevel.read-uncommitted");
        }
        if (dba.supportsOption(RDBMSAdapter.TX_ISOLATION_REPEATABLE_READ))
        {
            set.add("TransactionIsolationLevel.repeatable-read");
        }
        if (dba.supportsOption(RDBMSAdapter.TX_ISOLATION_SERIALIZABLE))
        {
            set.add("TransactionIsolationLevel.serializable");
        }
        PersistenceConfiguration conf = omfContext.getPersistenceConfiguration();
        String jdoqlImpl = conf.getStringProperty("datanucleus.query.JDOQL.implementation");
        if (jdoqlImpl != null && jdoqlImpl.equalsIgnoreCase("JDOQL2"))
        {
            // Query Cancel and Datastore Timeout is supported on JDOQL2
            set.add("Query.Cancel");
            set.add("Datastore.Timeout");
        }

        return set;
    }

    /**
     * Accessor for whether this mapping requires values inserting on an INSERT.
     * @param datastoreMapping The datastore mapping
     * @return Whether values are to be inserted into this mapping on an INSERT
     */
    public boolean insertValuesOnInsert(DatastoreMapping datastoreMapping)
    {
        return ((RDBMSMapping)datastoreMapping).insertValuesOnInsert();
    }

    /**
     * Convenience method to return if the datastore supports batching and the user wants batching.
     * @return If batching of statements is permissible
     */
    public boolean allowsBatching()
    {
        if (dba.supportsOption(RDBMSAdapter.STATEMENT_BATCHING) &&
            getOMFContext().getPersistenceConfiguration().getIntProperty("datanucleus.rdbms.statementBatchLimit") != 0)
        {
            return true;
        }
        return false;
    }

    public ResultObjectFactory newResultObjectFactory(AbstractClassMetaData acmd, 
        StatementClassMapping mappingDefinition, boolean ignoreCache, FetchPlan fetchPlan, 
        Class persistentClass)
    {
        return new PersistentClassROF(acmd, mappingDefinition, ignoreCache, fetchPlan, persistentClass);
    }

    protected FKArrayStore newFKArrayStore(AbstractMemberMetaData ammd, ClassLoaderResolver clr)
    {
        return new RDBMSFKArrayStore(ammd, this, clr);
    }

    protected FKListStore newFKListStore(AbstractMemberMetaData ammd, ClassLoaderResolver clr)
    {
        return new RDBMSFKListStore(ammd, this, clr);
    }

    protected FKSetStore newFKSetStore(AbstractMemberMetaData ammd, ClassLoaderResolver clr)
    {
        return new RDBMSFKSetStore(ammd, this, clr);
    }

    protected FKMapStore newFKMapStore(AbstractMemberMetaData ammd, ClassLoaderResolver clr)
    {
        return new RDBMSFKMapStore(ammd, this, clr);
    }

    protected JoinArrayStore newJoinArrayStore(AbstractMemberMetaData amd, ClassLoaderResolver clr, 
            DatastoreContainerObject table)
    {
        return new RDBMSJoinArrayStore(amd, (ArrayTable)table, clr);
    }

    protected JoinMapStore newJoinMapStore(AbstractMemberMetaData amd, ClassLoaderResolver clr,
            DatastoreContainerObject table)
    {
        return new RDBMSJoinMapStore((MapTable)table, clr);
    }

    protected JoinListStore newJoinListStore(AbstractMemberMetaData amd, ClassLoaderResolver clr,
            DatastoreContainerObject table)
    {
        return new RDBMSJoinListStore(amd, (CollectionTable)table, clr);
    }

    protected JoinSetStore newJoinSetStore(AbstractMemberMetaData amd, ClassLoaderResolver clr,
            DatastoreContainerObject table)
    {
        return new RDBMSJoinSetStore(amd, (CollectionTable)table, clr);
    }
}