/**********************************************************************
Copyright (c) 2003 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2004 David Ezzio - fix the use of the default package for the persistent class
2004 Erik Bengtson - removed Connection parameter for generateTableNameForMetaData method 
2004 Andy Jefferson - added getClassForAppIdKey()
2004 Andy Jefferson - moved collection/map join table naming to generateTableName
2005 Andy Jefferson - fixed use of catalog/schema so we always query multi-schema
    ...
**********************************************************************/
package org.datanucleus.store.rdbms;

import java.lang.reflect.Modifier;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.metadata.ClassMetaData;
import org.datanucleus.metadata.DiscriminatorMetaData;
import org.datanucleus.metadata.DiscriminatorStrategy;
import org.datanucleus.state.StateManagerFactory;
import org.datanucleus.store.StoreData;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.MappedStoreData;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.rdbms.query.RDBMSQueryUtils;
import org.datanucleus.store.rdbms.sql.DiscriminatorStatementGenerator;
import org.datanucleus.store.rdbms.sql.SQLStatement;
import org.datanucleus.store.rdbms.sql.SQLStatementHelper;
import org.datanucleus.store.rdbms.sql.SQLTable;
import org.datanucleus.store.rdbms.sql.expression.SQLExpression;
import org.datanucleus.store.rdbms.sql.expression.SQLExpressionFactory;
import org.datanucleus.store.rdbms.table.ClassTable;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * Provides a series of utilities assisting in the datastore management process for RDBMS datastores.
 */
public class RDBMSStoreHelper
{
    /**
     * Utility that takes an id and a list of possible class RDBMSStoreData
     * and finds which of these classes contains the object with that id.
     * Works via a query to the datastore. Operates for all types of identity.
     * @param om Object Manager
     * @param id The id
     * @param schemaDataOptions List of possible RDBMSStoreData
     * @return Name of the class with this key (or null if none found)
     */
    public static String getClassNameForIdKeyUsingUnion(ObjectManager om, Object id, List schemaDataOptions)
    {
        // Check for input error
        if (schemaDataOptions == null || id == null || schemaDataOptions.size() == 0)
        {
            return null;
        }

        // Calculate max length of class name for identifier
        int metadata_id_len = 0;
        Iterator optionsIter = schemaDataOptions.iterator();
        while (optionsIter.hasNext())
        {
            RDBMSStoreData schemaDataOption = (RDBMSStoreData)optionsIter.next();
            metadata_id_len = Math.max(schemaDataOption.getName().length(), metadata_id_len);
        }

        String className = null;
        RDBMSStoreManager storeMgr = (RDBMSStoreManager)om.getStoreManager();
        SQLExpressionFactory exprFactory = storeMgr.getSQLExpressionFactory();
        ClassLoaderResolver clr = om.getClassLoaderResolver();

        // Form the query to find which one of these classes has the instance with this id
        SQLStatement sqlStmtMain = null;
        optionsIter = schemaDataOptions.iterator();
        String baseClassName = null;
        while (optionsIter.hasNext())
        {
            RDBMSStoreData schemaDataOption = (RDBMSStoreData)optionsIter.next();
            ClassMetaData cmd = (ClassMetaData)schemaDataOption.getMetaData();
            Class cls = clr.classForName(cmd.getFullClassName());
            if (Modifier.isAbstract(cls.getModifiers()))
            {
                // This possible class is abstract so can't be instantiated.
                continue;
            }
            if (baseClassName == null)
            {
                baseClassName = cmd.getFullClassName();
            }
            DatastoreContainerObject table = schemaDataOption.getDatastoreContainerObject();

            SQLStatement sqlStmt = new SQLStatement(storeMgr, table, null, null);
            sqlStmt.setClassLoaderResolver(clr);

            // Select NUCLEUS_TYPE
            JavaTypeMapping typeMapping = storeMgr.getMappingManager().getMapping(String.class);
            String classname = StringUtils.leftAlignedPaddedString(schemaDataOption.getName(), metadata_id_len);
            sqlStmt.select(exprFactory.newLiteral(sqlStmt, typeMapping, classname), "NUCLEUS_TYPE");

            // LEFT OUTER JOIN to all direct subclasses
            Iterator subclass_iter =
                storeMgr.getSubClassesForClass(schemaDataOption.getName(), false, clr).iterator();
            while (subclass_iter.hasNext())
            {
                String subclass = (String)subclass_iter.next();
                DatastoreClass subclassTable = storeMgr.getDatastoreClass(subclass, clr);

                // No need to LEFT OUTER JOIN for "subclass-table" and "superclass-table" cases
                // "subclass-table" objects don't exist on their own
                // "superclass-table" are excluded using the discriminator clause
                if (subclassTable != null &&
                    !subclassTable.getIdentifier().equals(sqlStmt.getPrimaryTable().getTable().getIdentifier()))
                {
                    SQLTable subSqlTbl = sqlStmt.leftOuterJoin(
                        sqlStmt.getPrimaryTable(), sqlStmt.getPrimaryTable().getTable().getIdMapping(),
                        subclassTable, null, subclassTable.getIdMapping(), null,
                        sqlStmt.getPrimaryTable().getGroupName());
                    SQLExpression subIdExpr = exprFactory.newExpression(sqlStmt, subSqlTbl, subclassTable.getIdMapping());
                    SQLExpression nullExpr = exprFactory.newLiteral(sqlStmt, null, null);
                    sqlStmt.whereAnd(subIdExpr.eq(nullExpr), true);
                }
            }

            // WHERE (object id) = ?
            JavaTypeMapping idMapping = sqlStmt.getPrimaryTable().getTable().getIdMapping();

            // We have to create a StateManager here just to map fields from the AppId key object
            // to the table fields. Really the table should have some way of doing this. TODO : Refactor this
            Class pc_class = clr.classForName(schemaDataOption.getName());
            StateManager sm = StateManagerFactory.newStateManagerForHollow(om,pc_class,id);
            SQLExpression fieldExpr = exprFactory.newExpression(sqlStmt, sqlStmt.getPrimaryTable(), idMapping);
            SQLExpression fieldVal = exprFactory.newLiteralParameter(sqlStmt, idMapping, sm.getObject(), "ID");
            sqlStmt.whereAnd(fieldExpr.eq(fieldVal), true);
 
            // Discriminator for this class
            JavaTypeMapping discrimMapping = table.getDiscriminatorMapping(false);
            if (discrimMapping != null)
            {
                DiscriminatorMetaData discrimMetaData = cmd.getInheritanceMetaData().getDiscriminatorMetaData();
                Object value = null;
                if (cmd.getDiscriminatorStrategyForTable() == DiscriminatorStrategy.CLASS_NAME)
                {
                    value = schemaDataOption.getName();
                }
                else if (cmd.getDiscriminatorStrategyForTable() == DiscriminatorStrategy.VALUE_MAP)
                {
                    value = discrimMetaData.getValue();
                }
                SQLExpression discrimExpr =
                    exprFactory.newExpression(sqlStmt, sqlStmt.getPrimaryTable(), discrimMapping);
                SQLExpression discrimVal = exprFactory.newLiteralParameter(sqlStmt, discrimMapping, value, "DISCRIM");
                sqlStmt.whereAnd(discrimExpr.eq(discrimVal), false);
            }

            if (sqlStmtMain == null)
            {
                sqlStmtMain = sqlStmt;
            }
            else
            {
                sqlStmtMain.union(sqlStmt);
            }
        }

        // Perform the query
        try
        {
            ManagedConnection mconn = storeMgr.getConnection(om);
            SQLController sqlControl = storeMgr.getSQLController();
            if (om.getSerializeReadForClass(baseClassName))
            {
                sqlStmtMain.addExtension("lock-for-update", true);
            }

            try
            {
                PreparedStatement ps = SQLStatementHelper.getPreparedStatementForSQLStatement(sqlStmtMain,
                    om, mconn, null, null);
                String statement = sqlStmtMain.getSelectStatement().toSQL();
                try
                {
                    ResultSet rs = sqlControl.executeStatementQuery(mconn, statement, ps);
                    try
                    {
                        if (rs != null)
                        {
                            while (rs.next())
                            {
                                try
                                {
                                    className = rs.getString("NUCLEUS_TYPE").trim();
                                }
                                catch (SQLException sqle)
                                {
                                }
                            }
                        }
                    }
                    finally
                    {
                        rs.close();
                    }
                }
                finally
                {
                    sqlControl.closeStatement(mconn, ps);
                }
            }
            finally
            {
                mconn.release();
            }
        }
        catch (SQLException sqe)
        {
            NucleusLogger.DATASTORE.error(sqe);
            throw new NucleusDataStoreException(sqe.toString());
        }

        return className;
    }

    /**
     * Utility that takes an id and a list of possible class RDBMSStoreData
     * and finds which of these classes contains the object with that id.
     * Works via a query to the datastore. Operates for all types of identity.
     * @param om Object Manager
     * @param id The id
     * @param schemaDataOptions List of possible RDBMSStoreData
     * @return Name of the class with this key (or null if none found)
     */
    public static String getClassNameForIdKeyUsingDiscriminator(ObjectManager om, Object id, List schemaDataOptions)
    {
        // Check for input error
        if (schemaDataOptions == null || schemaDataOptions.size() == 0 || id == null)
        {
            return null;
        }

        String className = null;
        RDBMSStoreManager storeMgr = (RDBMSStoreManager)om.getStoreManager();
        SQLExpressionFactory exprFactory = storeMgr.getSQLExpressionFactory();
        ClassLoaderResolver clr = om.getClassLoaderResolver();

        Class primaryClass = clr.classForName(((RDBMSStoreData)schemaDataOptions.get(0)).getName());
        Class objectClass = primaryClass;
        if (Modifier.isAbstract(primaryClass.getModifiers()))
        {
            // The primary class is abstract so find one that isn't. We only need a valid object type
            Iterator optionsIter = schemaDataOptions.iterator();
            while (optionsIter.hasNext())
            {
                RDBMSStoreData data = (RDBMSStoreData)optionsIter.next();
                Class dataClass = clr.classForName(data.getName());
                if (!Modifier.isAbstract(dataClass.getModifiers()))
                {
                    objectClass = dataClass;
                    break;
                }
            }
        }
        if (Modifier.isAbstract(objectClass.getModifiers()))
        {
            throw new NucleusException("Class "+objectClass.getName()+" is abstract, and a non abstract was expected.").setFatal();
        }

        // Check all instances of this table to see if we have all possible candidates in our "options" list
        DatastoreClass primaryTable = storeMgr.getDatastoreClass(primaryClass.getName(), clr);
        StoreData[] storeData = storeMgr.getStoreDataForDatastoreContainerObject(primaryTable.getIdentifier());
        boolean haveAllCandidates = true;
        for (int i=0;i<storeData.length;i++)
        {
            if (storeData[i] instanceof MappedStoreData)
            {
                ClassTable tbl = (ClassTable)((MappedStoreData)storeData[i]).getDatastoreContainerObject();
                String[] managedClasses = tbl.getManagedClasses();
                for (int j=0;j<managedClasses.length;j++)
                {
                    boolean managedClassFound = false;
                    Iterator optionsIter = schemaDataOptions.iterator();
                    while (optionsIter.hasNext())
                    {
                        StoreData optionData = (StoreData)optionsIter.next();
                        if (optionData.getName().equals(managedClasses[j]))
                        {
                            managedClassFound = true;
                            break;
                        }
                    }
                    if (!managedClassFound)
                    {
                        haveAllCandidates = false;
                        break;
                    }
                }
                if (!haveAllCandidates)
                {
                    break;
                }
            }
        }

        // Form the query to find which one of these classes has the instance with this id
        DiscriminatorStatementGenerator stmtGen =
            new DiscriminatorStatementGenerator(storeMgr, clr, primaryClass, true, null, null);
        if (haveAllCandidates)
        {
            // We have all possible candidates that are stored in this table so we can omit the restriction on the discriminator
            stmtGen.unsetOption("restrictDiscriminator"); // Just retrieve the discriminator for this id (accept any possible disc values)
        }
        else
        {
            stmtGen.setOption("restrictDiscriminator");
        }
        SQLStatement sqlStmt = stmtGen.getStatement();
        JavaTypeMapping discrimMapping = primaryTable.getDiscriminatorMapping(true);
        SQLTable discrimSqlTbl = SQLStatementHelper.getSQLTableForMappingOfTable(sqlStmt, sqlStmt.getPrimaryTable(), discrimMapping);
        sqlStmt.select(discrimSqlTbl, discrimMapping, null);

        // Note : need to create a StateManager here just to map fields from the AppId key object to the columns
        // Really the table should have some way of doing this. TODO : Refactor this
        StateManager sm = StateManagerFactory.newStateManagerForHollow(om, objectClass, id);
        JavaTypeMapping idMapping = primaryTable.getIdMapping();
        SQLExpression sqlFldExpr = exprFactory.newExpression(sqlStmt, sqlStmt.getPrimaryTable(), idMapping);
        SQLExpression sqlFldVal = exprFactory.newLiteralParameter(sqlStmt, idMapping, sm.getObject(), "ID");
        sqlStmt.whereAnd(sqlFldExpr.eq(sqlFldVal), true);

        // Perform the query
        try
        {
            ManagedConnection mconn = storeMgr.getConnection(om);
            SQLController sqlControl = storeMgr.getSQLController();
            if (om.getSerializeReadForClass(primaryClass.getName()))
            {
                sqlStmt.addExtension("lock-for-update", true);
            }

            try
            {
                PreparedStatement ps = SQLStatementHelper.getPreparedStatementForSQLStatement(sqlStmt, om, 
                    mconn, null, null);
                String statement = sqlStmt.getSelectStatement().toSQL();
                try
                {
                    ResultSet rs = sqlControl.executeStatementQuery(mconn, statement, ps);
                    try
                    {
                        if (rs != null)
                        {
                            while (rs.next())
                            {
                                DiscriminatorMetaData dismd = discrimMapping.getDatastoreContainer().getDiscriminatorMetaData();
                                className = RDBMSQueryUtils.getClassNameFromDiscriminatorResultSetRow(
                                    discrimMapping, dismd, rs, om);
                            }
                        }
                    }
                    finally
                    {
                        rs.close();
                    }
                }
                finally
                {
                    sqlControl.closeStatement(mconn, ps);
                }
            }
            finally
            {
                mconn.release();
            }
        }
        catch (SQLException sqe)
        {
            NucleusLogger.DATASTORE.error(sqe);
            throw new NucleusDataStoreException(sqe.toString());
        }

        return className;
    }
}