/**********************************************************************
Copyright (c) 2007 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms;

import java.lang.reflect.InvocationTargetException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Map;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;
import javax.sql.XAConnection;
import javax.sql.XADataSource;
import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.OMFContext;
import org.datanucleus.PersistenceConfiguration;
import org.datanucleus.Transaction;
import org.datanucleus.UserTransaction;
import org.datanucleus.exceptions.ClassNotResolvedException;
import org.datanucleus.exceptions.ConnectionFactoryNotFoundException;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.exceptions.UnsupportedConnectionFactoryException;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.connection.AbstractConnectionFactory;
import org.datanucleus.store.connection.AbstractManagedConnection;
import org.datanucleus.store.connection.ConnectionFactory;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.rdbms.adapter.RDBMSAdapter;
import org.datanucleus.store.rdbms.datasource.DataNucleusDataSourceFactory;
import org.datanucleus.transaction.TransactionUtils;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * ConnectionFactory for RDBMS datastores.
 * Each instance is a factory of Transactional connection or NonTransactional connection.
 */
public class ConnectionFactoryImpl extends AbstractConnectionFactory
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER_RDBMS = Localiser.getInstance("org.datanucleus.store.rdbms.Localisation",
        RDBMSStoreManager.class.getClassLoader());

    /** datasources. */
    Object[] dataSource;

    String poolingType = null;

    /**
     * Constructor.
     * @param omfContext The OMF context
     * @param resourceType either tx or nontx
     */
    public ConnectionFactoryImpl(OMFContext omfContext, String resourceType)
    {
        super(omfContext, resourceType);

        PersistenceConfiguration config = omfContext.getPersistenceConfiguration();
        if (resourceType.equals("tx"))
        {
            initDataSourceTx(config);
        }
        else
        {
            initDataSourceNonTx(config);
        }
    }

    private void initDataSourceTx(PersistenceConfiguration config)
    {
        String configuredResourceTypeProperty = this.omfContext.getPersistenceConfiguration().getStringProperty(
            DATANUCLEUS_CONNECTION_RESOURCE_TYPE);
        if (configuredResourceTypeProperty!=null)
        {
            options.put(ConnectionFactory.RESOURCE_TYPE_OPTION, configuredResourceTypeProperty);
        }

        StoreManager storeMgr = omfContext.getStoreManager();

        // Set the transactional DataSource using connection factory (1)
        Object connectionFactory = storeMgr.getConnectionFactory();
        String connectionFactoryName = storeMgr.getConnectionFactoryName();
        if (connectionFactory != null)
        {
            if (!(connectionFactory instanceof DataSource) && !(connectionFactory instanceof XADataSource))
            {
                throw new UnsupportedConnectionFactoryException(connectionFactory);
            }
            dataSource = new DataSource[1];
            dataSource[0] = connectionFactory;
        }
        else if (connectionFactoryName != null)
        {
            String[] connectionFactoryNames = StringUtils.split(connectionFactoryName, ",");
            dataSource = new DataSource[connectionFactoryNames.length];
            for (int i=0; i<connectionFactoryNames.length; i++)
            {
                dataSource[i] = lookupDataSource(connectionFactoryNames[i]);
            }
        }
        else
        {
            dataSource = new DataSource[1];
            String poolingType = getPoolingType();

            // User has requested internal database connection pooling so check the registered plugins
            try
            {
                // Create the DataSource to be used
                DataNucleusDataSourceFactory dataSourceFactory =
                    (DataNucleusDataSourceFactory)omfContext.getPluginManager().createExecutableExtension(
                        "org.datanucleus.store.rdbms.datasource", "name", poolingType, "class-name", null, null);
                if (dataSourceFactory == null)
                {
                    // User has specified a pool plugin that has not registered
                    throw new NucleusUserException(LOCALISER_RDBMS.msg("047003", 
                        poolingType)).setFatal();
                }

                // Create the DataNucleusDataSourceFactory
                dataSource[0] = dataSourceFactory.makePooledDataSource(omfContext);
                if (NucleusLogger.CONNECTION.isDebugEnabled())
                {
                    NucleusLogger.CONNECTION.debug(LOCALISER_RDBMS.msg("047008", "transactional", poolingType));
                }
            }
            catch (ClassNotFoundException cnfe)
            {
                throw new NucleusUserException(LOCALISER_RDBMS.msg("047003", poolingType),
                    cnfe).setFatal();
            }
            catch (Exception e)
            {
                if (e instanceof InvocationTargetException)
                {
                    InvocationTargetException ite = (InvocationTargetException)e;
                    throw new NucleusException(LOCALISER_RDBMS.msg("047004", poolingType, 
                        ite.getTargetException().getMessage()), ite.getTargetException()).setFatal();
                }
                else
                {
                    throw new NucleusException(LOCALISER_RDBMS.msg("047004", poolingType, 
                        e.getMessage()),e).setFatal();
                }
            }
        }
    }
    
    private void initDataSourceNonTx(PersistenceConfiguration config)
    {
        String configuredResourceTypeProperty = this.omfContext.getPersistenceConfiguration().getStringProperty(
            DATANUCLEUS_CONNECTION2_RESOURCE_TYPE);
        if (configuredResourceTypeProperty!=null)
        {
            options.put(ConnectionFactory.RESOURCE_TYPE_OPTION, configuredResourceTypeProperty);
        }

        StoreManager storeMgr = omfContext.getStoreManager();

        // Set the transactional DataSource using connection factory (1)
        Object connectionFactory = storeMgr.getConnectionFactory2();
        String connectionFactoryName = storeMgr.getConnectionFactory2Name();
        if (connectionFactory != null)
        {
            if (!(connectionFactory instanceof DataSource) && !(connectionFactory instanceof XADataSource))
            {
                throw new UnsupportedConnectionFactoryException(connectionFactory);
            }
            dataSource = new DataSource[1];
            dataSource[0] = connectionFactory;
        }
        else if (connectionFactoryName != null)
        {
            String[] connectionFactoryNames = StringUtils.split(connectionFactoryName, ",");
            dataSource = new DataSource[connectionFactoryNames.length];
            for (int i=0; i<connectionFactoryNames.length; i++)
            {
                dataSource[i] = lookupDataSource(connectionFactoryNames[i]);
            }
        }
        else if (omfContext.getStoreManager().getConnectionURL() != null)
        {
            dataSource = new DataSource[1];
            String poolingType = getPoolingType();

            // User has requested internal database connection pooling so check the registered plugins
            try
            {
                // Create the DataSource to be used
                DataNucleusDataSourceFactory dataSourceFactory =
                    (DataNucleusDataSourceFactory)omfContext.getPluginManager().createExecutableExtension(
                        "org.datanucleus.store.rdbms.datasource", "name", poolingType, "class-name", null, null);
                if (dataSourceFactory == null)
                {
                    // User has specified a pool plugin that has not registered
                    throw new NucleusUserException(LOCALISER_RDBMS.msg("047003", poolingType)).setFatal();
                }

                // Create the DataNucleusDataSourceFactory
                dataSource[0] = dataSourceFactory.makePooledDataSource(omfContext);
                if (NucleusLogger.CONNECTION.isDebugEnabled())
                {
                    NucleusLogger.CONNECTION.debug(LOCALISER_RDBMS.msg("047008", "nontransactional", poolingType));
                }
            }
            catch (ClassNotFoundException cnfe)
            {
                throw new NucleusUserException(LOCALISER_RDBMS.msg("047003", poolingType), cnfe).setFatal();
            }
            catch (Exception e)
            {
                if (e instanceof InvocationTargetException)
                {
                    InvocationTargetException ite = (InvocationTargetException)e;
                    throw new NucleusException(LOCALISER_RDBMS.msg("047004", poolingType, 
                        ite.getTargetException().getMessage()), ite.getTargetException()).setFatal();
                }
                else
                {
                    throw new NucleusException(LOCALISER_RDBMS.msg("047004", poolingType, 
                        e.getMessage()),e).setFatal();
                }
            }
        }
        else
        {
            //defaults to the transactional datasource
            initDataSourceTx(config);
        }        
    }

    /**
     * Accessor for the pooling type (if any).
     * If set to a value will use that, otherwise (if unset) will check if any supported
     * pooling capabilities are present and available. Currently checks DBCP, C3P0, Proxool.
     * @return Pooling type to use (name of a pool type, or "default")
     */
    public String getPoolingType()
    {
        if (poolingType != null)
        {
            return poolingType;
        }

        PersistenceConfiguration config = omfContext.getPersistenceConfiguration();
        poolingType = config.getStringProperty("datanucleus.connectionPoolingType");
        ClassLoaderResolver clr = omfContext.getClassLoaderResolver(null);
        if (poolingType == null)
        {
            // Check if DBCP available
            try
            {
                // Need datanucleus-connectionpool, commons-dbcp, commons-pool, commons-collections
                clr.classForName("org.datanucleus.store.rdbms.datasource.dbcp.DBCPDataSourceFactory");
                clr.classForName("org.apache.commons.pool.ObjectPool");
                clr.classForName("org.apache.commons.dbcp.ConnectionFactory");
                poolingType = "DBCP";
            }
            catch (ClassNotResolvedException cnre)
            {
                // DBCP not available
            }
        }
        if (poolingType == null)
        {
            // Check if C3P0 is available
            try
            {
                // Need datanucleus-connectionpool, c3p0
                clr.classForName("org.datanucleus.store.rdbms.datasource.c3p0.C3P0DataSourceFactory");
                clr.classForName("com.mchange.v2.c3p0.ComboPooledDataSource");
                poolingType = "C3P0";
            }
            catch (ClassNotResolvedException cnre)
            {
                // C3P0 not available
            }
        }
        if (poolingType == null)
        {
            // Check if Proxool is available
            try
            {
                // Need datanucleus-connectionpool, proxool, commons-logging
                clr.classForName("org.datanucleus.store.rdbms.datasource.proxool.ProxoolDataSourceFactory");
                clr.classForName("org.logicalcobwebs.proxool.ProxoolDriver");
                clr.classForName("org.apache.commons.logging.Log");
                poolingType = "Proxool";
            }
            catch (ClassNotResolvedException cnre)
            {
                // Proxool not available
            }
        }
        if (poolingType == null || poolingType.equalsIgnoreCase("None"))
        {
            // Fallback to none ("default")
            poolingType = "default";
        }
        return poolingType;
    }

    /**
     * Method to return a connection (either already existing from cache, or newly allocated) enlisting
     * it in any transaction.
     * @param poolKey the object that is bound the connection during its lifecycle (or null)
     * @param options Options for when creating the connection
     * @return The ManagedConnection
     */
    public ManagedConnection getConnection(Object poolKey, org.datanucleus.Transaction transaction, Map options)
    {
        ManagedConnection mconn = super.getConnection(poolKey, transaction, options);
        ((ManagedConnectionImpl)mconn).incrementUseCount();
        return mconn;
    }

    /**
     * Method to create a new ManagedConnection.
     * @param poolKey the object that is bound the connection during its lifecycle (if any)
     * @param transactionOptions Transaction options
     * @return The ManagedConnection
     */
    public ManagedConnection createManagedConnection(Object poolKey, Map transactionOptions)
    {
        return new ManagedConnectionImpl(transactionOptions);
    }

    class ManagedConnectionImpl extends AbstractManagedConnection
    {
        int isolation;

        /** Count on the number of outstanding uses of this connection. Incremented on get. Decremented on close. */
        int useCount = 0;

        ManagedConnectionImpl(Map transactionOptions)
        {
            if (transactionOptions != null && transactionOptions.get(Transaction.TRANSACTION_ISOLATION_OPTION) != null)
            {
                isolation = ((Number) transactionOptions.get(Transaction.TRANSACTION_ISOLATION_OPTION)).intValue();
            }
            else
            {
                isolation = TransactionUtils.getTransactionIsolationLevelForName(
                    omfContext.getPersistenceConfiguration().getStringProperty("datanucleus.transactionIsolation"));
            }
        }

        void incrementUseCount()
        {
            useCount = useCount + 1;
        }

        /**
         * Release this connection.
         * Releasing this connection will allow this managed connection to be used one or more times
         * inside the same transaction. If this managed connection is managed by a transaction manager, 
         * release is a no-op, otherwise the physical connection is closed 
         */
        public void release()
        {
            if (!managed)
            {
                useCount = useCount -1;
                if (useCount == 0)
                {
                    // Close if this is the last use of the connection
                    close();
                }
            }
        }

        /**
         * Obtain a XAResource which can be enlisted in a transaction
         */
        public XAResource getXAResource()
        {
            if (getConnection() instanceof Connection)
            {
                return new EmulatedXAResource((Connection)getConnection());
            }
            else
            {
                try
                {
                    return ((XAConnection)getConnection()).getXAResource();
                }
                catch (SQLException e)
                {
                    throw new NucleusDataStoreException(e.getMessage(),e);
                }
            }                        
        }

        /**
         * Create a connection to the resource
         */
        public Object getConnection()
        {
            if (this.conn == null)
            {
                Connection cnx = null;
                try
                {
                    RDBMSStoreManager storeMgr = (RDBMSStoreManager)omfContext.getStoreManager();
                    if (storeMgr != null && storeMgr.getDatastoreAdapter() != null)
                    {
                        // Create Connection following DatastoreAdapter capabilities
                        RDBMSAdapter rdba = (RDBMSAdapter)storeMgr.getDatastoreAdapter();
                        int reqdIsolationLevel = isolation;
                        if (rdba.getRequiredTransactionIsolationLevel() >= 0)
                        {
                            // Override with the adapters required isolation level
                            reqdIsolationLevel = rdba.getRequiredTransactionIsolationLevel();
                        }

                        DataSource[] ds = (DataSource[])dataSource;
                        cnx = new ConnectionProviderPriorityList().getConnection(ds);
                        boolean succeeded = false;
                        try
                        {
                            if (reqdIsolationLevel == UserTransaction.TRANSACTION_NONE)
                            {
                                if (!cnx.getAutoCommit())
                                {
                                    NucleusLogger.CONNECTION.debug("Setting autocommit=true to connection: "+cnx.toString()); 
                                    cnx.setAutoCommit(true);
                                }
                            }
                            else
                            {
                                if (cnx.getAutoCommit())
                                {
                                    NucleusLogger.CONNECTION.debug("Setting autocommit=false to connection: "+cnx.toString()); 
                                    cnx.setAutoCommit(false);
                                }
                                if (rdba.supportsTransactionIsolation(reqdIsolationLevel))
                                {
                                    int currentIsolationLevel = cnx.getTransactionIsolation();
                                    if (currentIsolationLevel != reqdIsolationLevel)
                                    {
                                        NucleusLogger.CONNECTION.debug("Setting transaction isolation "+TransactionUtils.getNameForTransactionIsolationLevel(reqdIsolationLevel)+" to connection: "+cnx.toString()); 
                                        cnx.setTransactionIsolation(reqdIsolationLevel);
                                    }
                                }
                                else
                                {
                                    NucleusLogger.DATASTORE.warn(LOCALISER_RDBMS.msg(
                                        "051008", reqdIsolationLevel));
                                }
                            }

                            if (NucleusLogger.CONNECTION.isDebugEnabled())
                            {
                                NucleusLogger.CONNECTION.debug(LOCALISER_RDBMS.msg("052002",
                                    cnx.toString(), 
                                    TransactionUtils.getNameForTransactionIsolationLevel(reqdIsolationLevel)));
                            }

                            if (reqdIsolationLevel != isolation && isolation == UserTransaction.TRANSACTION_NONE)
                            {
                                // User asked for a level that implies auto-commit so make sure it has that
                                if (!cnx.getAutoCommit())
                                {
                                    NucleusLogger.CONNECTION.debug("Setting autocommit=true to connection: "+cnx.toString()); 
                                    cnx.setAutoCommit(true);
                                }
                            }

                            succeeded = true;
                        }
                        catch (SQLException e)
                        {
                            throw new NucleusDataStoreException(e.getMessage(),e);
                        }                        
                        finally
                        {
                            if (!succeeded)
                            {
                                try
                                {
                                    cnx.close();
                                }
                                catch (SQLException e)
                                {
                                }

                                if (NucleusLogger.CONNECTION.isDebugEnabled())
                                {
                                    String cnxStr = cnx.toString();
                                    NucleusLogger.CONNECTION.debug(LOCALISER_RDBMS.msg("052003", cnxStr));
                                }
                            }
                        }
                    }
                    else
                    {
                        // Create basic Connection since no DatastoreAdapter created yet
                        cnx = ((DataSource)dataSource[0]).getConnection();
                        if (cnx == null)
                        {
                            String msg = LOCALISER_RDBMS.msg("052000", dataSource[0]);
                            NucleusLogger.CONNECTION.error(msg);
                            throw new NucleusDataStoreException(msg);
                        }
                        if (NucleusLogger.CONNECTION.isDebugEnabled())
                        {
                            NucleusLogger.CONNECTION.debug(LOCALISER_RDBMS.msg("052001", cnx.toString()));
                        }
                    }
                }
                catch (SQLException e)
                {
                    throw new NucleusDataStoreException(e.getMessage(),e);
                }

                this.conn = cnx;
            }
            return this.conn;
        }

        /**
         * Close the connection
         */
        public void close()
        {
            for (int i=0; i<listeners.size(); i++)
            {
                listeners.get(i).managedConnectionPreClose();
            }
            Connection conn = null;
            if (this.conn != null && this.conn instanceof Connection)
            {
                conn = (Connection) this.conn;
            }
            else if (this.conn != null && this.conn instanceof XAConnection)
            {
                try
                {
                    conn = ((XAConnection) this.conn).getConnection();
                }
                catch (SQLException e)
                {
                    throw new NucleusDataStoreException(e.getMessage(), e);
                }
            }

            if (conn != null)
            {
                try
                {
                    // if this connection is not enlisted in a TransactionManager (such as the internal TM, 
                    // or an external JTA container) and autocommit is not enabled at connection, we commit 
                    // before closing. Usually needed by NonTransactionalRead/NonTransactionalWrite modes, 
                    // or connections used to obtain Object IDs  
                    if (!managed)
                    {
                        if (!conn.isClosed() && !conn.getAutoCommit())
                        {
                            conn.commit();
                            if (NucleusLogger.CONNECTION.isDebugEnabled())
                            {
                                NucleusLogger.CONNECTION.debug(LOCALISER_RDBMS.msg("052005", conn.toString()));
                            }
                        }
                    }
                    if (!conn.isClosed())
                    {
                        String connStr = conn.toString();
                        conn.close();
                        if (NucleusLogger.CONNECTION.isDebugEnabled())
                        {
                            NucleusLogger.CONNECTION.debug(LOCALISER_RDBMS.msg("052003", connStr));
                        }
                    }
                    else
                    {
                        if (NucleusLogger.CONNECTION.isDebugEnabled())
                        {
                            NucleusLogger.CONNECTION.debug(LOCALISER_RDBMS.msg("052004", conn.toString()));
                        }
                    }
                }
                catch (SQLException e)
                {
                    throw new NucleusDataStoreException(e.getMessage(),e);
                }
            }
            try
            {
                for (int i=0; i<listeners.size(); i++)
                {
                    listeners.get(i).managedConnectionPostClose();
                }
            }
            finally
            {
                listeners.clear();
            }
            this.conn = null;
        }
    }
    
    /**
     * Emulate the two phase protocol for non XA 
     */
    class EmulatedXAResource implements XAResource
    {
        Connection conn;
        
        EmulatedXAResource(Connection conn)
        {
            this.conn = conn;
        }
        public void commit(Xid xid, boolean flags) throws XAException
        {
            NucleusLogger.CONNECTION.debug("Managed connection "+this.toString()+" is committing for transaction "+xid.toString()+" with flags "+flags);
            
            try
            {
                conn.commit();
                NucleusLogger.CONNECTION.debug("Managed connection "+this.toString()+" committed connection for transaction "+xid.toString()+" with flags "+flags);
            }
            catch (SQLException e)
            {
                NucleusLogger.CONNECTION.debug("Managed connection "+this.toString()+" failed to commit connection for transaction "+xid.toString()+" with flags "+flags);
                XAException xe = new XAException(StringUtils.getStringFromStackTrace(e));
                xe.initCause(e);
                throw xe;
            }
        }

        public void end(Xid xid, int flags) throws XAException
        {
            NucleusLogger.CONNECTION.debug("Managed connection "+this.toString()+" is ending for transaction "+xid.toString()+" with flags "+flags);
            //ignore
        }

        public void forget(Xid arg0) throws XAException
        {
            //ignore
        }

        public int getTransactionTimeout() throws XAException
        {
            return 0;
        }

        public boolean isSameRM(XAResource xares) throws XAException
        {
            return (this == xares);
        }

        public int prepare(Xid xid) throws XAException
        {
            NucleusLogger.CONNECTION.debug("Managed connection "+this.toString()+" is preparing for transaction "+xid.toString());
            
            return 0;
        }

        public Xid[] recover(int flags) throws XAException
        {
            throw new XAException("Unsupported operation");
        }

        public void rollback(Xid xid) throws XAException
        {
            NucleusLogger.CONNECTION.debug("Managed connection "+this.toString()+" is rolling back for transaction "+xid.toString());
            try
            {
                conn.rollback();
                NucleusLogger.CONNECTION.debug("Managed connection "+this.toString()+" rolled back connection for transaction "+xid.toString());
            }
            catch (SQLException e)
            {
                NucleusLogger.CONNECTION.debug("Managed connection "+this.toString()+" failed to rollback connection for transaction "+xid.toString());
                XAException xe = new XAException(StringUtils.getStringFromStackTrace(e));
                xe.initCause(e);
                throw xe;
            }
        }

        public boolean setTransactionTimeout(int arg0) throws XAException
        {
            return false;
        }

        public void start(Xid xid, int flags) throws XAException
        {
            NucleusLogger.CONNECTION.debug("Managed connection "+this.toString()+" is starting for transaction "+xid.toString()+" with flags "+flags);
            //ignore
        }        
    }
    
    /**
     * Looks up a DataSource object in JNDI. This only permits lookup for DataSources locally
     * For remote DataSources usage, configure the PersistenceConfiguration as an object (ConnectionFactory), instead of 
     * name (ConnectionFactoryName)
     * @param name The JNDI name of the DataSource.
     * @return  The DataSource object.
     * @exception ConnectionFactoryNotFoundException If a JNDI lookup failure occurs.
     * @exception UnsupportedConnectionFactoryException If the object is not a javax.sql.DataSource.
     */
    private Object lookupDataSource(String name)
    {
        Object obj;

        try
        {
            obj = new InitialContext().lookup(name);
        }
        catch (NamingException e)
        {
            throw new ConnectionFactoryNotFoundException(name, e);
        }

        if (!(obj instanceof DataSource) && !(obj instanceof XADataSource))
        {
            throw new UnsupportedConnectionFactoryException(obj);
        }

        return obj;
    }
}