/*
 * Copyright (C) 2004-2014 Volker Bergmann (volker.bergmann@bergmann-it.de).
 * All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.databene.commons.accessor;

import org.databene.commons.Accessor;
import org.databene.commons.BeanUtil;
import org.databene.commons.Composite;
import org.databene.commons.ConfigurationError;
import org.databene.commons.Context;
import org.databene.commons.Escalator;
import org.databene.commons.LoggerEscalator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Map;

/**
 * Get values from Maps, Contexts, Composites and JavaBeans.<br/>
 * <br/>
 * Created: 12.06.2007 18:36:11
 * @author Volker Bergmann
 */
public class FeatureAccessor<C, V> implements Accessor<C, V> {
    
    private static Logger LOGGER = LoggerFactory.getLogger(FeatureAccessor.class);
    
    private static Escalator escalator = new LoggerEscalator();

    private String featureName;
    
    // constructors ----------------------------------------------------------------------------------------------------

    public FeatureAccessor(String featureName) {
        this(featureName, true);
    }
    
    public FeatureAccessor(String featureName, boolean strict) {
        LOGGER.debug("FeatureAccessor({}, {})", featureName, strict);
        this.featureName = featureName;
    }
    
	public String getFeatureName() {
		return featureName;
	}

	public void setFeatureName(String featureName) {
		this.featureName = featureName;
	}

    // Accessor interface implementation -------------------------------------------------------------------------------

	@Override
	@SuppressWarnings("unchecked")
    public V getValue(C target) {
        return (V) getValue(target, featureName);
    }

    // static convenience methods --------------------------------------------------------------------------------------

    public static Object getValue(Object target, String featureName) {
        LOGGER.debug("getValue({}, {})", target, featureName);
        return getValue(target, featureName, true);
    }

    @SuppressWarnings("unchecked")
    public static Object getValue(Object target, String featureName, boolean required) {
        if (target == null)
            return null;
        else if (target instanceof Map)
            return ((Map<String, Object>) target).get(featureName);
        else if (target instanceof Context)
            return ((Context) target).get(featureName);
        else if (target instanceof Composite)
            return ((Composite) target).getComponent(featureName);
        else {
        	// try generic get(String) method
        	Method genericGetMethod = BeanUtil.findMethod(target.getClass(), "get", String.class);
        	if (genericGetMethod != null)
        		return BeanUtil.invoke(target, genericGetMethod, new Object[] { featureName });
        	// try JavaBean property
            PropertyDescriptor propertyDescriptor = BeanUtil.getPropertyDescriptor(target.getClass(), featureName);
            if (propertyDescriptor != null) {
                try {
                    return propertyDescriptor.getReadMethod().invoke(target);
                } catch (Exception e) {
                    throw new ConfigurationError("Unable to access feature '" + featureName + "'", e);
                }
            } else {
            	Class<?> type = ((target instanceof Class) ? (Class<?>) target : target.getClass());
            	Field field = BeanUtil.getField(type, featureName);
            	if (field != null)
            		return BeanUtil.getFieldValue(field, target, false);
            }
        }
        // the feature has not been identified, yet - escalate or raise an exception
        if (required)
            throw new UnsupportedOperationException(
                    target.getClass() + " does not support a feature '" + featureName + "'");
        else
            escalator.escalate("Feature '" + featureName + "' not found in object " + target, FeatureAccessor.class, null);
        return null;
    }
    
    // java.lang.Object overrides --------------------------------------------------------------------------------------

    @Override
    public String toString() {
        return getClass().getSimpleName() + '[' + featureName + ']';
    }
}
