package org.danilopianini.gradle.git.hooks

import org.gradle.api.initialization.Settings
import java.io.File
import java.io.Serializable

/**
 * DSL entry point, to be applied to [settings].gradle.kts.
 */
open class GitHooksExtension(val settings: Settings) : Serializable {

    private var hooks: Map<String, String> = emptyMap()

    /**
     * The git repository root. If unset, it will be searched recursively from the project root towards the
     * filesystem root.
     */
    var path: File? = null
        get() =
            field ?: requireNotNull(generateSequence(settings.settingsDir) { it.parentFile }.find { it.isGitRoot() }) {
                "No git root could be found in ${settings.settingsDir.absolutePath} or any of its parent directories"
            }

    private inline fun <H : ScriptContext> hook(context: H, configuration: H.() -> Unit) {
        require(!hooks.containsKey(context.name)) {
            "it looks like the hook ${context.name} is being defined twice"
        }
        hooks = hooks + (context.name to context.apply(configuration).script)
    }

    /**
     * Defines a new hook with an arbitrary name.
     */
    fun hook(hookName: String, configuration: ScriptContext.() -> Unit) =
        hook(CommonScriptContext(hookName), configuration)

    /**
     * Pre-commit hook.
     */
    fun preCommit(configuration: ScriptContext.() -> Unit) = hook("pre-commit", configuration)

    /**
     * Commit-msg hook.
     */
    fun commitMsg(configuration: CommitMsgScriptContext.() -> Unit): Unit =
        hook(CommitMsgScriptContext(), configuration)

    /**
     * To be called to force the hook creation in case of necessity.
     * If passed `true`, overwrites in case the script is already present and different than expected.
     */
    fun createHooks(overwriteExisting: Boolean = false) {
        val root = requireNotNull(path?.takeIf { it.isGitRoot() }) {
            "${path?.absolutePath} is not a valid git root"
        }
        hooks.forEach { (name, script) ->
            val hook = File(root.absolutePath, "/.git/hooks/$name")
            if (!hook.exists()) {
                require(hook.createNewFile()) { "Cannot create file ${hook.absolutePath}" }
                hook.writeScript(script)
            } else {
                val oldScript = hook.readText()
                if (oldScript != script) {
                    println(
                        """
                        |The hook $name exists, but its content differs from the one generated by the git-hooks plugin.
                        |
                        |Original content:
                        ${oldScript.withMargins()}
                        |
                        |New content:
                        ${script.withMargins()}
                        |
                        |If you want to overwrite this file on every change add true to the createHooks(true) 
                        |method in your settings.gradle.kts
                        """.trimMargin().lines().joinToString(separator = "\n") { "WARNING: $it" }
                    )
                    if (overwriteExisting) {
                        println("WARNING: Overwriting git hook $name")
                        hook.writeScript(script)
                    }
                }
            }
        }
    }

    companion object {
        private const val serialVersionUID = 1L

        /**
         * Extension name.
         */
        const val name: String = "gitHooks"

        private fun String.withMargins() = lines().joinToString(separator = "\n|", prefix = "|")

        private fun File.writeScript(script: String) {
            writeText(script)
            setExecutable(true)
        }

        private fun File.isGitRoot(): Boolean = listFiles()
            ?.any { folder ->
                folder.isDirectory &&
                    folder.name == ".git" &&
                    folder.listFiles()?.any { it.isDirectory && it.name == "hooks" } ?: false
            }
            ?: false
    }
}
