/**
 * Copyright 2008 Bluestem Software LLC.  All Rights Reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package org.bluestemsoftware.open.eoa.system.plugin.release;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.ArtifactUtils;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.artifact.InvalidDependencyVersionException;

/**
 * Iterates over projects in reactor and throws an exception if any snapshot dependencies exist.
 * 
 * @aggregator
 * @goal snapshots
 */
public class ReleaseSnaphsotsMojo extends AbstractMojo {
    
    /**
     * Component used to create artifacts
     *
     * @component
     */
    private ArtifactFactory artifactFactory;
    
    /**
     * @parameter expression="${reactorProjects}"
     * @required
     * @readonly
     */
    private List<?> reactorProjects;
    
    /**
     * List of dependencies to exclude from repository.
     * 
     * @parameter
     */
    private Exclude[] check = new Exclude[]{}; 

    public void execute() throws MojoExecutionException {
        
        Set<String> excludes = new HashSet<String>();
        for (Exclude exclude : check) {
            String groupID = exclude.getGroupId();
            String artifactID = exclude.getArtifactId();
            String key = ArtifactUtils.versionlessKey(groupID, artifactID);
            excludes.add(key);
        }
        
        getLog().info("checking projects for snapshot dependencies ...");
        
        for (Iterator<?> i = reactorProjects.iterator(); i.hasNext();) {
            MavenProject project = (MavenProject)i.next();
            String groupID = project.getGroupId();
            String artifactID = project.getArtifactId();
            String key = ArtifactUtils.versionlessKey(groupID, artifactID);
            if (excludes.contains(key)) {
                getLog().debug("excluding project " + artifactID + " from snapshot dependency check");
            } else {
                getLog().debug("checking project " + artifactID + " for snapshot dependencies");
                checkProject(project);
            }
        }
        
    }

    private void checkProject(MavenProject project) throws MojoExecutionException {

        Map<?,?> artifactMap = ArtifactUtils.artifactMapByVersionlessId(project.getArtifacts());

        Set<Artifact> snapshotDependencies = new HashSet<Artifact> ();
        Set<Artifact>  snapshotReportDependencies = new HashSet<Artifact> ();
        Set<Artifact>  snapshotExtensionsDependencies = new HashSet<Artifact> ();
        Set<Artifact>  snapshotPluginDependencies = new HashSet<Artifact> ();

        if (project.getParentArtifact() != null) {
            if (checkArtifact(project.getParentArtifact(), artifactMap)) {
                snapshotDependencies.add(project.getParentArtifact());
            }
        }

        try {
            Set<?> dependencyArtifacts = project.createArtifacts(artifactFactory, null, null);
            for (Iterator<?> i = dependencyArtifacts.iterator(); i.hasNext();) {
                Artifact artifact = (Artifact)i.next();
                if (checkArtifact(artifact, artifactMap)) {
                    snapshotDependencies.add(artifact);
                }
            }
        } catch (InvalidDependencyVersionException e) {
            throw new MojoExecutionException("Failed to create dependency artifacts", e);
        }

        for (Iterator<?> i = project.getPluginArtifacts().iterator(); i.hasNext();) {
            Artifact artifact = (Artifact)i.next();
            if (checkArtifact(artifact, artifactMap)) {
                snapshotPluginDependencies.add(artifact);
            }
        }

        for (Iterator<?> i = project.getReportArtifacts().iterator(); i.hasNext();) {
            Artifact artifact = (Artifact)i.next();
            if (checkArtifact(artifact, artifactMap)) {
                snapshotReportDependencies.add(artifact);
            }
        }

        for (Iterator<?> i = project.getExtensionArtifacts().iterator(); i.hasNext();) {
            Artifact artifact = (Artifact)i.next();
            if (checkArtifact(artifact, artifactMap)) {
                snapshotExtensionsDependencies.add(artifact);
            }
        }

        if (!snapshotDependencies.isEmpty()
                || !snapshotReportDependencies.isEmpty()
                || !snapshotExtensionsDependencies.isEmpty()
                || !snapshotPluginDependencies.isEmpty()) {

            if (!snapshotDependencies.isEmpty()
                    || !snapshotReportDependencies.isEmpty()
                    || !snapshotExtensionsDependencies.isEmpty()
                    || !snapshotPluginDependencies.isEmpty()) {
                StringBuffer message = new StringBuffer();

                printSnapshotDependencies(snapshotDependencies, message);
                printSnapshotDependencies(snapshotReportDependencies, message);
                printSnapshotDependencies(snapshotExtensionsDependencies, message);
                printSnapshotDependencies(snapshotPluginDependencies, message);
                message.append("in project '" + project.getName() + "' (" + project.getId() + ")");

                throw new MojoExecutionException("Can't release project due to snapshot dependencies:"
                        + System.getProperty("line.separator")
                        + message);
            }
        }
    }

    private static boolean checkArtifact(Artifact artifact, Map<?,?> artifactMapByVersionlessId) {
        String versionlessId = ArtifactUtils.versionlessKey(artifact);
        Artifact temp = (Artifact)artifactMapByVersionlessId.get(versionlessId);
        if (temp == null) {
            return artifact.isSnapshot();
        } else {
            return temp.isSnapshot();
        }
    }

    @SuppressWarnings("unchecked")
    private StringBuffer printSnapshotDependencies(Set<Artifact> snapshotsSet, StringBuffer message) {
        List<Artifact> snapshotsList = new ArrayList<Artifact> (snapshotsSet);

        Collections.sort(snapshotsList);

        for (Iterator<?> i = snapshotsList.iterator(); i.hasNext();) {
            Artifact artifact = (Artifact)i.next();

            message.append("    ");

            message.append(artifact);

            message.append(System.getProperty("line.separator"));
        }

        return message;
    }

}
