/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tinkerpop.gremlin.driver;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.tinkerpop.gremlin.driver.Client;
import org.apache.tinkerpop.gremlin.driver.Cluster;
import org.apache.tinkerpop.gremlin.driver.Connection;
import org.apache.tinkerpop.gremlin.driver.Host;
import org.apache.tinkerpop.gremlin.driver.ResultSet;
import org.apache.tinkerpop.gremlin.driver.Settings;
import org.apache.tinkerpop.gremlin.driver.exception.ConnectionException;
import org.apache.tinkerpop.gremlin.driver.message.RequestMessage;
import org.apache.tinkerpop.gremlin.util.TimeUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class ConnectionPool {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionPool.class);
    public static final int MIN_POOL_SIZE = 2;
    public static final int MAX_POOL_SIZE = 8;
    public static final int MIN_SIMULTANEOUS_USAGE_PER_CONNECTION = 8;
    public static final int MAX_SIMULTANEOUS_USAGE_PER_CONNECTION = 16;
    public final Host host;
    private final Cluster cluster;
    private final Client client;
    private final List<Connection> connections;
    private final AtomicInteger open;
    private final Set<Connection> bin = new CopyOnWriteArraySet<Connection>();
    private final int minPoolSize;
    private final int maxPoolSize;
    private final int minSimultaneousUsagePerConnection;
    private final int maxSimultaneousUsagePerConnection;
    private final int minInProcess;
    private final String poolLabel;
    private final AtomicInteger scheduledForCreation = new AtomicInteger();
    private final AtomicReference<CompletableFuture<Void>> closeFuture = new AtomicReference();
    private volatile int waiter = 0;
    private final Lock waitLock = new ReentrantLock(true);
    private final Condition hasAvailableConnection = this.waitLock.newCondition();

    public ConnectionPool(Host host, Client client) {
        this(host, client, Optional.empty(), Optional.empty());
    }

    public ConnectionPool(Host host, Client client, Optional<Integer> overrideMinPoolSize, Optional<Integer> overrideMaxPoolSize) {
        this.host = host;
        this.client = client;
        this.cluster = client.cluster;
        this.poolLabel = String.format("Connection Pool {host=%s}", host);
        Settings.ConnectionPoolSettings settings = this.settings();
        this.minPoolSize = overrideMinPoolSize.orElse(settings.minSize);
        this.maxPoolSize = overrideMaxPoolSize.orElse(settings.maxSize);
        this.minSimultaneousUsagePerConnection = settings.minSimultaneousUsagePerConnection;
        this.maxSimultaneousUsagePerConnection = settings.maxSimultaneousUsagePerConnection;
        this.minInProcess = settings.minInProcessPerConnection;
        this.connections = new CopyOnWriteArrayList<Connection>();
        try {
            for (int i = 0; i < this.minPoolSize; ++i) {
                this.connections.add(new Connection(host.getHostUri(), this, settings.maxInProcessPerConnection));
            }
        }
        catch (ConnectionException ce) {
            logger.debug("Could not initialize connections in pool for {} - pool size at {}", (Object)host, (Object)this.connections.size());
            this.considerUnavailable();
        }
        this.open = new AtomicInteger(this.connections.size());
        logger.info("Opening connection pool on {} with core size of {}", (Object)host, (Object)this.minPoolSize);
    }

    public Settings.ConnectionPoolSettings settings() {
        return this.cluster.connectionPoolSettings();
    }

    public Connection borrowConnection(long timeout, TimeUnit unit) throws TimeoutException, ConnectionException {
        int borrowed;
        logger.debug("Borrowing connection from pool on {} - timeout in {} {}", new Object[]{this.host, timeout, unit});
        if (this.isClosed()) {
            throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
        }
        Connection leastUsedConn = this.selectLeastUsed();
        if (this.connections.isEmpty()) {
            logger.debug("Tried to borrow connection but the pool was empty for {} - scheduling pool creation and waiting for connection", (Object)this.host);
            for (int i = 0; i < this.minPoolSize; ++i) {
                if (this.scheduledForCreation.get() >= this.minPoolSize) continue;
                this.scheduledForCreation.incrementAndGet();
                this.newConnection();
            }
            return this.waitForConnection(timeout, unit);
        }
        if (null == leastUsedConn) {
            if (this.isClosed()) {
                throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
            }
            logger.debug("Pool was initialized but a connection could not be selected earlier - waiting for connection on {}", (Object)this.host);
            return this.waitForConnection(timeout, unit);
        }
        int currentPoolSize = this.connections.size();
        if (leastUsedConn.borrowed.get() >= this.maxSimultaneousUsagePerConnection && currentPoolSize < this.maxPoolSize) {
            if (logger.isDebugEnabled()) {
                logger.debug("Least used {} on {} exceeds maxSimultaneousUsagePerConnection but pool size {} < maxPoolSize - consider new connection", new Object[]{leastUsedConn.getConnectionInfo(), this.host, currentPoolSize});
            }
            this.considerNewConnection();
        }
        do {
            borrowed = leastUsedConn.borrowed.get();
            int availableInProcess = leastUsedConn.availableInProcess();
            if (borrowed < this.maxSimultaneousUsagePerConnection || leastUsedConn.availableInProcess() != 0) continue;
            logger.debug("Least used connection selected from pool for {} but borrowed [{}] >= availableInProcess [{}] - wait", new Object[]{this.host, borrowed, availableInProcess});
            return this.waitForConnection(timeout, unit);
        } while (!leastUsedConn.borrowed.compareAndSet(borrowed, borrowed + 1));
        if (logger.isDebugEnabled()) {
            logger.debug("Return least used {} on {}", (Object)leastUsedConn.getConnectionInfo(), (Object)this.host);
        }
        return leastUsedConn;
    }

    public void returnConnection(Connection connection) throws ConnectionException {
        logger.debug("Attempting to return {} on {}", (Object)connection, (Object)this.host);
        if (this.isClosed()) {
            throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
        }
        int borrowed = connection.borrowed.decrementAndGet();
        if (connection.isDead()) {
            logger.debug("Marking {} as dead", (Object)this.host);
            this.considerUnavailable();
        } else {
            if (this.bin.contains(connection) && borrowed == 0) {
                logger.debug("{} is already in the bin and it has no inflight requests so it is safe to close", (Object)connection);
                if (this.bin.remove(connection)) {
                    connection.closeAsync();
                }
                return;
            }
            int poolSize = this.connections.size();
            int availableInProcess = connection.availableInProcess();
            if (poolSize > this.minPoolSize && borrowed <= this.minSimultaneousUsagePerConnection) {
                if (logger.isDebugEnabled()) {
                    logger.debug("On {} pool size of {} > minPoolSize {} and borrowed of {} <= minSimultaneousUsagePerConnection {} so destroy {}", new Object[]{this.host, poolSize, this.minPoolSize, borrowed, this.minSimultaneousUsagePerConnection, connection.getConnectionInfo()});
                }
                this.destroyConnection(connection);
            } else if (availableInProcess < this.minInProcess && this.maxPoolSize > 1) {
                if (logger.isDebugEnabled()) {
                    logger.debug("On {} availableInProcess {} < minInProcess {} so replace {}", new Object[]{this.host, availableInProcess, this.minInProcess, connection.getConnectionInfo()});
                }
                this.replaceConnection(connection);
            } else {
                this.announceAvailableConnection();
            }
        }
    }

    Client getClient() {
        return this.client;
    }

    Cluster getCluster() {
        return this.cluster;
    }

    public boolean isClosed() {
        return this.closeFuture.get() != null;
    }

    public CompletableFuture<Void> closeAsync() {
        logger.info("Signalled closing of connection pool on {} with core size of {}", (Object)this.host, (Object)this.minPoolSize);
        CompletableFuture<Void> future = this.closeFuture.get();
        if (future != null) {
            return future;
        }
        this.announceAllAvailableConnection();
        future = CompletableFuture.allOf(this.killAvailableConnections());
        return this.closeFuture.compareAndSet(null, future) ? future : this.closeFuture.get();
    }

    private CompletableFuture[] killAvailableConnections() {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>(this.connections.size());
        for (Connection connection : this.connections) {
            CompletableFuture<Void> future = connection.closeAsync();
            future.thenRunAsync(this.open::decrementAndGet, this.cluster.executor());
            futures.add(future);
        }
        return futures.toArray(new CompletableFuture[futures.size()]);
    }

    void replaceConnection(Connection connection) {
        logger.debug("Replace {}", (Object)connection);
        this.considerNewConnection();
        this.definitelyDestroyConnection(connection);
    }

    private void considerNewConnection() {
        int inCreation;
        logger.debug("Considering new connection on {} where pool size is {}", (Object)this.host, (Object)this.connections.size());
        do {
            inCreation = this.scheduledForCreation.get();
            logger.debug("There are {} connections scheduled for creation on {}", (Object)inCreation, (Object)this.host);
            if (inCreation < 1) continue;
            return;
        } while (!this.scheduledForCreation.compareAndSet(inCreation, inCreation + 1));
        this.newConnection();
    }

    private void newConnection() {
        this.cluster.executor().submit(() -> {
            this.addConnectionIfUnderMaximum();
            this.scheduledForCreation.decrementAndGet();
            return null;
        });
    }

    private boolean addConnectionIfUnderMaximum() {
        int opened;
        do {
            if ((opened = this.open.get()) < this.maxPoolSize) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened + 1));
        if (this.isClosed()) {
            this.open.decrementAndGet();
            return false;
        }
        try {
            this.connections.add(new Connection(this.host.getHostUri(), this, this.settings().maxInProcessPerConnection));
        }
        catch (ConnectionException ce) {
            logger.debug("Connections were under max, but there was an error creating the connection.", (Throwable)ce);
            this.open.decrementAndGet();
            this.considerUnavailable();
            return false;
        }
        this.announceAvailableConnection();
        return true;
    }

    private boolean destroyConnection(Connection connection) {
        int opened;
        do {
            if ((opened = this.open.get()) > this.minPoolSize) continue;
            return false;
        } while (!this.open.compareAndSet(opened, opened - 1));
        this.definitelyDestroyConnection(connection);
        return true;
    }

    private void definitelyDestroyConnection(Connection connection) {
        if (!this.bin.contains(connection)) {
            this.bin.add(connection);
            this.connections.remove(connection);
            this.open.decrementAndGet();
        }
        if (connection.borrowed.get() == 0 && this.bin.remove(connection)) {
            connection.closeAsync();
            logger.debug("{} destroyed", (Object)connection.getConnectionInfo());
        }
    }

    private Connection waitForConnection(long timeout, TimeUnit unit) throws TimeoutException, ConnectionException {
        long start = System.nanoTime();
        long remaining = timeout;
        long to = timeout;
        do {
            block7: {
                try {
                    this.awaitAvailableConnection(remaining, unit);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    to = 0L;
                }
                if (this.isClosed()) {
                    throw new ConnectionException(this.host.getHostUri(), this.host.getAddress(), "Pool is shutdown");
                }
                Connection leastUsed = this.selectLeastUsed();
                if (leastUsed != null) {
                    int inFlight;
                    do {
                        int availableInProcess;
                        if ((inFlight = leastUsed.borrowed.get()) < (availableInProcess = leastUsed.availableInProcess())) continue;
                        logger.debug("Least used {} on {} has requests borrowed [{}] >= availableInProcess [{}] - may timeout waiting for connection", new Object[]{leastUsed, this.host, inFlight, availableInProcess});
                        break block7;
                    } while (!leastUsed.borrowed.compareAndSet(inFlight, inFlight + 1));
                    if (logger.isDebugEnabled()) {
                        logger.debug("Return least used {} on {} after waiting", (Object)leastUsed.getConnectionInfo(), (Object)this.host);
                    }
                    return leastUsed;
                }
            }
            remaining = to - TimeUtil.timeSince((long)start, (TimeUnit)unit);
            logger.debug("Continue to wait for connection on {} if {} > 0", (Object)this.host, (Object)remaining);
        } while (remaining > 0L);
        logger.debug("Timed-out waiting for connection on {} - possibly unavailable", (Object)this.host);
        this.considerUnavailable();
        throw new TimeoutException();
    }

    private void considerUnavailable() {
        this.host.makeUnavailable(this::tryReconnect);
        this.connections.forEach(this::definitelyDestroyConnection);
        this.cluster.loadBalancingStrategy().onUnavailable(this.host);
    }

    private boolean tryReconnect(Host h) {
        logger.debug("Trying to re-establish connection on {}", (Object)this.host);
        Connection connection = null;
        try {
            connection = this.borrowConnection(this.cluster.connectionPoolSettings().maxWaitForConnection, TimeUnit.MILLISECONDS);
            RequestMessage ping = RequestMessage.build("eval").add("gremlin", "''").create();
            CompletableFuture<ResultSet> f = new CompletableFuture<ResultSet>();
            connection.write(ping, f);
            f.get().all().get();
            this.cluster.loadBalancingStrategy().onAvailable(this.host);
            return true;
        }
        catch (Exception ex) {
            logger.debug("Failed reconnect attempt on {}", (Object)this.host);
            if (connection != null) {
                this.definitelyDestroyConnection(connection);
            }
            return false;
        }
    }

    private void announceAvailableConnection() {
        logger.debug("Announce connection available on {}", (Object)this.host);
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signal();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    private Connection selectLeastUsed() {
        int minInFlight = Integer.MAX_VALUE;
        Connection leastBusy = null;
        for (Connection connection : this.connections) {
            int inFlight = connection.borrowed.get();
            if (connection.isDead() || inFlight >= minInFlight) continue;
            minInFlight = inFlight;
            leastBusy = connection;
        }
        return leastBusy;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void awaitAvailableConnection(long timeout, TimeUnit unit) throws InterruptedException {
        logger.debug("Wait {} {} for an available connection on {} with {}", new Object[]{timeout, unit, this.host, Thread.currentThread()});
        this.waitLock.lock();
        ++this.waiter;
        try {
            this.hasAvailableConnection.await(timeout, unit);
        }
        finally {
            --this.waiter;
            this.waitLock.unlock();
        }
    }

    private void announceAllAvailableConnection() {
        if (this.waiter == 0) {
            return;
        }
        this.waitLock.lock();
        try {
            this.hasAvailableConnection.signalAll();
        }
        finally {
            this.waitLock.unlock();
        }
    }

    public String getPoolInfo() {
        StringBuilder sb = new StringBuilder("ConnectionPool (");
        sb.append(this.host);
        sb.append(") - ");
        this.connections.forEach(c -> {
            sb.append(c);
            sb.append(",");
        });
        return sb.toString().trim();
    }

    public String toString() {
        return this.poolLabel;
    }
}

