/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tez.runtime.library.shuffle.common;

import com.google.common.base.Stopwatch;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.util.concurrent.TimeUnit;
import javax.crypto.SecretKey;
import javax.net.ssl.HttpsURLConnection;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.security.ssl.SSLFactory;
import org.apache.tez.runtime.library.common.security.SecureShuffleUtils;

public class HttpConnection {
    private static final Log LOG = LogFactory.getLog(HttpConnection.class);
    private static final int UNIT_CONNECT_TIMEOUT = 60000;
    private URL url;
    private final String logIdentifier;
    private static SSLFactory sslFactory;
    private HttpURLConnection connection;
    private DataInputStream input;
    private boolean connectionSucceeed;
    private volatile boolean cleanup;
    private final SecretKey jobTokenSecret;
    private String encHash;
    private String msgToEncode;
    private final HttpConnectionParams httpConnParams;
    private final Stopwatch stopWatch;

    public HttpConnection(URL url, HttpConnectionParams connParams, String logIdentifier, SecretKey jobTokenSecret) throws IOException {
        this.logIdentifier = logIdentifier;
        this.jobTokenSecret = jobTokenSecret;
        this.httpConnParams = connParams;
        this.url = url;
        this.stopWatch = new Stopwatch();
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("MapOutput URL :" + url.toString()));
        }
    }

    private void setupConnection() throws IOException {
        this.connection = (HttpURLConnection)this.url.openConnection();
        if (sslFactory != null) {
            try {
                ((HttpsURLConnection)this.connection).setSSLSocketFactory(sslFactory.createSSLSocketFactory());
                ((HttpsURLConnection)this.connection).setHostnameVerifier(sslFactory.getHostnameVerifier());
            }
            catch (GeneralSecurityException ex) {
                throw new IOException(ex);
            }
        }
        this.msgToEncode = SecureShuffleUtils.buildMsgFrom(this.url);
        this.encHash = SecureShuffleUtils.hashFromString(this.msgToEncode, this.jobTokenSecret);
        this.connection.addRequestProperty("UrlHash", this.encHash);
        this.connection.setReadTimeout(this.httpConnParams.readTimeout);
        this.connection.addRequestProperty("name", "mapreduce");
        this.connection.addRequestProperty("version", "1.0.0");
    }

    public boolean connect() throws IOException {
        return this.connect(this.httpConnParams.connectionTimeout);
    }

    public boolean connect(int connectionTimeout) throws IOException {
        this.stopWatch.reset().start();
        if (this.connection == null) {
            this.setupConnection();
        }
        int unit = 0;
        if (connectionTimeout < 0) {
            throw new IOException("Invalid timeout [timeout = " + connectionTimeout + " ms]");
        }
        if (connectionTimeout > 0) {
            unit = Math.min(60000, connectionTimeout);
        }
        this.connection.setConnectTimeout(unit);
        int connectionFailures = 0;
        while (true) {
            try {
                this.connection.connect();
                this.connectionSucceeed = true;
            }
            catch (IOException ioe) {
                if (!this.cleanup) {
                    LOG.info((Object)("Cleanup is set to true. Not attempting to connect again. Last exception was: [" + ioe.getClass().getName() + ", " + ioe.getMessage() + "]"));
                    return false;
                }
                if ((connectionTimeout -= unit) == 0) {
                    throw ioe;
                }
                if (connectionTimeout < unit) {
                    unit = connectionTimeout;
                    this.connection.setConnectTimeout(unit);
                }
                ++connectionFailures;
                continue;
            }
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Time taken to connect to " + this.url.toString() + " " + this.stopWatch.elapsedTime(TimeUnit.MILLISECONDS) + " ms; connectionFailures=" + connectionFailures));
        }
        return true;
    }

    public void validate() throws IOException {
        this.stopWatch.reset().start();
        int rc = this.connection.getResponseCode();
        if (rc != 200) {
            throw new IOException("Got invalid response code " + rc + " from " + this.url + ": " + this.connection.getResponseMessage());
        }
        if (!"mapreduce".equals(this.connection.getHeaderField("name")) || !"1.0.0".equals(this.connection.getHeaderField("version"))) {
            throw new IOException("Incompatible shuffle response version");
        }
        String replyHash = this.connection.getHeaderField("ReplyHash");
        if (replyHash == null) {
            throw new IOException("security validation of TT Map output failed");
        }
        LOG.debug((Object)("url=" + this.msgToEncode + ";encHash=" + this.encHash + ";replyHash=" + replyHash));
        SecureShuffleUtils.verifyReply(replyHash, this.encHash, this.jobTokenSecret);
        LOG.info((Object)("for url=" + this.msgToEncode + " sent hash and receievd reply " + this.stopWatch.elapsedTime(TimeUnit.MILLISECONDS) + " ms"));
    }

    public DataInputStream getInputStream() throws IOException {
        this.stopWatch.reset().start();
        DataInputStream input = null;
        if (this.connectionSucceeed) {
            input = new DataInputStream(new BufferedInputStream(this.connection.getInputStream(), this.httpConnParams.bufferSize));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Time taken to getInputStream (connect) " + this.url + " " + this.stopWatch.elapsedTime(TimeUnit.MILLISECONDS) + " ms"));
        }
        return input;
    }

    public static synchronized void cleanupSSLFactory() {
        if (sslFactory != null) {
            sslFactory.destroy();
            sslFactory = null;
        }
    }

    public void cleanup(boolean disconnect) throws IOException {
        this.cleanup = true;
        this.stopWatch.reset().start();
        try {
            if (this.input != null) {
                LOG.info((Object)("Closing input on " + this.logIdentifier));
                this.input.close();
            }
            if (this.httpConnParams.keepAlive && this.connectionSucceeed) {
                this.readErrorStream(this.connection.getErrorStream());
            }
            if (this.connection != null && (disconnect || !this.httpConnParams.keepAlive)) {
                LOG.info((Object)("Closing connection on " + this.logIdentifier));
                this.connection.disconnect();
            }
        }
        catch (IOException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Exception while shutting down fetcher " + this.logIdentifier), (Throwable)e);
            }
            LOG.info((Object)("Exception while shutting down fetcher " + this.logIdentifier + ": " + e.getMessage()));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Time taken to cleanup connection to " + this.url + " " + this.stopWatch.elapsedTime(TimeUnit.MILLISECONDS) + " ms"));
        }
    }

    private void readErrorStream(InputStream errorStream) {
        if (errorStream == null) {
            return;
        }
        try {
            DataOutputBuffer errorBuffer = new DataOutputBuffer();
            IOUtils.copyBytes((InputStream)errorStream, (OutputStream)errorBuffer, (int)4096);
            IOUtils.closeStream((Closeable)errorBuffer);
            IOUtils.closeStream((Closeable)errorStream);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static class HttpConnectionParamsBuilder {
        private HttpConnectionParams params = new HttpConnectionParams();

        public HttpConnectionParamsBuilder setKeepAlive(boolean keepAlive, int keepAliveMaxConnections) {
            this.params.keepAlive = keepAlive;
            this.params.keepAliveMaxConnections = keepAliveMaxConnections;
            return this;
        }

        public HttpConnectionParamsBuilder setTimeout(int connectionTimeout, int readTimeout) {
            this.params.connectionTimeout = connectionTimeout;
            this.params.readTimeout = readTimeout;
            return this;
        }

        public synchronized HttpConnectionParamsBuilder setSSL(boolean sslEnabled, Configuration conf) {
            this.params.sslShuffle = sslEnabled;
            if (sslEnabled) {
                if (sslFactory == null || sslFactory.getKeystoresFactory().getTrustManagers() == null) {
                    LOG.info((Object)"Initializing SSL factory in HttpConnection");
                    sslFactory = new SSLFactory(SSLFactory.Mode.CLIENT, conf);
                    try {
                        sslFactory.init();
                    }
                    catch (Exception ex) {
                        sslFactory.destroy();
                        sslFactory = null;
                        throw new RuntimeException(ex);
                    }
                }
            } else {
                HttpConnection.cleanupSSLFactory();
            }
            return this;
        }

        public HttpConnectionParamsBuilder setBufferSize(int bufferSize) {
            this.params.bufferSize = bufferSize;
            return this;
        }

        public HttpConnectionParams build() {
            return this.params;
        }
    }

    public static class HttpConnectionParams {
        private boolean keepAlive;
        private int keepAliveMaxConnections;
        private int connectionTimeout;
        private int readTimeout;
        private int bufferSize;
        private boolean sslShuffle;

        public boolean getKeepAlive() {
            return this.keepAlive;
        }

        public int getKeepAliveMaxConnections() {
            return this.keepAliveMaxConnections;
        }

        public int getConnectionTimeout() {
            return this.connectionTimeout;
        }

        public int getReadTimeout() {
            return this.readTimeout;
        }

        public void setReadTimeout(int readTimeout) {
            this.readTimeout = readTimeout;
        }

        public int getBufferSize() {
            return this.bufferSize;
        }

        public boolean isSSLShuffleEnabled() {
            return this.sslShuffle;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("keepAlive=").append(this.keepAlive).append(", ");
            sb.append("keepAliveMaxConnections=").append(this.keepAliveMaxConnections).append(", ");
            sb.append("connectionTimeout=").append(this.connectionTimeout).append(", ");
            sb.append("readTimeout=").append(this.readTimeout).append(", ");
            sb.append("bufferSize=").append(this.bufferSize).append(", ");
            sb.append("sslShuffle=").append(this.sslShuffle);
            return sb.toString();
        }
    }
}

