/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.datastore;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.google.common.io.Files;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.lang.ref.SoftReference;
import java.nio.charset.Charset;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Callable;
import javax.annotation.Nonnull;
import org.apache.commons.io.FileUtils;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.plugins.blob.datastore.DataStoreBlobStore;
import org.apache.jackrabbit.oak.plugins.blob.datastore.InMemoryDataRecord;
import org.apache.jackrabbit.oak.plugins.blob.datastore.TextWriter;
import org.apache.jackrabbit.oak.plugins.index.fulltext.ExtractedText;
import org.apache.jackrabbit.oak.plugins.index.fulltext.PreExtractedTextProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataStoreTextWriter
implements TextWriter,
Closeable,
PreExtractedTextProvider {
    private static final String ERROR_BLOB_FILE = "blobs_error.txt";
    private static final String EMPTY_BLOB_FILE = "blobs_empty.txt";
    private static final Logger log = LoggerFactory.getLogger(DataStoreTextWriter.class);
    private File directory;
    private final SetHolder emptyBlobsHolder;
    private final SetHolder errorBlobsHolder;
    private boolean closed;
    private boolean dataStoreBlobId = true;
    private final boolean readOnlyMode;

    public DataStoreTextWriter(File directory, boolean readOnlyMode) throws IOException {
        if (!directory.exists()) {
            Preconditions.checkArgument((boolean)directory.mkdirs(), (String)"Cannot create directory %s", (Object[])new Object[]{directory.getAbsolutePath()});
        }
        this.directory = directory;
        this.readOnlyMode = readOnlyMode;
        this.emptyBlobsHolder = new SetHolder(this.createLoader(EMPTY_BLOB_FILE), readOnlyMode);
        this.errorBlobsHolder = new SetHolder(this.createLoader(ERROR_BLOB_FILE), readOnlyMode);
        if (!readOnlyMode) {
            log.info("Using {} to store the extracted text content. Empty count {}, Error count {}", new Object[]{directory.getAbsolutePath(), this.getEmptyBlobs().size(), this.getErrorBlobs().size()});
        } else {
            log.info("Using extracted store from {}", (Object)directory.getAbsolutePath());
        }
    }

    @Override
    public ExtractedText getText(String propertyPath, Blob blob) throws IOException {
        String blobId = blob.getContentIdentity();
        if (blobId == null) {
            log.debug("No id found for blob at path {}", (Object)propertyPath);
            return null;
        }
        if (InMemoryDataRecord.isInstance(blobId = this.stripLength(blobId))) {
            log.debug("Pre extraction is not supported for in memory records. Path {}, BlobId {}", (Object)propertyPath, (Object)blobId);
            return null;
        }
        ExtractedText result = null;
        if (this.getEmptyBlobs().contains(blobId)) {
            result = ExtractedText.EMPTY;
        } else if (this.getErrorBlobs().contains(blobId)) {
            result = ExtractedText.ERROR;
        } else {
            File textFile = this.getFile(blobId);
            if (textFile.exists()) {
                String text = Files.toString((File)textFile, (Charset)Charsets.UTF_8);
                result = new ExtractedText(ExtractedText.ExtractionResult.SUCCESS, text);
            }
        }
        if (log.isDebugEnabled()) {
            String extractionResult = result != null ? result.getExtractionResult().toString() : null;
            log.debug("Extraction result for [{}] at path [{}] is [{}]", new Object[]{blobId, propertyPath, extractionResult});
        }
        return result;
    }

    @Override
    public void write(@Nonnull String blobId, @Nonnull String text) throws IOException {
        this.checkIfReadOnlyModeEnabled();
        Preconditions.checkNotNull((Object)blobId, (Object)"BlobId cannot be null");
        Preconditions.checkNotNull((Object)text, (String)"Text passed for [%s] was null", (Object[])new Object[]{blobId});
        File textFile = this.getFile(this.stripLength(blobId));
        DataStoreTextWriter.ensureParentExists(textFile);
        Files.write((CharSequence)text, (File)textFile, (Charset)Charsets.UTF_8);
    }

    @Override
    public synchronized void markEmpty(String blobId) {
        this.checkIfReadOnlyModeEnabled();
        this.getEmptyBlobs().add(this.stripLength(blobId));
    }

    @Override
    public synchronized void markError(String blobId) {
        this.checkIfReadOnlyModeEnabled();
        this.getErrorBlobs().add(this.stripLength(blobId));
    }

    @Override
    public synchronized boolean isProcessed(String blobId) {
        blobId = this.stripLength(blobId);
        if (this.getEmptyBlobs().contains(blobId) || this.getErrorBlobs().contains(blobId)) {
            return true;
        }
        File textFile = this.getFile(blobId);
        return textFile.exists();
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.closed || this.readOnlyMode) {
            return;
        }
        this.writeToFile(EMPTY_BLOB_FILE, this.getEmptyBlobs());
        this.writeToFile(ERROR_BLOB_FILE, this.getErrorBlobs());
        this.closed = true;
    }

    public String toString() {
        return "FileDataStore based text provider";
    }

    SetHolder getEmptyBlobsHolder() {
        return this.emptyBlobsHolder;
    }

    SetHolder getErrorBlobsHolder() {
        return this.errorBlobsHolder;
    }

    private File getFile(String identifier) {
        File file = this.directory;
        file = new File(file, identifier.substring(0, 2));
        file = new File(file, identifier.substring(2, 4));
        file = new File(file, identifier.substring(4, 6));
        return new File(file, identifier);
    }

    private String stripLength(String blobId) {
        if (this.dataStoreBlobId) {
            return DataStoreBlobStore.BlobId.of(blobId).getBlobId();
        }
        return blobId;
    }

    private Set<String> getEmptyBlobs() {
        return this.emptyBlobsHolder.get();
    }

    private Set<String> getErrorBlobs() {
        return this.errorBlobsHolder.get();
    }

    private void checkIfReadOnlyModeEnabled() {
        Preconditions.checkState((!this.readOnlyMode ? 1 : 0) != 0, (Object)"Read only mode enabled");
    }

    private Callable<Set<String>> createLoader(String fileName) {
        final File file = new File(this.directory, fileName);
        return new Callable<Set<String>>(){

            @Override
            public Set<String> call() throws Exception {
                return DataStoreTextWriter.this.loadFromFile(file);
            }

            public String toString() {
                return "Loading state from " + file.getAbsolutePath();
            }
        };
    }

    private Set<String> loadFromFile(File file) throws IOException {
        HashSet result = Sets.newHashSet();
        if (file.exists()) {
            result.addAll(Files.readLines((File)file, (Charset)Charsets.UTF_8));
        }
        return result;
    }

    private void writeToFile(String fileName, Set<String> blobIds) throws IOException {
        if (blobIds.isEmpty()) {
            return;
        }
        File file = new File(this.directory, fileName);
        BufferedWriter bw = Files.newWriter((File)file, (Charset)Charsets.UTF_8);
        for (String id : blobIds) {
            bw.write(id);
            bw.newLine();
        }
        bw.close();
    }

    private static void ensureParentExists(File file) throws IOException {
        if (!file.exists()) {
            File parent = file.getParentFile();
            FileUtils.forceMkdir((File)parent);
        }
    }

    static class SetHolder {
        private final Set<String> state;
        private SoftReference<Set<String>> stateRef;
        private final Callable<Set<String>> loader;
        private int loadCount;

        public SetHolder(Callable<Set<String>> loader, boolean softRef) {
            this.loader = loader;
            this.state = softRef ? null : this.load();
        }

        public Set<String> get() {
            Set<String> result = this.state;
            if (result != null) {
                return result;
            }
            if (this.stateRef != null) {
                result = this.stateRef.get();
            }
            if (result == null) {
                result = this.load();
                this.stateRef = new SoftReference<Set<String>>(result);
            }
            return result;
        }

        public int getLoadCount() {
            return this.loadCount;
        }

        private Set<String> load() {
            try {
                ++this.loadCount;
                return this.loader.call();
            }
            catch (Exception e) {
                log.warn("Error occurred while loading the state via {}", this.loader, (Object)e);
                return Sets.newHashSet();
            }
        }
    }
}

