/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.Collection;
import java.util.Comparator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;
import javax.net.ssl.TrustManager;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.ratis.ContainerCommandRequestMessage;
import org.apache.hadoop.hdds.ratis.RatisHelper;
import org.apache.hadoop.hdds.ratis.conf.RatisClientConfig;
import org.apache.hadoop.hdds.scm.ErrorInjector;
import org.apache.hadoop.hdds.scm.XceiverClientManager;
import org.apache.hadoop.hdds.scm.XceiverClientMetrics;
import org.apache.hadoop.hdds.scm.XceiverClientReply;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.client.ClientTrustManager;
import org.apache.hadoop.hdds.scm.client.HddsClientUtils;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.tracing.TracingUtil;
import org.apache.ratis.client.RaftClient;
import org.apache.ratis.client.api.DataStreamApi;
import org.apache.ratis.grpc.GrpcTlsConfig;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.RaftClientReply;
import org.apache.ratis.protocol.exceptions.GroupMismatchException;
import org.apache.ratis.protocol.exceptions.NotReplicatedException;
import org.apache.ratis.protocol.exceptions.RaftException;
import org.apache.ratis.retry.RetryPolicy;
import org.apache.ratis.rpc.RpcType;
import org.apache.ratis.rpc.SupportedRpcType;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.thirdparty.com.google.protobuf.InvalidProtocolBufferException;
import org.apache.ratis.util.JavaUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class XceiverClientRatis
extends XceiverClientSpi {
    public static final Logger LOG = LoggerFactory.getLogger(XceiverClientRatis.class);
    private final Pipeline pipeline;
    private final RpcType rpcType;
    private final AtomicReference<RaftClient> client = new AtomicReference();
    private final RetryPolicy retryPolicy;
    private final GrpcTlsConfig tlsConfig;
    private final ConfigurationSource ozoneConfiguration;
    private final ConcurrentHashMap<UUID, Long> commitInfoMap;
    private final XceiverClientMetrics metrics = XceiverClientManager.getXceiverClientMetrics();
    private final RaftProtos.ReplicationLevel watchType;
    private final int majority;
    private final ErrorInjector errorInjector;

    private XceiverClientRatis(Pipeline pipeline, RpcType rpcType, RetryPolicy retryPolicy, GrpcTlsConfig tlsConfig, ConfigurationSource configuration, ErrorInjector errorInjector) {
        this.pipeline = pipeline;
        this.majority = pipeline.getReplicationConfig().getRequiredNodes() / 2 + 1;
        this.rpcType = rpcType;
        this.retryPolicy = retryPolicy;
        this.commitInfoMap = new ConcurrentHashMap();
        this.tlsConfig = tlsConfig;
        this.ozoneConfiguration = configuration;
        try {
            this.watchType = RaftProtos.ReplicationLevel.valueOf((String)((RatisClientConfig)configuration.getObject(RatisClientConfig.class)).getWatchType());
        }
        catch (Exception e) {
            throw new IllegalArgumentException(((RatisClientConfig)configuration.getObject(RatisClientConfig.class)).getWatchType() + " is not supported. Currently only ALL_COMMITTED or MAJORITY_COMMITTED are supported");
        }
        if (this.watchType != RaftProtos.ReplicationLevel.ALL_COMMITTED && this.watchType != RaftProtos.ReplicationLevel.MAJORITY_COMMITTED) {
            throw new IllegalArgumentException(this.watchType + " is not supported. Currently only ALL_COMMITTED or MAJORITY_COMMITTED are supported");
        }
        LOG.debug("WatchType {}. Majority {}, ", (Object)this.watchType, (Object)this.majority);
        if (LOG.isTraceEnabled()) {
            LOG.trace("new XceiverClientRatis for pipeline " + pipeline.getId(), new Throwable("TRACE"));
        }
        this.errorInjector = errorInjector;
    }

    public static XceiverClientRatis newXceiverClientRatis(Pipeline pipeline, ConfigurationSource ozoneConf) {
        return XceiverClientRatis.newXceiverClientRatis(pipeline, ozoneConf, null, null);
    }

    public static XceiverClientRatis newXceiverClientRatis(Pipeline pipeline, ConfigurationSource ozoneConf, ClientTrustManager trustManager, ErrorInjector errorInjector) {
        String rpcType = ozoneConf.get("hdds.container.ratis.rpc.type", "GRPC");
        RetryPolicy retryPolicy = RatisHelper.createRetryPolicy((ConfigurationSource)ozoneConf);
        GrpcTlsConfig tlsConfig = RatisHelper.createTlsClientConfig((SecurityConfig)new SecurityConfig(ozoneConf), (TrustManager)trustManager);
        return new XceiverClientRatis(pipeline, (RpcType)SupportedRpcType.valueOfIgnoreCase((String)rpcType), retryPolicy, tlsConfig, ozoneConf, errorInjector);
    }

    private long updateCommitInfosMap(RaftClientReply reply, RaftProtos.ReplicationLevel level) {
        return Optional.ofNullable(reply).filter(RaftClientReply::isSuccess).map(RaftClientReply::getCommitInfos).map(v -> this.updateCommitInfosMap((Collection<RaftProtos.CommitInfoProto>)v, level)).orElse(0L);
    }

    public long updateCommitInfosMap(Collection<RaftProtos.CommitInfoProto> commitInfoProtos) {
        return this.updateCommitInfosMap(commitInfoProtos, this.watchType);
    }

    public long updateCommitInfosMap(Collection<RaftProtos.CommitInfoProto> commitInfoProtos, RaftProtos.ReplicationLevel level) {
        Stream<Long> stream = this.commitInfoMap.isEmpty() ? commitInfoProtos.stream().map(this::putCommitInfo) : commitInfoProtos.stream().map(proto -> this.commitInfoMap.computeIfPresent(RatisHelper.toDatanodeId((RaftProtos.RaftPeerProto)proto.getServer()), (address, index) -> proto.getCommitIndex())).filter(Objects::nonNull);
        if (level == RaftProtos.ReplicationLevel.ALL_COMMITTED) {
            return stream.mapToLong(Long::longValue).min().orElse(0L);
        }
        return stream.sorted(Comparator.reverseOrder()).limit(this.majority).skip(this.majority - 1).findFirst().orElse(0L);
    }

    private long putCommitInfo(RaftProtos.CommitInfoProto proto) {
        long index = proto.getCommitIndex();
        this.commitInfoMap.put(RatisHelper.toDatanodeId((RaftProtos.RaftPeerProto)proto.getServer()), index);
        return index;
    }

    public HddsProtos.ReplicationType getPipelineType() {
        return HddsProtos.ReplicationType.RATIS;
    }

    public Pipeline getPipeline() {
        return this.pipeline;
    }

    public void connect() throws Exception {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connecting to pipeline:{} leaderDatanode:{}, primaryDatanode:{}", new Object[]{this.getPipeline().getId(), RatisHelper.toRaftPeerId((DatanodeDetails)this.pipeline.getLeaderNode()), RatisHelper.toRaftPeerId((DatanodeDetails)this.pipeline.getClosestNode())});
        }
        if (!this.client.compareAndSet(null, RatisHelper.newRaftClient((RpcType)this.rpcType, (Pipeline)this.getPipeline(), (RetryPolicy)this.retryPolicy, (GrpcTlsConfig)this.tlsConfig, (ConfigurationSource)this.ozoneConfiguration))) {
            throw new IllegalStateException("Client is already connected.");
        }
    }

    public void close() {
        RaftClient c = this.client.getAndSet(null);
        if (c != null) {
            this.closeRaftClient(c);
        }
    }

    private void closeRaftClient(RaftClient raftClient) {
        try {
            raftClient.close();
        }
        catch (IOException e) {
            throw new IllegalStateException(e);
        }
    }

    private RaftClient getClient() {
        return Objects.requireNonNull(this.client.get(), "client is null");
    }

    @VisibleForTesting
    public ConcurrentMap<UUID, Long> getCommitInfoMap() {
        return this.commitInfoMap;
    }

    private CompletableFuture<RaftClientReply> sendRequestAsync(ContainerProtos.ContainerCommandRequestProto request) {
        RaftClientReply response;
        if (this.errorInjector != null && (response = this.errorInjector.getResponse(request, this.getClient().getId(), this.pipeline)) != null) {
            return CompletableFuture.completedFuture(response);
        }
        return (CompletableFuture)TracingUtil.executeInNewSpan((String)("XceiverClientRatis." + request.getCmdType().name()), () -> {
            ContainerCommandRequestMessage message = ContainerCommandRequestMessage.toMessage((ContainerProtos.ContainerCommandRequestProto)request, (String)TracingUtil.exportCurrentSpan());
            if (HddsUtils.isReadOnly((ContainerProtos.ContainerCommandRequestProtoOrBuilder)request)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("sendCommandAsync ReadOnly {}", (Object)message);
                }
                return this.getClient().async().sendReadOnly((Message)message);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("sendCommandAsync {}", (Object)message);
            }
            return this.getClient().async().send((Message)message);
        });
    }

    public long getReplicatedMinCommitIndex() {
        return this.commitInfoMap.values().stream().mapToLong(Long::longValue).min().orElse(0L);
    }

    private void addDatanodetoReply(UUID address, XceiverClientReply reply) {
        DatanodeDetails.Builder builder = DatanodeDetails.newBuilder();
        builder.setUuid(address);
        reply.addDatanode(builder.build());
    }

    private XceiverClientReply newWatchReply(long watchIndex, Object reason, long replyIndex) {
        LOG.debug("watchForCommit({}) returns {} {}", new Object[]{watchIndex, reason, replyIndex});
        XceiverClientReply reply = new XceiverClientReply(null);
        reply.setLogIndex(replyIndex);
        return reply;
    }

    public CompletableFuture<XceiverClientReply> watchForCommit(long index) {
        long replicatedMin = this.getReplicatedMinCommitIndex();
        if (replicatedMin >= index) {
            return CompletableFuture.completedFuture(this.newWatchReply(index, "replicatedMin", replicatedMin));
        }
        CompletableFuture<XceiverClientReply> replyFuture = new CompletableFuture<XceiverClientReply>();
        ((CompletableFuture)this.getClient().async().watch(index, this.watchType).thenAccept(reply -> {
            long updated = this.updateCommitInfosMap((RaftClientReply)reply, this.watchType);
            Preconditions.checkState((updated >= index ? 1 : 0) != 0, (Object)("Returned index " + updated + " < expected " + index));
            replyFuture.complete(this.newWatchReply(index, this.watchType, updated));
        })).exceptionally(e -> {
            boolean isGroupMismatch;
            LOG.warn("{} way commit failed on pipeline {}", new Object[]{this.watchType, this.pipeline, e});
            boolean bl = isGroupMismatch = HddsClientUtils.containsException(e, GroupMismatchException.class) != null;
            if (!isGroupMismatch && this.watchType == RaftProtos.ReplicationLevel.ALL_COMMITTED) {
                Throwable nre = HddsClientUtils.containsException(e, NotReplicatedException.class);
                if (nre instanceof NotReplicatedException) {
                    Collection commitInfoProtoList = ((NotReplicatedException)nre).getCommitInfos();
                    replyFuture.complete(this.handleFailedAllCommit(index, commitInfoProtoList));
                } else {
                    ((CompletableFuture)this.getClient().async().watch(index, RaftProtos.ReplicationLevel.MAJORITY_COMMITTED).thenApply(reply -> this.handleFailedAllCommit(index, reply.getCommitInfos()))).whenComplete(JavaUtils.asBiConsumer((CompletableFuture)replyFuture));
                }
            } else {
                replyFuture.completeExceptionally((Throwable)e);
            }
            return null;
        });
        return replyFuture;
    }

    private XceiverClientReply handleFailedAllCommit(long index, Collection<RaftProtos.CommitInfoProto> commitInfoProtoList) {
        XceiverClientReply clientReply = this.newWatchReply(index, RaftProtos.ReplicationLevel.MAJORITY_COMMITTED, index);
        commitInfoProtoList.stream().filter(i -> i.getCommitIndex() < index).forEach(proto -> {
            UUID address = RatisHelper.toDatanodeId((RaftProtos.RaftPeerProto)proto.getServer());
            this.addDatanodetoReply(address, clientReply);
            this.commitInfoMap.remove(address);
            LOG.info("Could not commit index {} on pipeline {} to all the nodes. Server {} has failed. Committed by majority.", new Object[]{index, this.pipeline, address});
        });
        return clientReply;
    }

    public XceiverClientReply sendCommandAsync(ContainerProtos.ContainerCommandRequestProto request) {
        XceiverClientReply asyncReply = new XceiverClientReply(null);
        long requestTime = System.currentTimeMillis();
        CompletableFuture<RaftClientReply> raftClientReply = this.sendRequestAsync(request);
        this.metrics.incrPendingContainerOpsMetrics(request.getCmdType());
        CompletionStage containerCommandResponse = ((CompletableFuture)raftClientReply.whenComplete((reply, e) -> {
            if (LOG.isDebugEnabled()) {
                LOG.debug("received reply {} for request: cmdType={}, containerID={}, pipelineID={}, traceID={}", new Object[]{reply, request.getCmdType(), request.getContainerID(), request.getPipelineID(), request.getTraceID(), e});
            }
            this.metrics.decrPendingContainerOpsMetrics(request.getCmdType());
            this.metrics.addContainerOpsLatency(request.getCmdType(), System.currentTimeMillis() - requestTime);
        })).thenApply(reply -> {
            try {
                if (!reply.isSuccess()) {
                    RaftException exception = reply.getException();
                    Preconditions.checkNotNull((Object)exception, (Object)"Raft reply failure but no exception propagated.");
                    throw new CompletionException((Throwable)exception);
                }
                ContainerProtos.ContainerCommandResponseProto response = ContainerProtos.ContainerCommandResponseProto.parseFrom((ByteString)reply.getMessage().getContent());
                UUID serverId = RatisHelper.toDatanodeId((String)reply.getReplierId());
                if (response.getResult() == ContainerProtos.Result.SUCCESS) {
                    this.updateCommitInfosMap(reply.getCommitInfos(), this.watchType);
                }
                asyncReply.setLogIndex(reply.getLogIndex());
                this.addDatanodetoReply(serverId, asyncReply);
                return response;
            }
            catch (InvalidProtocolBufferException e) {
                throw new CompletionException(e);
            }
        });
        asyncReply.setResponse((CompletableFuture)containerCommandResponse);
        return asyncReply;
    }

    public Map<DatanodeDetails, ContainerProtos.ContainerCommandResponseProto> sendCommandOnAllNodes(ContainerProtos.ContainerCommandRequestProto request) {
        throw new UnsupportedOperationException("Operation Not supported for ratis client");
    }

    public DataStreamApi getDataStreamApi() {
        return this.getClient().getDataStreamApi();
    }
}

