/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.splunk;

import com.splunk.RequestMessage;
import com.splunk.ResponseMessage;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.ReadsAttribute;
import org.apache.nifi.annotation.behavior.ReadsAttributes;
import org.apache.nifi.annotation.configuration.DefaultSchedule;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.annotation.lifecycle.OnStopped;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.dto.splunk.EventIndexStatusRequest;
import org.apache.nifi.dto.splunk.EventIndexStatusResponse;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.migration.PropertyConfiguration;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.splunk.PutSplunkHTTP;
import org.apache.nifi.processors.splunk.SplunkAPICall;
import org.apache.nifi.scheduling.SchedulingStrategy;

@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@Tags(value={"splunk", "logs", "http", "acknowledgement"})
@CapabilityDescription(value="Queries Splunk server in order to acquire the status of indexing acknowledgement.")
@ReadsAttributes(value={@ReadsAttribute(attribute="splunk.acknowledgement.id", description="The indexing acknowledgement id provided by Splunk."), @ReadsAttribute(attribute="splunk.responded.at", description="The time of the response of put request for Splunk.")})
@SeeAlso(value={PutSplunkHTTP.class})
@DefaultSchedule(strategy=SchedulingStrategy.TIMER_DRIVEN, period="1 min")
public class QuerySplunkIndexingStatus
extends SplunkAPICall {
    private static final String ENDPOINT = "/services/collector/ack";
    static final Relationship RELATIONSHIP_ACKNOWLEDGED = new Relationship.Builder().name("success").description("A FlowFile is transferred to this relationship when the acknowledgement was successful.").build();
    static final Relationship RELATIONSHIP_UNACKNOWLEDGED = new Relationship.Builder().name("unacknowledged").description("A FlowFile is transferred to this relationship when the acknowledgement was not successful. This can happen when the acknowledgement did not happened within the time period set for Maximum Waiting Time. FlowFiles with acknowledgement id unknown for the Splunk server will be transferred to this relationship after the Maximum Waiting Time is reached.").build();
    static final Relationship RELATIONSHIP_UNDETERMINED = new Relationship.Builder().name("undetermined").description("A FlowFile is transferred to this relationship when the acknowledgement state is not determined. FlowFiles transferred to this relationship might be penalized. This happens when Splunk returns with HTTP 200 but with false response for the acknowledgement id in the flow file attribute.").build();
    static final Relationship RELATIONSHIP_FAILURE = new Relationship.Builder().name("failure").description("A FlowFile is transferred to this relationship when the acknowledgement was not successful due to errors during the communication. FlowFiles are timing out or unknown by the Splunk server will transferred to \"undetermined\" relationship.").build();
    private static final Set<Relationship> RELATIONSHIPS = Set.of(RELATIONSHIP_ACKNOWLEDGED, RELATIONSHIP_UNACKNOWLEDGED, RELATIONSHIP_UNDETERMINED, RELATIONSHIP_FAILURE);
    static final PropertyDescriptor TTL = new PropertyDescriptor.Builder().name("Maximum Waiting Time").description("The maximum time the processor tries to acquire acknowledgement confirmation for an index, from the point of registration. After the given amount of time, the processor considers the index as not acknowledged and transfers the FlowFile to the \"unacknowledged\" relationship.").defaultValue("1 hour").required(true).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).build();
    static final PropertyDescriptor MAX_QUERY_SIZE = new PropertyDescriptor.Builder().name("Maximum Query Size").description("The maximum number of acknowledgement identifiers the outgoing query contains in one batch. It is recommended not to set it too low in order to reduce network communication.").defaultValue("10000").required(true).addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).build();
    private static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = Stream.concat(QuerySplunkIndexingStatus.getCommonPropertyDescriptors().stream(), Stream.of(TTL, MAX_QUERY_SIZE)).toList();
    private volatile Integer maxQuerySize;
    private volatile Integer ttl;

    @Override
    public List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    @Override
    @OnScheduled
    public void onScheduled(ProcessContext context) {
        super.onScheduled(context);
        this.maxQuerySize = context.getProperty(MAX_QUERY_SIZE).asInteger();
        this.ttl = context.getProperty(TTL).asTimePeriod(TimeUnit.MILLISECONDS).intValue();
    }

    @Override
    @OnStopped
    public void onStopped() {
        super.onStopped();
        this.maxQuerySize = null;
        this.ttl = null;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        RequestMessage requestMessage;
        List flowFiles = session.get(this.maxQuerySize.intValue());
        if (flowFiles.isEmpty()) {
            return;
        }
        long currentTime = System.currentTimeMillis();
        HashMap<Long, FlowFile> undetermined = new HashMap<Long, FlowFile>();
        for (FlowFile flowFile : flowFiles) {
            Optional<Long> sentAt = QuerySplunkIndexingStatus.extractLong(flowFile.getAttribute("splunk.responded.at"));
            Optional<Long> ackId = QuerySplunkIndexingStatus.extractLong(flowFile.getAttribute("splunk.acknowledgement.id"));
            if (!sentAt.isPresent() || !ackId.isPresent()) {
                this.getLogger().error("Flow file ({}) attributes {} and {} are expected to be set using 64-bit integer values!", new Object[]{flowFile.getId(), "splunk.responded.at", "splunk.acknowledgement.id"});
                session.transfer(flowFile, RELATIONSHIP_FAILURE);
                continue;
            }
            undetermined.put(ackId.get(), flowFile);
        }
        if (undetermined.isEmpty()) {
            this.getLogger().debug("There was no eligible flow file to send request to Splunk.");
            return;
        }
        try {
            requestMessage = this.createRequestMessage(undetermined);
        }
        catch (IOException e) {
            this.getLogger().error("Could not prepare Splunk request!", (Throwable)e);
            session.transfer(undetermined.values(), RELATIONSHIP_FAILURE);
            return;
        }
        try {
            ResponseMessage responseMessage = this.call(ENDPOINT, requestMessage);
            if (responseMessage.getStatus() == 200) {
                EventIndexStatusResponse splunkResponse = this.unmarshallResult(responseMessage.getContent(), EventIndexStatusResponse.class);
                splunkResponse.getAcks().forEach((flowFileId, isAcknowledged) -> {
                    FlowFile toTransfer = (FlowFile)undetermined.get(flowFileId);
                    if (isAcknowledged.booleanValue()) {
                        session.transfer(toTransfer, RELATIONSHIP_ACKNOWLEDGED);
                    } else {
                        Long sentAt = QuerySplunkIndexingStatus.extractLong(toTransfer.getAttribute("splunk.responded.at")).get();
                        if (sentAt + (long)this.ttl.intValue() < currentTime) {
                            session.transfer(toTransfer, RELATIONSHIP_UNACKNOWLEDGED);
                        } else {
                            session.penalize(toTransfer);
                            session.transfer(toTransfer, RELATIONSHIP_UNDETERMINED);
                        }
                    }
                });
            } else {
                this.getLogger().error("Query index status was not successful because of ({}) {}", new Object[]{responseMessage.getStatus(), responseMessage.getContent()});
                context.yield();
                session.transfer(undetermined.values(), RELATIONSHIP_UNDETERMINED);
            }
        }
        catch (Exception e) {
            this.getLogger().error("Error during communication with Splunk server", (Throwable)e);
            session.transfer(undetermined.values(), RELATIONSHIP_FAILURE);
        }
    }

    @Override
    public void migrateProperties(PropertyConfiguration config) {
        super.migrateProperties(config);
        config.renameProperty("ttl", TTL.getName());
        config.renameProperty("max-query-size", MAX_QUERY_SIZE.getName());
    }

    private RequestMessage createRequestMessage(Map<Long, FlowFile> undetermined) throws IOException {
        RequestMessage requestMessage = new RequestMessage("POST");
        requestMessage.getHeader().put("Content-Type", "application/json");
        requestMessage.setContent(this.generateContent(undetermined));
        return requestMessage;
    }

    private String generateContent(Map<Long, FlowFile> undetermined) throws IOException {
        EventIndexStatusRequest splunkRequest = new EventIndexStatusRequest();
        splunkRequest.setAcks(new ArrayList<Long>(undetermined.keySet()));
        return this.marshalRequest(splunkRequest);
    }

    private static Optional<Long> extractLong(String value) {
        try {
            return Optional.ofNullable(value).map(Long::valueOf);
        }
        catch (NumberFormatException e) {
            return Optional.empty();
        }
    }
}

