/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.security.util.crypto.scrypt;

import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.security.util.crypto.CipherUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Scrypt {
    private static final Logger logger = LoggerFactory.getLogger(Scrypt.class);
    private static final int DEFAULT_SALT_LENGTH = 16;
    private static final Pattern SCRYPT_PATTERN = Pattern.compile("^\\$\\w{2}\\$\\w{5,}\\$[\\w\\/\\=\\+]{11,64}\\$[\\w\\/\\=\\+]{1,256}$");

    public static String scrypt(String password, int n, int r, int p, int dkLen) {
        byte[] salt = new byte[16];
        new SecureRandom().nextBytes(salt);
        return Scrypt.scrypt(password, salt, n, r, p, dkLen);
    }

    public static String scrypt(String password, byte[] salt, int n, int r, int p, int dkLen) {
        try {
            byte[] derived = Scrypt.deriveScryptKey(password.getBytes(StandardCharsets.UTF_8), salt, n, r, p, dkLen);
            return Scrypt.formatHash(salt, n, r, p, derived);
        }
        catch (GeneralSecurityException e) {
            throw new IllegalStateException("JVM doesn't support SHA1PRNG or HMAC_SHA256?");
        }
    }

    public static String formatSalt(byte[] salt, int n, int r, int p) {
        String params = Scrypt.encodeParams(n, r, p);
        StringBuilder sb = new StringBuilder(salt.length * 2);
        sb.append("$s0$").append(params).append('$');
        sb.append(CipherUtility.encodeBase64NoPadding(salt));
        return sb.toString();
    }

    private static String encodeParams(int n, int r, int p) {
        return Long.toString(Scrypt.log2(n) << 16 | r << 8 | p, 16);
    }

    private static String formatHash(byte[] salt, int n, int r, int p, byte[] derived) {
        StringBuilder sb = new StringBuilder((salt.length + derived.length) * 2);
        sb.append(Scrypt.formatSalt(salt, n, r, p)).append('$');
        sb.append(CipherUtility.encodeBase64NoPadding(derived));
        return sb.toString();
    }

    public static int calculateExpectedMemory(int n, int r, int p) {
        return 128 * r * n + 128 * r * p;
    }

    public static boolean check(String password, String hashed) {
        try {
            if (StringUtils.isEmpty((CharSequence)password)) {
                throw new IllegalArgumentException("Password cannot be empty");
            }
            if (StringUtils.isEmpty((CharSequence)hashed)) {
                throw new IllegalArgumentException("Hash cannot be empty");
            }
            if (!Scrypt.verifyHashFormat(hashed)) {
                throw new IllegalArgumentException("Hash is not properly formatted");
            }
            String[] parts = hashed.split("\\$");
            List<Integer> splitParams = Scrypt.parseParameters(parts[2]);
            int n = splitParams.get(0);
            int r = splitParams.get(1);
            int p = splitParams.get(2);
            byte[] salt = Base64.decodeBase64((String)parts[3]);
            byte[] derived0 = Base64.decodeBase64((String)parts[4]);
            int hashLength = derived0.length * 8;
            byte[] derived1 = Scrypt.deriveScryptKey(password.getBytes(StandardCharsets.UTF_8), salt, n, r, p, hashLength);
            if (derived0.length != derived1.length) {
                return false;
            }
            int result = 0;
            for (int i = 0; i < derived0.length; ++i) {
                result |= derived0[i] ^ derived1[i];
            }
            return result == 0;
        }
        catch (GeneralSecurityException e) {
            throw new IllegalStateException("JVM doesn't support SHA1PRNG or HMAC_SHA256?");
        }
    }

    public static boolean verifyHashFormat(String hash) {
        if (StringUtils.isBlank((CharSequence)hash)) {
            return false;
        }
        if (!hash.startsWith("$s0$")) {
            return false;
        }
        return SCRYPT_PATTERN.matcher(hash).matches();
    }

    public static List<Integer> parseParameters(String encodedParams) {
        long params = Long.parseLong(encodedParams, 16);
        ArrayList<Integer> paramsList = new ArrayList<Integer>(3);
        paramsList.add((int)Math.pow(2.0, params >> 16 & 0xFFFFL));
        paramsList.add((int)params >> 8 & 0xFF);
        paramsList.add((int)params & 0xFF);
        return paramsList;
    }

    private static int log2(int n) {
        int log = 0;
        if ((n & 0xFFFF0000) != 0) {
            n >>>= 16;
            log = 16;
        }
        if (n >= 256) {
            n >>>= 8;
            log += 8;
        }
        if (n >= 16) {
            n >>>= 4;
            log += 4;
        }
        if (n >= 4) {
            n >>>= 2;
            log += 2;
        }
        return log + (n >>> 1);
    }

    protected static byte[] deriveScryptKey(byte[] password, byte[] salt, int n, int r, int p, int dkLen) throws GeneralSecurityException {
        int saltLength;
        if (n < 2 || (n & n - 1) != 0) {
            throw new IllegalArgumentException("N must be a power of 2 greater than 1");
        }
        if (r < 1) {
            throw new IllegalArgumentException("Parameter r must be 1 or greater");
        }
        if (p < 1) {
            throw new IllegalArgumentException("Parameter p must be 1 or greater");
        }
        if (n > 0xFFFFFF / r) {
            throw new IllegalArgumentException("Parameter N is too large");
        }
        if (p > 0xFFFFFF) {
            throw new IllegalArgumentException("Parameter p is too large");
        }
        if (r > 0xFFFFFF / p) {
            throw new IllegalArgumentException("Parameter r is too large");
        }
        if (password == null || password.length == 0) {
            throw new IllegalArgumentException("Password cannot be empty");
        }
        int n2 = saltLength = salt == null ? 0 : salt.length;
        if (salt == null || saltLength == 0) {
            logger.warn("An empty salt was used for scrypt key derivation");
            if (salt == null) {
                salt = new byte[]{};
            }
        }
        if (saltLength < 8 || saltLength > 32) {
            logger.warn("A salt of length {} was used for scrypt key derivation", (Object)saltLength);
        }
        Mac mac = Mac.getInstance("HmacSHA256");
        mac.init(new SecretKeySpec(password, "HmacSHA256"));
        byte[] b = new byte[128 * r * p];
        byte[] xy = new byte[256 * r];
        byte[] v = new byte[128 * r * n];
        Scrypt.pbkdf2(mac, salt, 1, b, p * 128 * r);
        for (int i = 0; i < p; ++i) {
            Scrypt.smix(b, i * 128 * r, r, n, v, xy);
        }
        byte[] dk = new byte[dkLen / 8];
        Scrypt.pbkdf2(mac, b, 1, dk, dkLen / 8);
        return dk;
    }

    private static byte[] pbkdf2(String alg, byte[] p, byte[] s, int c, int dkLen) throws GeneralSecurityException {
        Mac mac = Mac.getInstance(alg);
        mac.init(new SecretKeySpec(p, alg));
        byte[] dk = new byte[dkLen];
        Scrypt.pbkdf2(mac, s, c, dk, dkLen);
        return dk;
    }

    private static void pbkdf2(Mac mac, byte[] s, int c, byte[] dk, int dkLen) throws GeneralSecurityException {
        int hLen = mac.getMacLength();
        if ((double)dkLen > (Math.pow(2.0, 32.0) - 1.0) * (double)hLen) {
            throw new GeneralSecurityException("Requested key length too long");
        }
        byte[] U = new byte[hLen];
        byte[] T = new byte[hLen];
        byte[] block1 = new byte[s.length + 4];
        int l = (int)Math.ceil((double)dkLen / (double)hLen);
        int r = dkLen - (l - 1) * hLen;
        System.arraycopy(s, 0, block1, 0, s.length);
        for (int i = 1; i <= l; ++i) {
            block1[s.length + 0] = (byte)(i >> 24 & 0xFF);
            block1[s.length + 1] = (byte)(i >> 16 & 0xFF);
            block1[s.length + 2] = (byte)(i >> 8 & 0xFF);
            block1[s.length + 3] = (byte)(i >> 0 & 0xFF);
            mac.update(block1);
            mac.doFinal(U, 0);
            System.arraycopy(U, 0, T, 0, hLen);
            for (int j = 1; j < c; ++j) {
                mac.update(U);
                mac.doFinal(U, 0);
                for (int k = 0; k < hLen; ++k) {
                    int n = k;
                    T[n] = (byte)(T[n] ^ U[k]);
                }
            }
            System.arraycopy(T, 0, dk, (i - 1) * hLen, i == l ? r : hLen);
        }
    }

    private static void smix(byte[] b, int bi, int r, int n, byte[] v, byte[] xy) {
        int i;
        int xi = 0;
        int yi = 128 * r;
        System.arraycopy(b, bi, xy, xi, 128 * r);
        for (i = 0; i < n; ++i) {
            System.arraycopy(xy, xi, v, i * (128 * r), 128 * r);
            Scrypt.blockmix_salsa8(xy, xi, yi, r);
        }
        for (i = 0; i < n; ++i) {
            int j = Scrypt.integerify(xy, xi, r) & n - 1;
            Scrypt.blockxor(v, j * (128 * r), xy, xi, 128 * r);
            Scrypt.blockmix_salsa8(xy, xi, yi, r);
        }
        System.arraycopy(xy, xi, b, bi, 128 * r);
    }

    private static void blockmix_salsa8(byte[] by, int bi, int yi, int r) {
        int i;
        byte[] X = new byte[64];
        System.arraycopy(by, bi + (2 * r - 1) * 64, X, 0, 64);
        for (i = 0; i < 2 * r; ++i) {
            Scrypt.blockxor(by, i * 64, X, 0, 64);
            Scrypt.salsa20_8(X);
            System.arraycopy(X, 0, by, yi + i * 64, 64);
        }
        for (i = 0; i < r; ++i) {
            System.arraycopy(by, yi + i * 2 * 64, by, bi + i * 64, 64);
        }
        for (i = 0; i < r; ++i) {
            System.arraycopy(by, yi + (i * 2 + 1) * 64, by, bi + (i + r) * 64, 64);
        }
    }

    private static int r(int a, int b) {
        return a << b | a >>> 32 - b;
    }

    private static void salsa20_8(byte[] b) {
        int i;
        int[] b32 = new int[16];
        int[] x = new int[16];
        for (i = 0; i < 16; ++i) {
            b32[i] = (b[i * 4 + 0] & 0xFF) << 0;
            int n = i;
            b32[n] = b32[n] | (b[i * 4 + 1] & 0xFF) << 8;
            int n2 = i;
            b32[n2] = b32[n2] | (b[i * 4 + 2] & 0xFF) << 16;
            int n3 = i;
            b32[n3] = b32[n3] | (b[i * 4 + 3] & 0xFF) << 24;
        }
        System.arraycopy(b32, 0, x, 0, 16);
        for (i = 8; i > 0; i -= 2) {
            x[4] = x[4] ^ Scrypt.r(x[0] + x[12], 7);
            x[8] = x[8] ^ Scrypt.r(x[4] + x[0], 9);
            x[12] = x[12] ^ Scrypt.r(x[8] + x[4], 13);
            x[0] = x[0] ^ Scrypt.r(x[12] + x[8], 18);
            x[9] = x[9] ^ Scrypt.r(x[5] + x[1], 7);
            x[13] = x[13] ^ Scrypt.r(x[9] + x[5], 9);
            x[1] = x[1] ^ Scrypt.r(x[13] + x[9], 13);
            x[5] = x[5] ^ Scrypt.r(x[1] + x[13], 18);
            x[14] = x[14] ^ Scrypt.r(x[10] + x[6], 7);
            x[2] = x[2] ^ Scrypt.r(x[14] + x[10], 9);
            x[6] = x[6] ^ Scrypt.r(x[2] + x[14], 13);
            x[10] = x[10] ^ Scrypt.r(x[6] + x[2], 18);
            x[3] = x[3] ^ Scrypt.r(x[15] + x[11], 7);
            x[7] = x[7] ^ Scrypt.r(x[3] + x[15], 9);
            x[11] = x[11] ^ Scrypt.r(x[7] + x[3], 13);
            x[15] = x[15] ^ Scrypt.r(x[11] + x[7], 18);
            x[1] = x[1] ^ Scrypt.r(x[0] + x[3], 7);
            x[2] = x[2] ^ Scrypt.r(x[1] + x[0], 9);
            x[3] = x[3] ^ Scrypt.r(x[2] + x[1], 13);
            x[0] = x[0] ^ Scrypt.r(x[3] + x[2], 18);
            x[6] = x[6] ^ Scrypt.r(x[5] + x[4], 7);
            x[7] = x[7] ^ Scrypt.r(x[6] + x[5], 9);
            x[4] = x[4] ^ Scrypt.r(x[7] + x[6], 13);
            x[5] = x[5] ^ Scrypt.r(x[4] + x[7], 18);
            x[11] = x[11] ^ Scrypt.r(x[10] + x[9], 7);
            x[8] = x[8] ^ Scrypt.r(x[11] + x[10], 9);
            x[9] = x[9] ^ Scrypt.r(x[8] + x[11], 13);
            x[10] = x[10] ^ Scrypt.r(x[9] + x[8], 18);
            x[12] = x[12] ^ Scrypt.r(x[15] + x[14], 7);
            x[13] = x[13] ^ Scrypt.r(x[12] + x[15], 9);
            x[14] = x[14] ^ Scrypt.r(x[13] + x[12], 13);
            x[15] = x[15] ^ Scrypt.r(x[14] + x[13], 18);
        }
        for (i = 0; i < 16; ++i) {
            b32[i] = x[i] + b32[i];
        }
        for (i = 0; i < 16; ++i) {
            b[i * 4 + 0] = (byte)(b32[i] >> 0 & 0xFF);
            b[i * 4 + 1] = (byte)(b32[i] >> 8 & 0xFF);
            b[i * 4 + 2] = (byte)(b32[i] >> 16 & 0xFF);
            b[i * 4 + 3] = (byte)(b32[i] >> 24 & 0xFF);
        }
    }

    private static void blockxor(byte[] s, int si, byte[] d, int di, int len) {
        for (int i = 0; i < len; ++i) {
            int n = di + i;
            d[n] = (byte)(d[n] ^ s[si + i]);
        }
    }

    private static int integerify(byte[] b, int bi, int r) {
        int n = (b[(bi += (2 * r - 1) * 64) + 0] & 0xFF) << 0;
        n |= (b[bi + 1] & 0xFF) << 8;
        n |= (b[bi + 2] & 0xFF) << 16;
        return n |= (b[bi + 3] & 0xFF) << 24;
    }

    public static int getDefaultSaltLength() {
        return 16;
    }
}

