/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.asset;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HexFormat;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.nifi.asset.Asset;
import org.apache.nifi.asset.AssetManager;
import org.apache.nifi.asset.AssetManagerInitializationContext;
import org.apache.nifi.asset.StandardAsset;
import org.apache.nifi.nar.FileDigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardAssetManager
implements AssetManager {
    private static final Logger logger = LoggerFactory.getLogger(StandardAssetManager.class);
    public static final String ASSET_STORAGE_LOCATION_PROPERTY = "directory";
    public static final String DEFAULT_ASSET_STORAGE_LOCATION = "./assets";
    private volatile File assetStorageLocation;
    private final Map<String, Asset> assets = new ConcurrentHashMap<String, Asset>();

    public void initialize(AssetManagerInitializationContext context) {
        String storageLocation = this.getStorageLocation(context);
        this.assetStorageLocation = new File(storageLocation);
        if (!this.assetStorageLocation.exists()) {
            try {
                Files.createDirectories(this.assetStorageLocation.toPath(), new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new RuntimeException("The Asset Manager's [%s] property is set to [%s] but the directory does not exist and cannot be created".formatted(ASSET_STORAGE_LOCATION_PROPERTY, storageLocation), e);
            }
        }
        try {
            this.recoverLocalAssets();
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to access assets", e);
        }
    }

    public Asset createAsset(String parameterContextId, String assetName, InputStream contents) throws IOException {
        String assetId = this.createAssetId(parameterContextId, assetName);
        File file = this.getFile(parameterContextId, assetName);
        File dir = file.getParentFile();
        if (!dir.exists()) {
            try {
                Files.createDirectories(dir.toPath(), new FileAttribute[0]);
            }
            catch (IOException ioe) {
                throw new IOException("Could not create directory in order to store asset", ioe);
            }
        }
        File tempFile = new File(dir, file.getName() + ".tmp");
        logger.debug("Writing temp asset file [{}]", (Object)tempFile.getAbsolutePath());
        try {
            Files.copy(contents, tempFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (Exception e) {
            throw new IOException("Failed to write asset to file " + tempFile.getAbsolutePath(), e);
        }
        Files.move(tempFile.toPath(), file.toPath(), StandardCopyOption.REPLACE_EXISTING);
        String digest = this.computeDigest(file);
        StandardAsset asset = new StandardAsset(assetId, parameterContextId, assetName, file, digest);
        this.assets.put(assetId, asset);
        return asset;
    }

    public Optional<Asset> getAsset(String id) {
        return Optional.ofNullable(this.assets.get(id));
    }

    public List<Asset> getAssets(String parameterContextId) {
        ArrayList<Asset> allAssets = new ArrayList<Asset>(this.assets.values());
        ArrayList<Asset> paramContextAssets = new ArrayList<Asset>();
        for (Asset asset : allAssets) {
            if (!asset.getParameterContextIdentifier().equals(parameterContextId)) continue;
            paramContextAssets.add(asset);
        }
        return paramContextAssets;
    }

    public Asset createMissingAsset(String parameterContextId, String assetName) {
        String assetId = this.createAssetId(parameterContextId, assetName);
        File file = this.getFile(parameterContextId, assetName);
        StandardAsset asset = new StandardAsset(assetId, parameterContextId, assetName, file, null);
        this.assets.put(assetId, asset);
        return asset;
    }

    public Optional<Asset> deleteAsset(String id) {
        Asset removed = this.assets.remove(id);
        if (removed == null) {
            return Optional.empty();
        }
        File file = removed.getFile();
        if (file.exists()) {
            try {
                Files.delete(file.toPath());
            }
            catch (IOException e) {
                logger.warn("Failed to remove asset file {}", (Object)file.getAbsolutePath(), (Object)e);
            }
            File parentDir = file.getParentFile();
            File[] children = parentDir.listFiles();
            if (children != null && children.length == 0) {
                try {
                    Files.delete(parentDir.toPath());
                }
                catch (IOException e) {
                    logger.warn("Failed to remove empty asset directory {}", (Object)parentDir.getAbsolutePath(), (Object)e);
                }
            }
        }
        return Optional.of(removed);
    }

    private String createAssetId(String parameterContextId, String assetName) {
        String seed = parameterContextId + "/" + assetName;
        return UUID.nameUUIDFromBytes(seed.getBytes(StandardCharsets.UTF_8)).toString();
    }

    private File getFile(String paramContextId, String assetName) {
        Path parentPath = this.assetStorageLocation.toPath().normalize();
        Path assetPath = Paths.get(paramContextId, assetName).normalize();
        Path fullPath = parentPath.resolve(assetPath);
        return fullPath.toFile();
    }

    private String getStorageLocation(AssetManagerInitializationContext initializationContext) {
        String storageLocation = (String)initializationContext.getProperties().get(ASSET_STORAGE_LOCATION_PROPERTY);
        return storageLocation == null ? DEFAULT_ASSET_STORAGE_LOCATION : storageLocation;
    }

    private void recoverLocalAssets() throws IOException {
        File[] files = this.assetStorageLocation.listFiles();
        if (files == null) {
            throw new IOException("Unable to list files for asset storage location %s".formatted(this.assetStorageLocation.getAbsolutePath()));
        }
        for (File file : files) {
            if (!file.isDirectory()) continue;
            String contextId = file.getName();
            File[] assetFiles = file.listFiles();
            if (assetFiles == null) {
                logger.warn("Unable to determine which assets exist for Parameter Context {}", (Object)contextId);
                continue;
            }
            for (File assetFile : assetFiles) {
                String assetId = this.createAssetId(contextId, assetFile.getName());
                String digest = this.computeDigest(assetFile);
                StandardAsset asset = new StandardAsset(assetId, contextId, assetFile.getName(), assetFile, digest);
                this.assets.put(assetId, asset);
            }
        }
    }

    private String computeDigest(File file) throws IOException {
        return HexFormat.of().formatHex(FileDigestUtils.getDigest((File)file));
    }
}

