/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.store;

import java.io.Closeable;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.zip.CRC32;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexFileNames;
import org.apache.lucene.index.IndexNotFoundException;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.BaseDirectoryWrapper;
import org.apache.lucene.store.ChecksumIndexInput;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.MockDirectoryWrapper;
import org.apache.lucene.store.RandomAccessInput;
import org.apache.lucene.util.IOUtils;
import org.apache.lucene.util.LuceneTestCase;
import org.apache.lucene.util.TestUtil;
import org.junit.Assert;

public abstract class BaseDirectoryTestCase
extends LuceneTestCase {
    protected abstract Directory getDirectory(Path var1) throws IOException;

    public void testCopyFrom() throws Exception {
        Directory source = this.getDirectory(BaseDirectoryTestCase.createTempDir("testCopy"));
        BaseDirectoryWrapper dest = BaseDirectoryTestCase.newDirectory();
        IndexOutput output = source.createOutput("foobar", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int numBytes = BaseDirectoryTestCase.random().nextInt(20000);
        byte[] bytes = new byte[numBytes];
        BaseDirectoryTestCase.random().nextBytes(bytes);
        output.writeBytes(bytes, bytes.length);
        output.close();
        dest.copyFrom(source, "foobar", "foobaz", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertTrue((boolean)BaseDirectoryTestCase.slowFileExists((Directory)dest, "foobaz"));
        IndexInput input = dest.openInput("foobaz", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        byte[] bytes2 = new byte[numBytes];
        input.readBytes(bytes2, 0, bytes2.length);
        BaseDirectoryTestCase.assertEquals((long)input.length(), (long)numBytes);
        input.close();
        BaseDirectoryTestCase.assertArrayEquals((byte[])bytes, (byte[])bytes2);
        IOUtils.close((Closeable[])new Closeable[]{source, dest});
    }

    public void testCopyFromDestination() throws Exception {
        BaseDirectoryWrapper source = BaseDirectoryTestCase.newDirectory();
        Directory dest = this.getDirectory(BaseDirectoryTestCase.createTempDir("testCopyDestination"));
        IndexOutput output = source.createOutput("foobar", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int numBytes = BaseDirectoryTestCase.random().nextInt(20000);
        byte[] bytes = new byte[numBytes];
        BaseDirectoryTestCase.random().nextBytes(bytes);
        output.writeBytes(bytes, bytes.length);
        output.close();
        dest.copyFrom((Directory)source, "foobar", "foobaz", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertTrue((boolean)BaseDirectoryTestCase.slowFileExists(dest, "foobaz"));
        IndexInput input = dest.openInput("foobaz", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        byte[] bytes2 = new byte[numBytes];
        input.readBytes(bytes2, 0, bytes2.length);
        BaseDirectoryTestCase.assertEquals((long)input.length(), (long)numBytes);
        input.close();
        BaseDirectoryTestCase.assertArrayEquals((byte[])bytes, (byte[])bytes2);
        IOUtils.close((Closeable[])new Closeable[]{source, dest});
    }

    public void testRename() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testRename"));
        IndexOutput output = dir.createOutput("foobar", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int numBytes = BaseDirectoryTestCase.random().nextInt(20000);
        byte[] bytes = new byte[numBytes];
        BaseDirectoryTestCase.random().nextBytes(bytes);
        output.writeBytes(bytes, bytes.length);
        output.close();
        dir.rename("foobar", "foobaz");
        IndexInput input = dir.openInput("foobaz", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        byte[] bytes2 = new byte[numBytes];
        input.readBytes(bytes2, 0, bytes2.length);
        BaseDirectoryTestCase.assertEquals((long)input.length(), (long)numBytes);
        input.close();
        BaseDirectoryTestCase.assertArrayEquals((byte[])bytes, (byte[])bytes2);
        dir.close();
    }

    public void testDeleteFile() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testDeleteFile"));
        int count = dir.listAll().length;
        dir.createOutput("foo.txt", IOContext.DEFAULT).close();
        BaseDirectoryTestCase.assertEquals((long)(count + 1), (long)dir.listAll().length);
        dir.deleteFile("foo.txt");
        BaseDirectoryTestCase.assertEquals((long)count, (long)dir.listAll().length);
        dir.close();
    }

    public void testByte() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testByte"));
        IndexOutput output = dir.createOutput("byte", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeByte((byte)-128);
        output.close();
        IndexInput input = dir.openInput("byte", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertEquals((long)1L, (long)input.length());
        BaseDirectoryTestCase.assertEquals((long)-128L, (long)input.readByte());
        input.close();
        dir.close();
    }

    public void testShort() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testShort"));
        IndexOutput output = dir.createOutput("short", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeShort((short)-20);
        output.close();
        IndexInput input = dir.openInput("short", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertEquals((long)2L, (long)input.length());
        BaseDirectoryTestCase.assertEquals((long)-20L, (long)input.readShort());
        input.close();
        dir.close();
    }

    public void testInt() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testInt"));
        IndexOutput output = dir.createOutput("int", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeInt(-500);
        output.close();
        IndexInput input = dir.openInput("int", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertEquals((long)4L, (long)input.length());
        BaseDirectoryTestCase.assertEquals((long)-500L, (long)input.readInt());
        input.close();
        dir.close();
    }

    public void testLong() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testLong"));
        IndexOutput output = dir.createOutput("long", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeLong(-5000L);
        output.close();
        IndexInput input = dir.openInput("long", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertEquals((long)8L, (long)input.length());
        BaseDirectoryTestCase.assertEquals((long)-5000L, (long)input.readLong());
        input.close();
        dir.close();
    }

    public void testString() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testString"));
        IndexOutput output = dir.createOutput("string", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeString("hello!");
        output.close();
        IndexInput input = dir.openInput("string", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertEquals((Object)"hello!", (Object)input.readString());
        BaseDirectoryTestCase.assertEquals((long)7L, (long)input.length());
        input.close();
        dir.close();
    }

    public void testVInt() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testVInt"));
        IndexOutput output = dir.createOutput("vint", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeVInt(500);
        output.close();
        IndexInput input = dir.openInput("vint", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertEquals((long)2L, (long)input.length());
        BaseDirectoryTestCase.assertEquals((long)500L, (long)input.readVInt());
        input.close();
        dir.close();
    }

    public void testVLong() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testVLong"));
        IndexOutput output = dir.createOutput("vlong", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeVLong(Long.MAX_VALUE);
        output.close();
        IndexInput input = dir.openInput("vlong", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertEquals((long)9L, (long)input.length());
        BaseDirectoryTestCase.assertEquals((long)Long.MAX_VALUE, (long)input.readVLong());
        input.close();
        dir.close();
    }

    public void testZInt() throws Exception {
        int[] ints = new int[BaseDirectoryTestCase.random().nextInt(10)];
        block5: for (int i = 0; i < ints.length; ++i) {
            switch (BaseDirectoryTestCase.random().nextInt(3)) {
                case 0: {
                    ints[i] = BaseDirectoryTestCase.random().nextInt();
                    continue block5;
                }
                case 1: {
                    ints[i] = BaseDirectoryTestCase.random().nextBoolean() ? Integer.MIN_VALUE : Integer.MAX_VALUE;
                    continue block5;
                }
                case 2: {
                    ints[i] = (BaseDirectoryTestCase.random().nextBoolean() ? -1 : 1) * BaseDirectoryTestCase.random().nextInt(1024);
                    continue block5;
                }
                default: {
                    throw new AssertionError();
                }
            }
        }
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testZInt"));
        IndexOutput output = dir.createOutput("zint", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        for (int i : ints) {
            output.writeZInt(i);
        }
        output.close();
        IndexInput input = dir.openInput("zint", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        for (int i : ints) {
            BaseDirectoryTestCase.assertEquals((long)i, (long)input.readZInt());
        }
        BaseDirectoryTestCase.assertEquals((long)input.length(), (long)input.getFilePointer());
        input.close();
        dir.close();
    }

    public void testZLong() throws Exception {
        long[] longs = new long[BaseDirectoryTestCase.random().nextInt(10)];
        block5: for (int i = 0; i < longs.length; ++i) {
            switch (BaseDirectoryTestCase.random().nextInt(3)) {
                case 0: {
                    longs[i] = BaseDirectoryTestCase.random().nextLong();
                    continue block5;
                }
                case 1: {
                    longs[i] = BaseDirectoryTestCase.random().nextBoolean() ? Long.MIN_VALUE : Long.MAX_VALUE;
                    continue block5;
                }
                case 2: {
                    longs[i] = (BaseDirectoryTestCase.random().nextBoolean() ? -1 : 1) * BaseDirectoryTestCase.random().nextInt(1024);
                    continue block5;
                }
                default: {
                    throw new AssertionError();
                }
            }
        }
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testZLong"));
        IndexOutput output = dir.createOutput("zlong", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        for (long l : longs) {
            output.writeZLong(l);
        }
        output.close();
        IndexInput input = dir.openInput("zlong", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        for (long l : longs) {
            BaseDirectoryTestCase.assertEquals((long)l, (long)input.readZLong());
        }
        BaseDirectoryTestCase.assertEquals((long)input.length(), (long)input.getFilePointer());
        input.close();
        dir.close();
    }

    public void testSetOfStrings() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testSetOfStrings"));
        IndexOutput output = dir.createOutput("stringset", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeSetOfStrings(BaseDirectoryTestCase.asSet("test1", "test2"));
        output.writeSetOfStrings(Collections.emptySet());
        output.writeSetOfStrings(BaseDirectoryTestCase.asSet("test3"));
        output.close();
        IndexInput input = dir.openInput("stringset", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        Set set1 = input.readSetOfStrings();
        BaseDirectoryTestCase.assertEquals(BaseDirectoryTestCase.asSet("test1", "test2"), (Object)set1);
        BaseDirectoryTestCase.expectThrows(UnsupportedOperationException.class, () -> set1.add("bogus"));
        Set set2 = input.readSetOfStrings();
        BaseDirectoryTestCase.assertEquals(Collections.emptySet(), (Object)set2);
        BaseDirectoryTestCase.expectThrows(UnsupportedOperationException.class, () -> set2.add("bogus"));
        Set set3 = input.readSetOfStrings();
        BaseDirectoryTestCase.assertEquals(Collections.singleton("test3"), (Object)set3);
        BaseDirectoryTestCase.expectThrows(UnsupportedOperationException.class, () -> set3.add("bogus"));
        BaseDirectoryTestCase.assertEquals((long)input.length(), (long)input.getFilePointer());
        input.close();
        dir.close();
    }

    public void testMapOfStrings() throws Exception {
        HashMap<String, String> m = new HashMap<String, String>();
        m.put("test1", "value1");
        m.put("test2", "value2");
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testMapOfStrings"));
        IndexOutput output = dir.createOutput("stringmap", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeMapOfStrings(m);
        output.writeMapOfStrings(Collections.emptyMap());
        output.writeMapOfStrings(Collections.singletonMap("key", "value"));
        output.close();
        IndexInput input = dir.openInput("stringmap", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        Map map1 = input.readMapOfStrings();
        BaseDirectoryTestCase.assertEquals(m, (Object)map1);
        BaseDirectoryTestCase.expectThrows(UnsupportedOperationException.class, () -> map1.put("bogus1", "bogus2"));
        Map map2 = input.readMapOfStrings();
        BaseDirectoryTestCase.assertEquals(Collections.emptyMap(), (Object)map2);
        BaseDirectoryTestCase.expectThrows(UnsupportedOperationException.class, () -> map2.put("bogus1", "bogus2"));
        Map map3 = input.readMapOfStrings();
        BaseDirectoryTestCase.assertEquals(Collections.singletonMap("key", "value"), (Object)map3);
        BaseDirectoryTestCase.expectThrows(UnsupportedOperationException.class, () -> map3.put("bogus1", "bogus2"));
        BaseDirectoryTestCase.assertEquals((long)input.length(), (long)input.getFilePointer());
        input.close();
        dir.close();
    }

    public void testChecksum() throws Exception {
        CRC32 expected = new CRC32();
        int numBytes = BaseDirectoryTestCase.random().nextInt(20000);
        byte[] bytes = new byte[numBytes];
        BaseDirectoryTestCase.random().nextBytes(bytes);
        expected.update(bytes);
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testChecksum"));
        IndexOutput output = dir.createOutput("checksum", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        output.writeBytes(bytes, 0, bytes.length);
        output.close();
        ChecksumIndexInput input = dir.openChecksumInput("checksum", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        input.skipBytes((long)numBytes);
        BaseDirectoryTestCase.assertEquals((long)expected.getValue(), (long)input.getChecksum());
        input.close();
        dir.close();
    }

    public void testDetectClose() throws Throwable {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testDetectClose"));
        dir.close();
        BaseDirectoryTestCase.expectThrows(AlreadyClosedException.class, () -> dir.createOutput("test", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random())));
    }

    public void testThreadSafety() throws Exception {
        final Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testThreadSafety"));
        if (dir instanceof BaseDirectoryWrapper) {
            ((BaseDirectoryWrapper)dir).setCheckIndexOnClose(false);
        }
        if (dir instanceof MockDirectoryWrapper) {
            ((MockDirectoryWrapper)dir).setThrottling(MockDirectoryWrapper.Throttling.NEVER);
        }
        if (VERBOSE) {
            System.out.println(dir);
        }
        class TheThread
        extends Thread {
            private String name;

            public TheThread(String name) {
                this.name = name;
            }

            @Override
            public void run() {
                for (int i = 0; i < 1000; ++i) {
                    String fileName = this.name + i;
                    try {
                        IndexOutput output = dir.createOutput(fileName, LuceneTestCase.newIOContext(LuceneTestCase.random()));
                        output.close();
                        Assert.assertTrue((boolean)LuceneTestCase.slowFileExists(dir, fileName));
                        continue;
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        TheThread theThread = new TheThread("t1");
        class TheThread2
        extends Thread {
            private String name;
            private volatile boolean stop;

            public TheThread2(String name) {
                this.name = name;
            }

            @Override
            public void run() {
                block5: while (!this.stop) {
                    try {
                        String[] files;
                        for (String file : files = dir.listAll()) {
                            block7: {
                                if (!file.startsWith(this.name)) continue;
                                try {
                                    IndexInput input = dir.openInput(file, LuceneTestCase.newIOContext(LuceneTestCase.random()));
                                    input.close();
                                }
                                catch (FileNotFoundException | NoSuchFileException input) {
                                }
                                catch (IOException e) {
                                    if (e.getMessage() != null && e.getMessage().contains("still open for writing")) break block7;
                                    throw new RuntimeException(e);
                                }
                            }
                            if (LuceneTestCase.random().nextBoolean()) continue block5;
                        }
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        TheThread2 theThread2 = new TheThread2("t2");
        theThread.start();
        theThread2.start();
        theThread.join();
        theThread2.stop = true;
        theThread2.join();
        dir.close();
    }

    public void testDirectoryFilter() throws IOException {
        String name = "file";
        try (Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testDirectoryFilter"));){
            dir.createOutput(name, BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random())).close();
            BaseDirectoryTestCase.assertTrue((boolean)BaseDirectoryTestCase.slowFileExists(dir, name));
            BaseDirectoryTestCase.assertTrue((boolean)Arrays.asList(dir.listAll()).contains(name));
        }
    }

    public void testSeekToEOFThenBack() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testSeekToEOFThenBack"));
        IndexOutput o = dir.createOutput("out", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        byte[] bytes = new byte[3072];
        o.writeBytes(bytes, 0, bytes.length);
        o.close();
        IndexInput i = dir.openInput("out", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        i.seek(2047L);
        i.seek(3072L);
        i.seek(1024L);
        i.readBytes(bytes, 0, 2048);
        i.close();
        dir.close();
    }

    public void testIllegalEOF() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testIllegalEOF"));
        IndexOutput o = dir.createOutput("out", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        byte[] b = new byte[1024];
        o.writeBytes(b, 0, 1024);
        o.close();
        IndexInput i = dir.openInput("out", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        i.seek(1024L);
        i.close();
        dir.close();
    }

    public void testSeekPastEOF() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testSeekPastEOF"));
        IndexOutput o = dir.createOutput("out", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int len = BaseDirectoryTestCase.random().nextInt(2048);
        byte[] b = new byte[len];
        o.writeBytes(b, 0, len);
        o.close();
        IndexInput i = dir.openInput("out", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.expectThrows(EOFException.class, () -> {
            i.seek((long)(len + BaseDirectoryTestCase.random().nextInt(2048)));
            i.readByte();
        });
        i.close();
        dir.close();
    }

    public void testSliceOutOfBounds() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testSliceOutOfBounds"));
        IndexOutput o = dir.createOutput("out", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int len = BaseDirectoryTestCase.random().nextInt(2040) + 8;
        byte[] b = new byte[len];
        o.writeBytes(b, 0, len);
        o.close();
        IndexInput i = dir.openInput("out", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.expectThrows(IllegalArgumentException.class, () -> i.slice("slice1", 0L, (long)(len + 1)));
        BaseDirectoryTestCase.expectThrows(IllegalArgumentException.class, () -> i.slice("slice2", -1L, (long)len));
        IndexInput slice = i.slice("slice3", 4L, (long)(len / 2));
        BaseDirectoryTestCase.expectThrows(IllegalArgumentException.class, () -> slice.slice("slice3sub", 1L, (long)(len / 2)));
        i.close();
        dir.close();
    }

    public void testNoDir() throws Throwable {
        Path tempDir = BaseDirectoryTestCase.createTempDir("doesnotexist");
        IOUtils.rm((Path[])new Path[]{tempDir});
        Directory dir = this.getDirectory(tempDir);
        try {
            DirectoryReader.open((Directory)dir);
            BaseDirectoryTestCase.fail((String)"did not hit expected exception");
        }
        catch (NoSuchFileException | IndexNotFoundException throwable) {
            // empty catch block
        }
        dir.close();
    }

    public void testCopyBytes() throws Exception {
        BaseDirectoryTestCase.testCopyBytes(this.getDirectory(BaseDirectoryTestCase.createTempDir("testCopyBytes")));
    }

    private static byte value(int idx) {
        return (byte)(idx % 256 * (1 + idx / 256));
    }

    public static void testCopyBytes(Directory dir) throws Exception {
        int upto;
        IndexOutput out = dir.createOutput("test", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        byte[] bytes = new byte[TestUtil.nextInt(BaseDirectoryTestCase.random(), 1, 77777)];
        int size = TestUtil.nextInt(BaseDirectoryTestCase.random(), 1, 1777777);
        int byteUpto = 0;
        for (upto = 0; upto < size; ++upto) {
            bytes[byteUpto++] = BaseDirectoryTestCase.value(upto);
            if (byteUpto != bytes.length) continue;
            out.writeBytes(bytes, 0, bytes.length);
            byteUpto = 0;
        }
        out.writeBytes(bytes, 0, byteUpto);
        BaseDirectoryTestCase.assertEquals((long)size, (long)out.getFilePointer());
        out.close();
        BaseDirectoryTestCase.assertEquals((long)size, (long)dir.fileLength("test"));
        IndexInput in = dir.openInput("test", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        out = dir.createOutput("test2", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        upto = 0;
        while (upto < size) {
            if (BaseDirectoryTestCase.random().nextBoolean()) {
                out.writeByte(in.readByte());
                ++upto;
                continue;
            }
            int chunk = Math.min(TestUtil.nextInt(BaseDirectoryTestCase.random(), 1, bytes.length), size - upto);
            out.copyBytes((DataInput)in, (long)chunk);
            upto += chunk;
        }
        BaseDirectoryTestCase.assertEquals((long)size, (long)upto);
        out.close();
        in.close();
        IndexInput in2 = dir.openInput("test2", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        upto = 0;
        while (upto < size) {
            if (BaseDirectoryTestCase.random().nextBoolean()) {
                byte v = in2.readByte();
                BaseDirectoryTestCase.assertEquals((long)BaseDirectoryTestCase.value(upto), (long)v);
                ++upto;
                continue;
            }
            int limit = Math.min(TestUtil.nextInt(BaseDirectoryTestCase.random(), 1, bytes.length), size - upto);
            in2.readBytes(bytes, 0, limit);
            for (int byteIdx = 0; byteIdx < limit; ++byteIdx) {
                BaseDirectoryTestCase.assertEquals((long)BaseDirectoryTestCase.value(upto), (long)bytes[byteIdx]);
                ++upto;
            }
        }
        in2.close();
        dir.deleteFile("test");
        dir.deleteFile("test2");
        dir.close();
    }

    public void testCopyBytesWithThreads() throws Exception {
        BaseDirectoryTestCase.testCopyBytesWithThreads(this.getDirectory(BaseDirectoryTestCase.createTempDir("testCopyBytesWithThreads")));
    }

    public static void testCopyBytesWithThreads(Directory d) throws Exception {
        int i;
        int datalen = TestUtil.nextInt(BaseDirectoryTestCase.random(), 101, 10000);
        byte[] data = new byte[datalen];
        BaseDirectoryTestCase.random().nextBytes(data);
        IndexOutput output = d.createOutput("data", IOContext.DEFAULT);
        output.writeBytes(data, 0, datalen);
        output.close();
        IndexInput input = d.openInput("data", IOContext.DEFAULT);
        IndexOutput outputHeader = d.createOutput("header", IOContext.DEFAULT);
        outputHeader.copyBytes((DataInput)input, 100L);
        outputHeader.close();
        CopyThread[] copies = new CopyThread[10];
        for (i = 0; i < copies.length; ++i) {
            copies[i] = new CopyThread(input.clone(), d.createOutput("copy" + i, IOContext.DEFAULT));
        }
        for (i = 0; i < copies.length; ++i) {
            copies[i].start();
        }
        for (i = 0; i < copies.length; ++i) {
            copies[i].join();
        }
        for (i = 0; i < copies.length; ++i) {
            IndexInput copiedData = d.openInput("copy" + i, IOContext.DEFAULT);
            byte[] dataCopy = new byte[datalen];
            System.arraycopy(data, 0, dataCopy, 0, 100);
            copiedData.readBytes(dataCopy, 100, datalen - 100);
            BaseDirectoryTestCase.assertArrayEquals((byte[])data, (byte[])dataCopy);
            copiedData.close();
        }
        input.close();
        d.close();
    }

    public void testFsyncDoesntCreateNewFiles() throws Exception {
        Path path = BaseDirectoryTestCase.createTempDir("nocreate");
        Directory fsdir = this.getDirectory(path);
        if (!(fsdir instanceof FSDirectory)) {
            fsdir.close();
            BaseDirectoryTestCase.assumeTrue("test only works for FSDirectory subclasses", false);
        }
        IndexOutput out = fsdir.createOutput("afile", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        out.writeString("boo");
        out.close();
        Files.delete(path.resolve("afile"));
        int fileCount = fsdir.listAll().length;
        try {
            fsdir.sync(Collections.singleton("afile"));
            BaseDirectoryTestCase.fail((String)("didn't get expected exception, instead fsync created new files: " + Arrays.asList(fsdir.listAll())));
        }
        catch (FileNotFoundException | NoSuchFileException iOException) {
            // empty catch block
        }
        BaseDirectoryTestCase.assertEquals((long)fileCount, (long)fsdir.listAll().length);
        fsdir.close();
    }

    public void testRandomLong() throws Exception {
        int i;
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testLongs"));
        IndexOutput output = dir.createOutput("longs", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int num = TestUtil.nextInt(BaseDirectoryTestCase.random(), 50, 3000);
        long[] longs = new long[num];
        for (int i2 = 0; i2 < longs.length; ++i2) {
            longs[i2] = TestUtil.nextLong(BaseDirectoryTestCase.random(), Long.MIN_VALUE, Long.MAX_VALUE);
            output.writeLong(longs[i2]);
        }
        output.close();
        IndexInput input = dir.openInput("longs", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        RandomAccessInput slice = input.randomAccessSlice(0L, input.length());
        for (i = 0; i < longs.length; ++i) {
            BaseDirectoryTestCase.assertEquals((long)longs[i], (long)slice.readLong((long)(i * 8)));
        }
        for (i = 1; i < longs.length; ++i) {
            long offset = i * 8;
            RandomAccessInput subslice = input.randomAccessSlice(offset, input.length() - offset);
            for (int j = i; j < longs.length; ++j) {
                BaseDirectoryTestCase.assertEquals((long)longs[j], (long)subslice.readLong((long)((j - i) * 8)));
            }
        }
        for (i = 0; i < 7; ++i) {
            String name = "longs-" + i;
            IndexOutput o = dir.createOutput(name, BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            byte[] junk = new byte[i];
            BaseDirectoryTestCase.random().nextBytes(junk);
            o.writeBytes(junk, junk.length);
            input.seek(0L);
            o.copyBytes((DataInput)input, input.length());
            o.close();
            IndexInput padded = dir.openInput(name, BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            RandomAccessInput whole = padded.randomAccessSlice((long)i, padded.length() - (long)i);
            for (int j = 0; j < longs.length; ++j) {
                BaseDirectoryTestCase.assertEquals((long)longs[j], (long)whole.readLong((long)(j * 8)));
            }
            padded.close();
        }
        input.close();
        dir.close();
    }

    public void testRandomInt() throws Exception {
        int i;
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testInts"));
        IndexOutput output = dir.createOutput("ints", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int num = TestUtil.nextInt(BaseDirectoryTestCase.random(), 50, 3000);
        int[] ints = new int[num];
        for (int i2 = 0; i2 < ints.length; ++i2) {
            ints[i2] = BaseDirectoryTestCase.random().nextInt();
            output.writeInt(ints[i2]);
        }
        output.close();
        IndexInput input = dir.openInput("ints", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        RandomAccessInput slice = input.randomAccessSlice(0L, input.length());
        for (i = 0; i < ints.length; ++i) {
            BaseDirectoryTestCase.assertEquals((long)ints[i], (long)slice.readInt((long)(i * 4)));
        }
        for (i = 1; i < ints.length; ++i) {
            long offset = i * 4;
            RandomAccessInput subslice = input.randomAccessSlice(offset, input.length() - offset);
            for (int j = i; j < ints.length; ++j) {
                BaseDirectoryTestCase.assertEquals((long)ints[j], (long)subslice.readInt((long)((j - i) * 4)));
            }
        }
        for (i = 0; i < 7; ++i) {
            String name = "ints-" + i;
            IndexOutput o = dir.createOutput(name, BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            byte[] junk = new byte[i];
            BaseDirectoryTestCase.random().nextBytes(junk);
            o.writeBytes(junk, junk.length);
            input.seek(0L);
            o.copyBytes((DataInput)input, input.length());
            o.close();
            IndexInput padded = dir.openInput(name, BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            RandomAccessInput whole = padded.randomAccessSlice((long)i, padded.length() - (long)i);
            for (int j = 0; j < ints.length; ++j) {
                BaseDirectoryTestCase.assertEquals((long)ints[j], (long)whole.readInt((long)(j * 4)));
            }
            padded.close();
        }
        input.close();
        dir.close();
    }

    public void testRandomShort() throws Exception {
        int i;
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testShorts"));
        IndexOutput output = dir.createOutput("shorts", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int num = TestUtil.nextInt(BaseDirectoryTestCase.random(), 50, 3000);
        short[] shorts = new short[num];
        for (int i2 = 0; i2 < shorts.length; ++i2) {
            shorts[i2] = (short)BaseDirectoryTestCase.random().nextInt();
            output.writeShort(shorts[i2]);
        }
        output.close();
        IndexInput input = dir.openInput("shorts", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        RandomAccessInput slice = input.randomAccessSlice(0L, input.length());
        for (i = 0; i < shorts.length; ++i) {
            BaseDirectoryTestCase.assertEquals((long)shorts[i], (long)slice.readShort((long)(i * 2)));
        }
        for (i = 1; i < shorts.length; ++i) {
            long offset = i * 2;
            RandomAccessInput subslice = input.randomAccessSlice(offset, input.length() - offset);
            for (int j = i; j < shorts.length; ++j) {
                BaseDirectoryTestCase.assertEquals((long)shorts[j], (long)subslice.readShort((long)((j - i) * 2)));
            }
        }
        for (i = 0; i < 7; ++i) {
            String name = "shorts-" + i;
            IndexOutput o = dir.createOutput(name, BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            byte[] junk = new byte[i];
            BaseDirectoryTestCase.random().nextBytes(junk);
            o.writeBytes(junk, junk.length);
            input.seek(0L);
            o.copyBytes((DataInput)input, input.length());
            o.close();
            IndexInput padded = dir.openInput(name, BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            RandomAccessInput whole = padded.randomAccessSlice((long)i, padded.length() - (long)i);
            for (int j = 0; j < shorts.length; ++j) {
                BaseDirectoryTestCase.assertEquals((long)shorts[j], (long)whole.readShort((long)(j * 2)));
            }
            padded.close();
        }
        input.close();
        dir.close();
    }

    public void testRandomByte() throws Exception {
        int i;
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("testBytes"));
        IndexOutput output = dir.createOutput("bytes", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int num = TestUtil.nextInt(BaseDirectoryTestCase.random(), 50, 3000);
        byte[] bytes = new byte[num];
        BaseDirectoryTestCase.random().nextBytes(bytes);
        for (int i2 = 0; i2 < bytes.length; ++i2) {
            output.writeByte(bytes[i2]);
        }
        output.close();
        IndexInput input = dir.openInput("bytes", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        RandomAccessInput slice = input.randomAccessSlice(0L, input.length());
        for (i = 0; i < bytes.length; ++i) {
            BaseDirectoryTestCase.assertEquals((long)bytes[i], (long)slice.readByte((long)i));
        }
        for (i = 1; i < bytes.length; ++i) {
            long offset = i;
            RandomAccessInput subslice = input.randomAccessSlice(offset, input.length() - offset);
            for (int j = i; j < bytes.length; ++j) {
                BaseDirectoryTestCase.assertEquals((long)bytes[j], (long)subslice.readByte((long)(j - i)));
            }
        }
        for (i = 0; i < 7; ++i) {
            String name = "bytes-" + i;
            IndexOutput o = dir.createOutput(name, BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            byte[] junk = new byte[i];
            BaseDirectoryTestCase.random().nextBytes(junk);
            o.writeBytes(junk, junk.length);
            input.seek(0L);
            o.copyBytes((DataInput)input, input.length());
            o.close();
            IndexInput padded = dir.openInput(name, BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            RandomAccessInput whole = padded.randomAccessSlice((long)i, padded.length() - (long)i);
            for (int j = 0; j < bytes.length; ++j) {
                BaseDirectoryTestCase.assertEquals((long)bytes[j], (long)whole.readByte((long)j));
            }
            padded.close();
        }
        input.close();
        dir.close();
    }

    public void testSliceOfSlice() throws Exception {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("sliceOfSlice"));
        IndexOutput output = dir.createOutput("bytes", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        int num = TEST_NIGHTLY ? TestUtil.nextInt(BaseDirectoryTestCase.random(), 250, 2500) : TestUtil.nextInt(BaseDirectoryTestCase.random(), 50, 250);
        byte[] bytes = new byte[num];
        BaseDirectoryTestCase.random().nextBytes(bytes);
        for (int i = 0; i < bytes.length; ++i) {
            output.writeByte(bytes[i]);
        }
        output.close();
        IndexInput input = dir.openInput("bytes", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        input.seek(TestUtil.nextLong(BaseDirectoryTestCase.random(), 0L, input.length()));
        for (int i = 0; i < num; i += 16) {
            IndexInput slice1 = input.slice("slice1", (long)i, (long)(num - i));
            BaseDirectoryTestCase.assertEquals((long)0L, (long)slice1.getFilePointer());
            BaseDirectoryTestCase.assertEquals((long)(num - i), (long)slice1.length());
            slice1.seek(TestUtil.nextLong(BaseDirectoryTestCase.random(), 0L, slice1.length()));
            int j = 0;
            while ((long)j < slice1.length()) {
                IndexInput slice2 = slice1.slice("slice2", (long)j, (long)(num - i - j));
                BaseDirectoryTestCase.assertEquals((long)0L, (long)slice2.getFilePointer());
                BaseDirectoryTestCase.assertEquals((long)(num - i - j), (long)slice2.length());
                byte[] data = new byte[num];
                System.arraycopy(bytes, 0, data, 0, i + j);
                if (BaseDirectoryTestCase.random().nextBoolean()) {
                    slice2.readBytes(data, i + j, num - i - j);
                } else {
                    long seek = TestUtil.nextLong(BaseDirectoryTestCase.random(), 0L, slice2.length());
                    slice2.seek(seek);
                    slice2.readBytes(data, (int)((long)(i + j) + seek), (int)((long)(num - i - j) - seek));
                    slice2.seek(0L);
                    slice2.readBytes(data, i + j, (int)seek);
                }
                BaseDirectoryTestCase.assertArrayEquals((byte[])bytes, (byte[])data);
                j += 16;
            }
        }
        input.close();
        dir.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testLargeWrites() throws IOException {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir("largeWrites"));
        IndexOutput os = dir.createOutput("testBufferStart.txt", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        byte[] largeBuf = new byte[2048];
        BaseDirectoryTestCase.random().nextBytes(largeBuf);
        long currentPos = os.getFilePointer();
        os.writeBytes(largeBuf, largeBuf.length);
        try {
            BaseDirectoryTestCase.assertEquals((long)(currentPos + (long)largeBuf.length), (long)os.getFilePointer());
        }
        finally {
            os.close();
        }
        dir.close();
    }

    public void testIndexOutputToString() throws Throwable {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir());
        IndexOutput out = dir.createOutput("camelCase.txt", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertTrue((String)out.toString(), (boolean)out.toString().contains("camelCase.txt"));
        out.close();
        dir.close();
    }

    public void testDoubleCloseDirectory() throws Throwable {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir());
        IndexOutput out = dir.createOutput("foobar", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        out.writeString("testing");
        out.close();
        dir.close();
        dir.close();
    }

    public void testDoubleCloseOutput() throws Throwable {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir());
        IndexOutput out = dir.createOutput("foobar", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        out.writeString("testing");
        out.close();
        out.close();
        dir.close();
    }

    public void testDoubleCloseInput() throws Throwable {
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir());
        IndexOutput out = dir.createOutput("foobar", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        out.writeString("testing");
        out.close();
        IndexInput in = dir.openInput("foobar", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
        BaseDirectoryTestCase.assertEquals((Object)"testing", (Object)in.readString());
        in.close();
        in.close();
        dir.close();
    }

    public void testCreateTempOutput() throws Throwable {
        int iter;
        Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir());
        ArrayList<String> names = new ArrayList<String>();
        int iters = BaseDirectoryTestCase.atLeast(50);
        for (iter = 0; iter < iters; ++iter) {
            IndexOutput out = dir.createTempOutput("foo", "bar", BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            names.add(out.getName());
            out.writeVInt(iter);
            out.close();
        }
        for (iter = 0; iter < iters; ++iter) {
            IndexInput in = dir.openInput((String)names.get(iter), BaseDirectoryTestCase.newIOContext(BaseDirectoryTestCase.random()));
            BaseDirectoryTestCase.assertEquals((long)iter, (long)in.readVInt());
            in.close();
        }
        HashSet<String> files = new HashSet<String>(Arrays.asList(dir.listAll()));
        files.remove("extra0");
        BaseDirectoryTestCase.assertEquals(new HashSet(names), files);
        dir.close();
    }

    public void testSeekToEndOfFile() throws IOException {
        try (Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir());){
            try (IndexOutput out = dir.createOutput("a", IOContext.DEFAULT);){
                for (int i = 0; i < 1024; ++i) {
                    out.writeByte((byte)0);
                }
            }
            var4_6 = null;
            try (IndexInput in = dir.openInput("a", IOContext.DEFAULT);){
                in.seek(100L);
                BaseDirectoryTestCase.assertEquals((long)100L, (long)in.getFilePointer());
                in.seek(1024L);
                BaseDirectoryTestCase.assertEquals((long)1024L, (long)in.getFilePointer());
            }
            catch (Throwable throwable) {
                var4_6 = throwable;
                throw throwable;
            }
        }
    }

    public void testSeekBeyondEndOfFile() throws IOException {
        try (Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir());){
            try (IndexOutput out = dir.createOutput("a", IOContext.DEFAULT);){
                for (int i = 0; i < 1024; ++i) {
                    out.writeByte((byte)0);
                }
            }
            var4_6 = null;
            try (IndexInput in = dir.openInput("a", IOContext.DEFAULT);){
                in.seek(100L);
                BaseDirectoryTestCase.assertEquals((long)100L, (long)in.getFilePointer());
                BaseDirectoryTestCase.expectThrows(EOFException.class, () -> in.seek(1025L));
            }
            catch (Throwable throwable) {
                var4_6 = throwable;
                throw throwable;
            }
        }
    }

    public void testPendingDeletions() throws IOException {
        try (Directory dir = this.getDirectory(BaseDirectoryTestCase.addVirusChecker(BaseDirectoryTestCase.createTempDir()));){
            String candidate;
            BaseDirectoryTestCase.assumeTrue("we can only install VirusCheckingFS on an FSDirectory", dir instanceof FSDirectory);
            FSDirectory fsDir = (FSDirectory)dir;
            do {
                candidate = IndexFileNames.segmentFileName((String)TestUtil.randomSimpleString(BaseDirectoryTestCase.random(), 1, 6), (String)TestUtil.randomSimpleString(BaseDirectoryTestCase.random()), (String)"test");
                try (IndexOutput out = dir.createOutput(candidate, IOContext.DEFAULT);){
                    out.getFilePointer();
                }
                fsDir.deleteFile(candidate);
            } while (fsDir.getPendingDeletions().size() <= 0);
            String fileName = candidate;
            BaseDirectoryTestCase.assertFalse((boolean)Arrays.asList(fsDir.listAll()).contains(fileName));
            BaseDirectoryTestCase.expectThrows(NoSuchFileException.class, () -> fsDir.fileLength(fileName));
            BaseDirectoryTestCase.expectThrows(NoSuchFileException.class, () -> fsDir.rename(fileName, "file2"));
            BaseDirectoryTestCase.expectThrows(NoSuchFileException.class, () -> fsDir.deleteFile(fileName));
            BaseDirectoryTestCase.expectThrows(NoSuchFileException.class, () -> fsDir.openInput(fileName, IOContext.DEFAULT));
        }
    }

    public void testListAllIsSorted() throws IOException {
        try (Directory dir = this.getDirectory(BaseDirectoryTestCase.createTempDir());){
            int count = BaseDirectoryTestCase.atLeast(20);
            HashSet<String> names = new HashSet<String>();
            while (names.size() < count) {
                IndexOutput out;
                String name = IndexFileNames.segmentFileName((String)TestUtil.randomSimpleString(BaseDirectoryTestCase.random(), 1, 6), (String)TestUtil.randomSimpleString(BaseDirectoryTestCase.random()), (String)"test");
                if (BaseDirectoryTestCase.random().nextInt(5) == 1) {
                    out = dir.createTempOutput(name, "foo", IOContext.DEFAULT);
                    names.add(out.getName());
                    out.close();
                    continue;
                }
                if (names.contains(name)) continue;
                out = dir.createOutput(name, IOContext.DEFAULT);
                names.add(out.getName());
                out.close();
            }
            Object[] actual = dir.listAll();
            Object[] expected = (String[])actual.clone();
            Arrays.sort(expected);
            BaseDirectoryTestCase.assertEquals((Object[])expected, (Object[])actual);
        }
    }

    static class CopyThread
    extends Thread {
        final IndexInput src;
        final IndexOutput dst;

        CopyThread(IndexInput src, IndexOutput dst) {
            this.src = src;
            this.dst = dst;
        }

        @Override
        public void run() {
            try {
                this.dst.copyBytes((DataInput)this.src, this.src.length() - 100L);
                this.dst.close();
            }
            catch (IOException ex) {
                throw new RuntimeException(ex);
            }
        }
    }
}

