/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file;

import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import org.apache.jackrabbit.oak.api.jmx.CacheStatsMBean;
import org.apache.jackrabbit.oak.segment.CachingSegmentReader;
import org.apache.jackrabbit.oak.segment.RecordType;
import org.apache.jackrabbit.oak.segment.Revisions;
import org.apache.jackrabbit.oak.segment.Segment;
import org.apache.jackrabbit.oak.segment.SegmentBlob;
import org.apache.jackrabbit.oak.segment.SegmentCache;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.SegmentIdFactory;
import org.apache.jackrabbit.oak.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.segment.SegmentReader;
import org.apache.jackrabbit.oak.segment.SegmentStore;
import org.apache.jackrabbit.oak.segment.SegmentTracker;
import org.apache.jackrabbit.oak.segment.SegmentWriter;
import org.apache.jackrabbit.oak.segment.file.FileStoreBuilder;
import org.apache.jackrabbit.oak.segment.file.InvalidFileStoreVersionException;
import org.apache.jackrabbit.oak.segment.file.Manifest;
import org.apache.jackrabbit.oak.segment.file.TarRecovery;
import org.apache.jackrabbit.oak.segment.file.TarWriter;
import org.apache.jackrabbit.oak.spi.blob.BlobStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractFileStore
implements SegmentStore,
Closeable {
    private static final Logger log = LoggerFactory.getLogger(AbstractFileStore.class);
    private static final String MANIFEST_FILE_NAME = "manifest";
    private static final int INVALID_STORE_VERSION = 0;
    static final int CURRENT_STORE_VERSION = 1;
    private static final Pattern FILE_NAME_PATTERN = Pattern.compile("(data|bulk)((0|[1-9][0-9]*)[0-9]{4})([a-z])?.tar");
    static final String FILE_NAME_FORMAT = "data%05d%s.tar";
    @Nonnull
    final SegmentTracker tracker;
    @Nonnull
    final CachingSegmentReader segmentReader;
    final File directory;
    private final BlobStore blobStore;
    final boolean memoryMapping;
    @Nonnull
    final SegmentCache segmentCache;
    final TarRecovery recovery = new TarRecovery(){

        @Override
        public void recoverEntry(UUID uuid, byte[] data, TarWriter writer) throws IOException {
            AbstractFileStore.this.writeSegment(uuid, data, writer);
        }
    };
    @Nonnull
    private final SegmentIdFactory segmentIdFactory = new SegmentIdFactory(){

        @Override
        @Nonnull
        public SegmentId newSegmentId(long msb, long lsb) {
            return new SegmentId(AbstractFileStore.this, msb, lsb);
        }
    };

    AbstractFileStore(FileStoreBuilder builder) {
        this.directory = builder.getDirectory();
        this.tracker = new SegmentTracker();
        this.blobStore = builder.getBlobStore();
        this.segmentCache = new SegmentCache(builder.getSegmentCacheSize());
        this.segmentReader = new CachingSegmentReader(new Supplier<SegmentWriter>(){

            public SegmentWriter get() {
                return AbstractFileStore.this.getWriter();
            }
        }, this.blobStore, builder.getStringCacheSize(), builder.getTemplateCacheSize());
        this.memoryMapping = builder.getMemoryMapping();
    }

    File getManifestFile() {
        return new File(this.directory, MANIFEST_FILE_NAME);
    }

    Manifest openManifest() throws IOException {
        File file = this.getManifestFile();
        if (file.exists()) {
            return Manifest.load(file);
        }
        return null;
    }

    static Manifest checkManifest(Manifest manifest) throws InvalidFileStoreVersionException {
        if (manifest == null) {
            throw new InvalidFileStoreVersionException("Using oak-segment-tar, but oak-segment should be used");
        }
        int storeVersion = manifest.getStoreVersion(0);
        if (storeVersion <= 0) {
            throw new IllegalStateException("Invalid store version");
        }
        if (storeVersion < 1) {
            throw new InvalidFileStoreVersionException("Using a too recent version of oak-segment-tar");
        }
        if (storeVersion > 1) {
            throw new InvalidFileStoreVersionException("Using a too old version of oak-segment tar");
        }
        return manifest;
    }

    @Nonnull
    public CacheStatsMBean getSegmentCacheStats() {
        return this.segmentCache.getCacheStats();
    }

    @Nonnull
    public CacheStatsMBean getStringCacheStats() {
        return this.segmentReader.getStringCacheStats();
    }

    @Nonnull
    public CacheStatsMBean getTemplateCacheStats() {
        return this.segmentReader.getTemplateCacheStats();
    }

    static Map<Integer, Map<Character, File>> collectFiles(File directory) {
        Map files;
        HashMap dataFiles = Maps.newHashMap();
        HashMap bulkFiles = Maps.newHashMap();
        for (File file : AbstractFileStore.listFiles(directory)) {
            Matcher matcher = FILE_NAME_PATTERN.matcher(file.getName());
            if (!matcher.matches()) continue;
            Object index = Integer.parseInt(matcher.group(2));
            if ("data".equals(matcher.group(1))) {
                files = (Map)dataFiles.get(index);
                if (files == null) {
                    files = Maps.newHashMap();
                    dataFiles.put(index, files);
                }
                Character generation = Character.valueOf('a');
                if (matcher.group(4) != null) {
                    generation = Character.valueOf(matcher.group(4).charAt(0));
                }
                Preconditions.checkState((files.put(generation, file) == null ? 1 : 0) != 0);
                continue;
            }
            Preconditions.checkState((bulkFiles.put(index, file) == null ? 1 : 0) != 0);
        }
        if (!bulkFiles.isEmpty()) {
            Integer newIndex;
            int position;
            Object[] indices;
            log.info("Upgrading TarMK file names in {}", (Object)directory);
            if (!dataFiles.isEmpty()) {
                indices = dataFiles.keySet().toArray(new Integer[dataFiles.size()]);
                Arrays.sort(indices);
                position = Math.max((Integer)indices[indices.length - 1] + 1, bulkFiles.size());
                for (Object index : indices) {
                    files = (Map)dataFiles.remove(index);
                    newIndex = position++;
                    for (Character generation : Sets.newHashSet(files.keySet())) {
                        File file = (File)files.get(generation);
                        File newFile = new File(directory, String.format(FILE_NAME_FORMAT, newIndex, generation));
                        log.info("Renaming {} to {}", (Object)file, (Object)newFile);
                        file.renameTo(newFile);
                        files.put(generation, newFile);
                    }
                    dataFiles.put(newIndex, files);
                }
            }
            indices = bulkFiles.keySet().toArray(new Integer[bulkFiles.size()]);
            Arrays.sort(indices);
            position = 0;
            for (Object index : indices) {
                File file = (File)bulkFiles.remove(index);
                newIndex = position++;
                File newFile = new File(directory, String.format(FILE_NAME_FORMAT, newIndex, "a"));
                log.info("Renaming {} to {}", (Object)file, (Object)newFile);
                file.renameTo(newFile);
                dataFiles.put(newIndex, Collections.singletonMap(Character.valueOf('a'), newFile));
            }
        }
        return dataFiles;
    }

    @Nonnull
    private static File[] listFiles(File directory) {
        File[] files = directory.listFiles();
        return files == null ? new File[]{} : files;
    }

    @Nonnull
    public SegmentTracker getTracker() {
        return this.tracker;
    }

    @Nonnull
    public abstract SegmentWriter getWriter();

    @Nonnull
    public SegmentReader getReader() {
        return this.segmentReader;
    }

    public abstract Revisions getRevisions();

    @Nonnull
    public SegmentNodeState getHead() {
        return this.segmentReader.readHeadState(this.getRevisions());
    }

    @Override
    @Nonnull
    public SegmentId newSegmentId(long msb, long lsb) {
        return this.tracker.newSegmentId(msb, lsb, this.segmentIdFactory);
    }

    @Override
    @Nonnull
    public SegmentId newBulkSegmentId() {
        return this.tracker.newBulkSegmentId(this.segmentIdFactory);
    }

    @Override
    @Nonnull
    public SegmentId newDataSegmentId() {
        return this.tracker.newDataSegmentId(this.segmentIdFactory);
    }

    @CheckForNull
    public BlobStore getBlobStore() {
        return this.blobStore;
    }

    private void writeSegment(UUID id, byte[] data, TarWriter w) throws IOException {
        long msb = id.getMostSignificantBits();
        long lsb = id.getLeastSignificantBits();
        ByteBuffer buffer = ByteBuffer.wrap(data);
        int generation = Segment.getGcGeneration(buffer, id);
        w.writeEntry(msb, lsb, data, 0, data.length, generation);
        if (SegmentId.isDataSegmentId(lsb)) {
            Segment segment = new Segment(this, this.segmentReader, this.newSegmentId(msb, lsb), buffer);
            AbstractFileStore.populateTarGraph(segment, w);
            AbstractFileStore.populateTarBinaryReferences(segment, w);
        }
    }

    static void populateTarGraph(Segment segment, TarWriter w) {
        UUID from = segment.getSegmentId().asUUID();
        for (int i = 0; i < segment.getReferencedSegmentIdCount(); ++i) {
            w.addGraphEdge(from, segment.getReferencedSegmentId(i));
        }
    }

    static void populateTarBinaryReferences(final Segment segment, final TarWriter w) {
        final int generation = segment.getGcGeneration();
        final UUID id = segment.getSegmentId().asUUID();
        segment.forEachRecord(new Segment.RecordConsumer(){

            @Override
            public void consume(int number, RecordType type, int offset) {
                if (type == RecordType.BLOB_ID) {
                    w.addBinaryReference(generation, id, SegmentBlob.readBlobId(segment, number));
                }
            }
        });
    }

    static void closeAndLogOnFail(Closeable closeable) {
        if (closeable != null) {
            try {
                closeable.close();
            }
            catch (IOException ioe) {
                log.error(ioe.getMessage(), (Throwable)ioe);
            }
        }
    }
}

