/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Collection;
import java.util.HashSet;
import org.apache.commons.io.HexDump;
import org.apache.jackrabbit.oak.segment.BinaryUtils;
import org.apache.jackrabbit.oak.segment.MutableRecordNumbers;
import org.apache.jackrabbit.oak.segment.MutableSegmentReferences;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.RecordNumbers;
import org.apache.jackrabbit.oak.segment.RecordType;
import org.apache.jackrabbit.oak.segment.RecordWriters;
import org.apache.jackrabbit.oak.segment.Segment;
import org.apache.jackrabbit.oak.segment.SegmentDump;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.SegmentIdProvider;
import org.apache.jackrabbit.oak.segment.SegmentReader;
import org.apache.jackrabbit.oak.segment.SegmentStore;
import org.apache.jackrabbit.oak.segment.SegmentVersion;
import org.apache.jackrabbit.oak.segment.WriteOperationHandler;
import org.apache.jackrabbit.oak.segment.file.tar.GCGeneration;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SegmentBufferWriter
implements WriteOperationHandler {
    private static final Logger LOG = LoggerFactory.getLogger(SegmentBufferWriter.class);
    private MutableRecordNumbers recordNumbers = new MutableRecordNumbers();
    private MutableSegmentReferences segmentReferences = new MutableSegmentReferences();
    @NotNull
    private final SegmentIdProvider idProvider;
    @NotNull
    private final SegmentReader reader;
    @NotNull
    private final String wid;
    @NotNull
    private final GCGeneration gcGeneration;
    private byte[] buffer;
    private Segment segment;
    private int length;
    private int position;
    private Statistics statistics;
    private boolean dirty;

    public SegmentBufferWriter(@NotNull SegmentIdProvider idProvider, @NotNull SegmentReader reader, @Nullable String wid, @NotNull GCGeneration gcGeneration) {
        this.idProvider = (SegmentIdProvider)Preconditions.checkNotNull((Object)idProvider);
        this.reader = (SegmentReader)Preconditions.checkNotNull((Object)reader);
        this.wid = wid == null ? "w-" + System.identityHashCode(this) : wid;
        this.gcGeneration = (GCGeneration)Preconditions.checkNotNull((Object)gcGeneration);
    }

    @Override
    @NotNull
    public RecordId execute(@NotNull GCGeneration gcGeneration, @NotNull WriteOperationHandler.WriteOperation writeOperation) throws IOException {
        Preconditions.checkState((boolean)gcGeneration.equals(this.gcGeneration));
        return writeOperation.execute(this);
    }

    @Override
    @NotNull
    public GCGeneration getGCGeneration() {
        return this.gcGeneration;
    }

    private void newSegment(SegmentStore store) throws IOException {
        this.buffer = new byte[262144];
        this.buffer[0] = 48;
        this.buffer[1] = 97;
        this.buffer[2] = 75;
        this.buffer[3] = SegmentVersion.asByte(SegmentVersion.LATEST_VERSION);
        this.buffer[4] = 0;
        this.buffer[5] = 0;
        int generation = this.gcGeneration.getGeneration();
        this.buffer[10] = (byte)(generation >> 24);
        this.buffer[11] = (byte)(generation >> 16);
        this.buffer[12] = (byte)(generation >> 8);
        this.buffer[13] = (byte)generation;
        int fullGeneration = this.gcGeneration.getFullGeneration();
        if (this.gcGeneration.isCompacted()) {
            fullGeneration |= Integer.MIN_VALUE;
        }
        this.buffer[4] = (byte)(fullGeneration >> 24);
        this.buffer[5] = (byte)(fullGeneration >> 16);
        this.buffer[6] = (byte)(fullGeneration >> 8);
        this.buffer[7] = (byte)fullGeneration;
        this.length = 0;
        this.position = this.buffer.length;
        this.recordNumbers = new MutableRecordNumbers();
        this.segmentReferences = new MutableSegmentReferences();
        String metaInfo = "{\"wid\":\"" + this.wid + '\"' + ",\"sno\":" + this.idProvider.getSegmentIdCount() + ",\"t\":" + System.currentTimeMillis() + "}";
        this.segment = new Segment(this.idProvider.newDataSegmentId(), this.reader, this.buffer, this.recordNumbers, this.segmentReferences, metaInfo);
        this.statistics = new Statistics();
        this.statistics.id = this.segment.getSegmentId();
        byte[] data = metaInfo.getBytes(Charsets.UTF_8);
        RecordWriters.newValueWriter(data.length, data).write(this, store);
        this.dirty = false;
    }

    public void writeByte(byte value) {
        this.position = BinaryUtils.writeByte(this.buffer, this.position, value);
        this.dirty = true;
    }

    public void writeShort(short value) {
        this.position = BinaryUtils.writeShort(this.buffer, this.position, value);
        this.dirty = true;
    }

    public void writeInt(int value) {
        this.position = BinaryUtils.writeInt(this.buffer, this.position, value);
        this.dirty = true;
    }

    public void writeLong(long value) {
        this.position = BinaryUtils.writeLong(this.buffer, this.position, value);
        this.dirty = true;
    }

    public void writeRecordId(RecordId recordId) {
        Preconditions.checkNotNull((Object)recordId);
        Preconditions.checkState((this.segmentReferences.size() + 1 < 65535 ? 1 : 0) != 0, (Object)"Segment cannot have more than 0xffff references");
        this.writeShort(SegmentBufferWriter.toShort(this.writeSegmentIdReference(recordId.getSegmentId())));
        this.writeInt(recordId.getRecordNumber());
        ++this.statistics.recordIdCount;
        this.dirty = true;
    }

    private static short toShort(int value) {
        return (short)value;
    }

    private int writeSegmentIdReference(SegmentId id) {
        if (id.equals(this.segment.getSegmentId())) {
            return 0;
        }
        return this.segmentReferences.addOrReference(id);
    }

    private static String info(Segment segment) {
        String info = segment.getSegmentId().toString();
        if (SegmentId.isDataSegmentId(segment.getSegmentId().getLeastSignificantBits())) {
            info = info + " " + segment.getSegmentInfo();
        }
        return info;
    }

    public void writeBytes(byte[] data, int offset, int length) {
        System.arraycopy(data, offset, this.buffer, this.position, length);
        this.position += length;
        this.dirty = true;
    }

    private String dumpSegmentBuffer() {
        return SegmentDump.dumpSegment(this.segment != null ? this.segment.getSegmentId() : null, this.length, this.segment != null ? this.segment.getSegmentInfo() : null, this.gcGeneration, this.segmentReferences, this.recordNumbers, stream -> {
            try {
                HexDump.dump((byte[])this.buffer, (long)0L, (OutputStream)stream, (int)0);
            }
            catch (IOException e) {
                e.printStackTrace(new PrintStream((OutputStream)stream));
            }
        });
    }

    @Override
    public void flush(@NotNull SegmentStore store) throws IOException {
        if (this.dirty) {
            int referencedSegmentIdCount = this.segmentReferences.size();
            BinaryUtils.writeInt(this.buffer, 14, referencedSegmentIdCount);
            this.statistics.segmentIdCount = referencedSegmentIdCount;
            int recordNumberCount = this.recordNumbers.size();
            BinaryUtils.writeInt(this.buffer, 18, recordNumberCount);
            int totalLength = Segment.align(32 + referencedSegmentIdCount * 16 + recordNumberCount * 9 + this.length, 16);
            if (totalLength > this.buffer.length) {
                LOG.warn("Segment buffer corruption detected\n{}", (Object)this.dumpSegmentBuffer());
                throw new IllegalStateException(String.format("Too much data for a segment %s (referencedSegmentIdCount=%d, recordNumberCount=%d, length=%d, totalLength=%d)", this.segment.getSegmentId(), referencedSegmentIdCount, recordNumberCount, this.length, totalLength));
            }
            this.statistics.size = this.length = totalLength;
            int pos = 32;
            if (pos + this.length <= this.buffer.length) {
                System.arraycopy(this.buffer, 0, this.buffer, this.buffer.length - this.length, pos);
                pos += this.buffer.length - this.length;
            } else {
                this.length = this.buffer.length;
            }
            for (SegmentId segmentId : this.segmentReferences) {
                pos = BinaryUtils.writeLong(this.buffer, pos, segmentId.getMostSignificantBits());
                pos = BinaryUtils.writeLong(this.buffer, pos, segmentId.getLeastSignificantBits());
            }
            for (RecordNumbers.Entry entry : this.recordNumbers) {
                pos = BinaryUtils.writeInt(this.buffer, pos, entry.getRecordNumber());
                pos = BinaryUtils.writeByte(this.buffer, pos, (byte)entry.getType().ordinal());
                pos = BinaryUtils.writeInt(this.buffer, pos, entry.getOffset());
            }
            SegmentId segmentId = this.segment.getSegmentId();
            LOG.debug("Writing data segment: {} ", (Object)this.statistics);
            store.writeSegment(segmentId, this.buffer, this.buffer.length - this.length, this.length);
            this.newSegment(store);
        }
    }

    public RecordId prepare(RecordType type, int size, Collection<RecordId> ids, SegmentStore store) throws IOException {
        Preconditions.checkArgument((size >= 0 ? 1 : 0) != 0);
        Preconditions.checkNotNull(ids);
        if (this.segment == null) {
            this.newSegment(store);
        }
        int idCount = ids.size();
        int recordSize = Segment.align(size + idCount * 6, 4);
        int recordNumbersCount = this.recordNumbers.size() + 1;
        int referencedIdCount = this.segmentReferences.size() + ids.size();
        int headerSize = 32 + referencedIdCount * 16 + recordNumbersCount * 9;
        int segmentSize = Segment.align(headerSize + recordSize + this.length, 16);
        if (segmentSize > this.buffer.length) {
            HashSet segmentIds = Sets.newHashSet();
            for (RecordId recordId : ids) {
                SegmentId segmentId = recordId.getSegmentId();
                if (this.segmentReferences.contains(segmentId)) continue;
                segmentIds.add(segmentId);
            }
            referencedIdCount = this.segmentReferences.size() + segmentIds.size();
            headerSize = 32 + referencedIdCount * 16 + recordNumbersCount * 9;
            segmentSize = Segment.align(headerSize + recordSize + this.length, 16);
        }
        if (segmentSize > this.buffer.length) {
            if (this.dirty) {
                LOG.debug("Flushing full segment {} (headerSize={}, recordSize={}, length={}, segmentSize={})", new Object[]{this.segment.getSegmentId(), headerSize, recordSize, this.length, segmentSize});
                this.flush(store);
                return this.prepare(type, size, ids, store);
            }
            throw new IllegalArgumentException(String.format("Record too big: type=%s, size=%s, recordIds=%s, total=%s", new Object[]{type, size, ids.size(), recordSize}));
        }
        ++this.statistics.recordCount;
        this.length += recordSize;
        this.position = this.buffer.length - this.length;
        int recordNumber = this.recordNumbers.addRecord(type, this.position);
        return new RecordId(this.segment.getSegmentId(), recordNumber);
    }

    private static final class Statistics {
        int segmentIdCount;
        int recordIdCount;
        int recordCount;
        int size;
        SegmentId id;

        private Statistics() {
        }

        public String toString() {
            return "id=" + this.id + ",size=" + this.size + ",segmentIdCount=" + this.segmentIdCount + ",recordIdCount=" + this.recordIdCount + ",recordCount=" + this.recordCount;
        }
    }
}

