/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.storageengine.dataregion.compaction.execute.utils;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.iotdb.commons.conf.CommonDescriptor;
import org.apache.iotdb.commons.service.metric.MetricService;
import org.apache.iotdb.commons.service.metric.enums.Tag;
import org.apache.iotdb.db.service.metrics.CompactionMetrics;
import org.apache.iotdb.db.service.metrics.FileMetrics;
import org.apache.iotdb.db.storageengine.dataregion.compaction.constant.CompactionTaskType;
import org.apache.iotdb.db.storageengine.dataregion.compaction.schedule.CompactionTaskManager;
import org.apache.iotdb.db.storageengine.dataregion.modification.Modification;
import org.apache.iotdb.db.storageengine.dataregion.modification.ModificationFile;
import org.apache.iotdb.db.storageengine.dataregion.tsfile.TsFileResource;
import org.apache.iotdb.db.storageengine.dataregion.tsfile.timeindex.DeviceTimeIndex;
import org.apache.iotdb.metrics.utils.MetricLevel;
import org.apache.iotdb.metrics.utils.SystemMetric;
import org.apache.tsfile.file.metadata.ChunkMetadata;
import org.apache.tsfile.file.metadata.IDeviceID;
import org.apache.tsfile.fileSystem.FSFactoryProducer;
import org.apache.tsfile.write.writer.TsFileIOWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionUtils {
    private static final Logger logger = LoggerFactory.getLogger((String)"COMPACTION");
    private static final String SYSTEM = "system";

    private CompactionUtils() {
    }

    public static void moveTargetFile(List<TsFileResource> targetResources, CompactionTaskType type, String fullStorageGroupName) throws IOException {
        String fileSuffix = CompactionUtils.getTmpFileSuffix(type);
        for (TsFileResource targetResource : targetResources) {
            if (targetResource == null) continue;
            CompactionUtils.moveOneTargetFile(targetResource, fileSuffix, fullStorageGroupName);
        }
    }

    public static String getTmpFileSuffix(CompactionTaskType type) {
        switch (type) {
            case INNER_UNSEQ: 
            case INNER_SEQ: 
            case REPAIR: {
                return ".inner";
            }
            case CROSS: {
                return ".cross";
            }
            case SETTLE: {
                return ".settle";
            }
        }
        logger.error("Current task type {} does not have tmp file suffix.", (Object)type);
        return "";
    }

    private static void moveOneTargetFile(TsFileResource targetResource, String tmpFileSuffix, String fullStorageGroupName) throws IOException {
        if (!targetResource.getTsFile().exists()) {
            logger.info("{} [Compaction] Tmp target tsfile {} may be deleted after compaction.", (Object)fullStorageGroupName, (Object)targetResource.getTsFilePath());
            return;
        }
        File newFile = new File(targetResource.getTsFilePath().replace(tmpFileSuffix, ".tsfile"));
        if (!newFile.exists()) {
            FSFactoryProducer.getFSFactory().moveFile(targetResource.getTsFile(), newFile);
        }
        targetResource.setFile(newFile);
        targetResource.serialize();
        targetResource.closeWithoutSettingStatus();
    }

    public static void combineModsInCrossCompaction(List<TsFileResource> seqResources, List<TsFileResource> unseqResources, List<TsFileResource> targetResources) throws IOException {
        HashSet<Modification> modifications = new HashSet<Modification>();
        for (TsFileResource unseqFile : unseqResources) {
            modifications.addAll(ModificationFile.getCompactionMods(unseqFile).getModifications());
        }
        for (int i = 0; i < targetResources.size(); ++i) {
            TsFileResource targetResource = targetResources.get(i);
            if (targetResource == null) continue;
            HashSet<Modification> seqModifications = new HashSet<Modification>(ModificationFile.getCompactionMods(seqResources.get(i)).getModifications());
            modifications.addAll(seqModifications);
            CompactionUtils.updateOneTargetMods(targetResource, modifications);
            modifications.removeAll(seqModifications);
        }
    }

    public static void combineModsInInnerCompaction(Collection<TsFileResource> sourceFiles, TsFileResource targetTsFile) throws IOException {
        HashSet<Modification> modifications = new HashSet<Modification>();
        for (TsFileResource mergeTsFile : sourceFiles) {
            ModificationFile sourceCompactionModificationFile = ModificationFile.getCompactionMods(mergeTsFile);
            try {
                modifications.addAll(sourceCompactionModificationFile.getModifications());
            }
            finally {
                if (sourceCompactionModificationFile == null) continue;
                sourceCompactionModificationFile.close();
            }
        }
        CompactionUtils.updateOneTargetMods(targetTsFile, modifications);
    }

    public static void combineModsInInnerCompaction(Collection<TsFileResource> sourceFiles, List<TsFileResource> targetTsFiles) throws IOException {
        HashSet<Modification> modifications = new HashSet<Modification>();
        for (TsFileResource mergeTsFile : sourceFiles) {
            ModificationFile sourceCompactionModificationFile = ModificationFile.getCompactionMods(mergeTsFile);
            try {
                modifications.addAll(sourceCompactionModificationFile.getModifications());
            }
            finally {
                if (sourceCompactionModificationFile == null) continue;
                sourceCompactionModificationFile.close();
            }
        }
        for (TsFileResource targetTsFile : targetTsFiles) {
            CompactionUtils.updateOneTargetMods(targetTsFile, modifications);
        }
    }

    public static void addFilesToFileMetrics(TsFileResource resource) {
        FileMetrics.getInstance().addTsFile(resource.getDatabaseName(), resource.getDataRegionId(), resource.getTsFile().length(), resource.isSeq(), resource.getTsFile().getName());
        if (resource.modFileExists()) {
            FileMetrics.getInstance().increaseModFileNum(1);
            FileMetrics.getInstance().increaseModFileSize(resource.getModFile().getSize());
        }
    }

    private static void updateOneTargetMods(TsFileResource targetFile, Set<Modification> modifications) throws IOException {
        if (!modifications.isEmpty()) {
            try (ModificationFile modificationFile = ModificationFile.getNormalMods(targetFile);){
                for (Modification modification : modifications) {
                    modification.setFileOffset(Long.MAX_VALUE);
                    modificationFile.write(modification);
                }
            }
        }
    }

    public static void deleteCompactionModsFile(List<TsFileResource> selectedSeqTsFileResourceList, List<TsFileResource> selectedUnSeqTsFileResourceList) throws IOException {
        ModificationFile modificationFile;
        for (TsFileResource seqFile : selectedSeqTsFileResourceList) {
            modificationFile = seqFile.getCompactionModFile();
            if (!modificationFile.exists()) continue;
            modificationFile.remove();
        }
        for (TsFileResource unseqFile : selectedUnSeqTsFileResourceList) {
            modificationFile = unseqFile.getCompactionModFile();
            if (!modificationFile.exists()) continue;
            modificationFile.remove();
        }
    }

    public static boolean deleteTsFilesInDisk(Collection<TsFileResource> mergeTsFiles, String storageGroupName) {
        logger.info("{} [Compaction] Compaction starts to delete real file ", (Object)storageGroupName);
        boolean result = true;
        for (TsFileResource mergeTsFile : mergeTsFiles) {
            if (!mergeTsFile.remove()) {
                result = false;
            }
            logger.info("{} [Compaction] delete TsFile {}", (Object)storageGroupName, (Object)mergeTsFile.getTsFilePath());
        }
        return result;
    }

    public static void deleteModificationForSourceFile(Collection<TsFileResource> sourceFiles, String storageGroupName) throws IOException {
        logger.info("{} [Compaction] Start to delete modifications of source files", (Object)storageGroupName);
        for (TsFileResource tsFileResource : sourceFiles) {
            ModificationFile normalModification;
            ModificationFile compactionModificationFile = ModificationFile.getCompactionMods(tsFileResource);
            if (compactionModificationFile.exists()) {
                compactionModificationFile.remove();
            }
            if (!(normalModification = ModificationFile.getNormalMods(tsFileResource)).exists()) continue;
            FileMetrics.getInstance().decreaseModFileNum(1);
            FileMetrics.getInstance().decreaseModFileSize(tsFileResource.getModFile().getSize());
            normalModification.remove();
        }
    }

    public static void updateResource(TsFileResource resource, TsFileIOWriter tsFileIoWriter, IDeviceID deviceId) {
        List chunkMetadatasOfCurrentDevice = tsFileIoWriter.getChunkMetadataListOfCurrentDeviceInMemory();
        if (chunkMetadatasOfCurrentDevice != null) {
            for (ChunkMetadata chunkMetadata : chunkMetadatasOfCurrentDevice) {
                if (chunkMetadata.getMask() == 64) continue;
                resource.updateStartTime(deviceId, chunkMetadata.getStatistics().getStartTime());
                resource.updateEndTime(deviceId, chunkMetadata.getStatistics().getEndTime());
            }
        }
    }

    public static void updateProgressIndexAndMark(List<TsFileResource> targetResources, List<TsFileResource> seqResources, List<TsFileResource> unseqResources) {
        for (TsFileResource targetResource : targetResources) {
            for (TsFileResource unseqResource : unseqResources) {
                targetResource.updateProgressIndex(unseqResource.getMaxProgressIndexAfterClose());
                targetResource.setGeneratedByPipe(unseqResource.isGeneratedByPipe() && targetResource.isGeneratedByPipe());
                targetResource.setGeneratedByPipeConsensus(unseqResource.isGeneratedByPipeConsensus() && targetResource.isGeneratedByPipeConsensus());
            }
            for (TsFileResource seqResource : seqResources) {
                targetResource.updateProgressIndex(seqResource.getMaxProgressIndexAfterClose());
                targetResource.setGeneratedByPipe(seqResource.isGeneratedByPipe() && targetResource.isGeneratedByPipe());
                targetResource.setGeneratedByPipeConsensus(seqResource.isGeneratedByPipeConsensus() && targetResource.isGeneratedByPipeConsensus());
            }
        }
    }

    public static void updatePlanIndexes(List<TsFileResource> targetResources, List<TsFileResource> seqResources, List<TsFileResource> unseqResources) {
        for (TsFileResource targetResource : targetResources) {
            for (TsFileResource unseqResource : unseqResources) {
                targetResource.updatePlanIndexes(unseqResource);
            }
            for (TsFileResource seqResource : seqResources) {
                targetResource.updatePlanIndexes(seqResource);
            }
        }
    }

    public static void deleteSourceTsFileAndUpdateFileMetrics(List<TsFileResource> sourceSeqResourceList, List<TsFileResource> sourceUnseqResourceList) {
        CompactionUtils.deleteSourceTsFileAndUpdateFileMetrics(sourceSeqResourceList, true);
        CompactionUtils.deleteSourceTsFileAndUpdateFileMetrics(sourceUnseqResourceList, false);
    }

    public static void deleteSourceTsFileAndUpdateFileMetrics(List<TsFileResource> resources, boolean seq) {
        for (TsFileResource resource : resources) {
            if (resource.getModFile().exists()) {
                FileMetrics.getInstance().decreaseModFileNum(1);
                FileMetrics.getInstance().decreaseModFileSize(resource.getModFile().getSize());
            }
            CompactionUtils.deleteTsFileResourceWithoutLock(resource);
        }
        FileMetrics.getInstance().deleteTsFile(seq, resources);
    }

    public static void deleteTsFileResourceWithoutLock(TsFileResource resource) {
        if (!resource.remove()) {
            logger.warn("[Compaction] delete file failed, file path is {}", (Object)resource.getTsFile().getAbsolutePath());
        } else {
            logger.info("[Compaction] delete file: {}", (Object)resource.getTsFile().getAbsolutePath());
        }
    }

    public static boolean isDiskHasSpace() {
        return CompactionUtils.isDiskHasSpace(0.0);
    }

    public static boolean isDiskHasSpace(double redundancy) {
        double availableDisk = MetricService.getInstance().getAutoGauge(SystemMetric.SYS_DISK_AVAILABLE_SPACE.toString(), MetricLevel.CORE, new String[]{Tag.NAME.toString(), SYSTEM}).getValue();
        double totalDisk = MetricService.getInstance().getAutoGauge(SystemMetric.SYS_DISK_TOTAL_SPACE.toString(), MetricLevel.CORE, new String[]{Tag.NAME.toString(), SYSTEM}).getValue();
        if (availableDisk != 0.0 && totalDisk != 0.0) {
            return availableDisk / totalDisk > CommonDescriptor.getInstance().getConfig().getDiskSpaceWarningThreshold() + redundancy;
        }
        return true;
    }

    public static DeviceTimeIndex buildDeviceTimeIndex(TsFileResource resource) throws IOException {
        int readSize;
        long resourceFileSize;
        CompactionTaskManager.getInstance().getCompactionReadOperationRateLimiter().acquire(1);
        CompactionMetrics.getInstance().recordDeserializeResourceInfo(resourceFileSize);
        for (resourceFileSize = new File(new StringBuilder().append(resource.getTsFilePath()).append(".resource").toString()).length(); resourceFileSize > 0L; resourceFileSize -= (long)readSize) {
            readSize = (int)Math.min(resourceFileSize, Integer.MAX_VALUE);
            CompactionTaskManager.getInstance().getCompactionReadRateLimiter().acquire(readSize);
        }
        return resource.buildDeviceTimeIndex();
    }
}

