/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.it.utils;

import java.io.File;
import java.io.IOException;
import java.time.LocalDate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.apache.tsfile.exception.write.WriteProcessException;
import org.apache.tsfile.file.metadata.TableSchema;
import org.apache.tsfile.write.TsFileWriter;
import org.apache.tsfile.write.record.Tablet;
import org.apache.tsfile.write.schema.IMeasurementSchema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TsFileTableGenerator
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(TsFileTableGenerator.class);
    private final File tsFile;
    private final TsFileWriter writer;
    private final Map<String, TreeSet<Long>> table2TimeSet;
    private final Map<String, List<IMeasurementSchema>> table2MeasurementSchema;
    private final Map<String, List<Tablet.ColumnCategory>> table2ColumnCategory;
    private Random random;

    public TsFileTableGenerator(File tsFile) throws IOException {
        this.tsFile = tsFile;
        this.writer = new TsFileWriter(tsFile);
        this.table2TimeSet = new HashMap<String, TreeSet<Long>>();
        this.table2MeasurementSchema = new HashMap<String, List<IMeasurementSchema>>();
        this.table2ColumnCategory = new HashMap<String, List<Tablet.ColumnCategory>>();
        this.random = new Random();
    }

    public void registerTable(String tableName, List<IMeasurementSchema> columnSchemasList, List<Tablet.ColumnCategory> columnCategoryList) {
        if (this.table2MeasurementSchema.containsKey(tableName)) {
            LOGGER.warn("Table {} already exists", (Object)tableName);
            return;
        }
        this.writer.registerTableSchema(new TableSchema(tableName, columnSchemasList, columnCategoryList));
        this.table2TimeSet.put(tableName, new TreeSet());
        this.table2MeasurementSchema.put(tableName, columnSchemasList);
        this.table2ColumnCategory.put(tableName, columnCategoryList);
    }

    public void generateData(String tableName, int number, long timeGap) throws IOException, WriteProcessException {
        List<IMeasurementSchema> schemas = this.table2MeasurementSchema.get(tableName);
        List columnNameList = schemas.stream().map(IMeasurementSchema::getMeasurementName).collect(Collectors.toList());
        List dataTypeList = schemas.stream().map(IMeasurementSchema::getType).collect(Collectors.toList());
        List<Tablet.ColumnCategory> columnCategoryList = this.table2ColumnCategory.get(tableName);
        TreeSet<Long> timeSet = this.table2TimeSet.get(tableName);
        Tablet tablet = new Tablet(tableName, columnNameList, dataTypeList, columnCategoryList);
        Object[] values = tablet.getValues();
        long sensorNum = schemas.size();
        long startTime = timeSet.isEmpty() ? 0L : timeSet.last();
        for (long r = 0L; r < (long)number; ++r) {
            int row = tablet.getRowSize();
            tablet.addTimestamp(row, startTime += timeGap);
            timeSet.add(startTime);
            int i = 0;
            while ((long)i < sensorNum) {
                this.generateDataPoint(tablet, i, row, schemas.get(i));
                ++i;
            }
            if (tablet.getRowSize() != tablet.getMaxRowNumber()) continue;
            this.writer.writeTable(tablet);
            tablet.reset();
        }
        if (tablet.getRowSize() != 0) {
            this.writer.writeTable(tablet);
            tablet.reset();
        }
        LOGGER.info("Write {} points into table {}", (Object)number, (Object)tableName);
    }

    private void generateDataPoint(Tablet tablet, int column, int row, IMeasurementSchema schema) {
        switch (schema.getType()) {
            case INT32: {
                this.generateINT32(tablet, column, row);
                break;
            }
            case DATE: {
                this.generateDATE(tablet, column, row);
                break;
            }
            case INT64: 
            case TIMESTAMP: {
                this.generateINT64(tablet, column, row);
                break;
            }
            case FLOAT: {
                this.generateFLOAT(tablet, column, row);
                break;
            }
            case DOUBLE: {
                this.generateDOUBLE(tablet, column, row);
                break;
            }
            case BOOLEAN: {
                this.generateBOOLEAN(tablet, column, row);
                break;
            }
            case TEXT: 
            case BLOB: 
            case STRING: {
                this.generateTEXT(tablet, column, row);
                break;
            }
            default: {
                LOGGER.error("Wrong data type {}.", (Object)schema.getType());
            }
        }
    }

    private void generateINT32(Tablet tablet, int column, int row) {
        tablet.addValue(row, column, this.random.nextInt());
    }

    private void generateDATE(Tablet tablet, int column, int row) {
        tablet.addValue(row, column, LocalDate.of(1000 + this.random.nextInt(9000), 1 + this.random.nextInt(12), 1 + this.random.nextInt(28)));
    }

    private void generateINT64(Tablet tablet, int column, int row) {
        tablet.addValue(row, column, this.random.nextLong());
    }

    private void generateFLOAT(Tablet tablet, int column, int row) {
        tablet.addValue(row, column, this.random.nextFloat());
    }

    private void generateDOUBLE(Tablet tablet, int column, int row) {
        tablet.addValue(row, column, this.random.nextDouble());
    }

    private void generateBOOLEAN(Tablet tablet, int column, int row) {
        tablet.addValue(row, column, this.random.nextBoolean());
    }

    private void generateTEXT(Tablet tablet, int column, int row) {
        tablet.addValue(row, column, String.format("test point %d", this.random.nextInt()));
    }

    public long getTotalNumber() {
        return this.table2TimeSet.entrySet().stream().mapToInt(entry -> ((TreeSet)entry.getValue()).size() * this.table2MeasurementSchema.get(entry.getKey()).size()).sum();
    }

    @Override
    public void close() throws Exception {
        this.writer.close();
    }
}

