/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.password;

import com.google.common.base.Charsets;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.io.Closer;
import com.google.common.io.LineReader;
import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.gobblin.configuration.State;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.jasypt.util.text.BasicTextEncryptor;
import org.jasypt.util.text.StrongTextEncryptor;
import org.jasypt.util.text.TextEncryptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PasswordManager {
    private static final Logger LOG = LoggerFactory.getLogger(PasswordManager.class);
    private static final long CACHE_SIZE = 100L;
    private static final long CACHE_EXPIRATION_MIN = 10L;
    private static final Pattern PASSWORD_PATTERN = Pattern.compile("ENC\\((.*)\\)");
    private final boolean useStrongEncryptor;
    private FileSystem fs;
    private List<TextEncryptor> encryptors;
    private static final LoadingCache<CachedInstanceKey, PasswordManager> CACHED_INSTANCES = CacheBuilder.newBuilder().maximumSize(100L).expireAfterAccess(10L, TimeUnit.MINUTES).build((CacheLoader)new CacheLoader<CachedInstanceKey, PasswordManager>(){

        public PasswordManager load(CachedInstanceKey cacheKey) {
            return new PasswordManager(cacheKey);
        }
    });

    private PasswordManager(CachedInstanceKey cacheKey) {
        this.useStrongEncryptor = cacheKey.useStrongEncryptor;
        try {
            this.fs = cacheKey.fsURI != null ? FileSystem.get((URI)URI.create(cacheKey.fsURI), (Configuration)new Configuration()) : (cacheKey.masterPasswordFile != null ? new Path(cacheKey.masterPasswordFile).getFileSystem(new Configuration()) : null);
        }
        catch (IOException e) {
            LOG.warn("Failed to instantiate FileSystem.", (Throwable)e);
        }
        this.encryptors = this.getEncryptors(cacheKey);
    }

    private List<TextEncryptor> getEncryptors(CachedInstanceKey cacheKey) {
        ArrayList<TextEncryptor> encryptors = new ArrayList<TextEncryptor>();
        int numOfEncryptionKeys = cacheKey.numOfEncryptionKeys;
        String suffix = "";
        int i = 1;
        if (cacheKey.masterPasswordFile == null || numOfEncryptionKeys < 1) {
            return encryptors;
        }
        IOException exception = null;
        do {
            Path currentMasterPasswordFile = new Path(cacheKey.masterPasswordFile + suffix);
            try (Closer closer = Closer.create();){
                if (this.fs.exists(currentMasterPasswordFile) && !this.fs.getFileStatus(currentMasterPasswordFile).isDirectory()) {
                    InputStream in = (InputStream)closer.register((Closeable)this.fs.open(currentMasterPasswordFile));
                    String masterPassword = new LineReader((Readable)new InputStreamReader(in, Charsets.UTF_8)).readLine();
                    StrongTextEncryptor encryptor = this.useStrongEncryptor ? new StrongTextEncryptor() : new BasicTextEncryptor();
                    encryptor.getClass().getMethod("setPassword", String.class).invoke((Object)encryptor, masterPassword);
                    encryptors.add((TextEncryptor)encryptor);
                    suffix = "." + String.valueOf(i);
                }
            }
            catch (FileNotFoundException fnf) {
                LOG.warn("Master password file " + currentMasterPasswordFile + " not found.");
            }
            catch (IOException ioe) {
                exception = ioe;
                LOG.warn("Master password could not be read from file " + currentMasterPasswordFile);
            }
            catch (Exception e) {
                LOG.warn("Encryptor could not be instantiated.");
            }
        } while (i++ < numOfEncryptionKeys);
        if (encryptors.size() < 1 && exception != null) {
            throw new RuntimeException("Master Password could not be read from any master password file.", exception);
        }
        return encryptors;
    }

    public static PasswordManager getInstance() {
        try {
            return (PasswordManager)CACHED_INSTANCES.get((Object)new CachedInstanceKey());
        }
        catch (ExecutionException e) {
            throw new RuntimeException("Unable to get an instance of PasswordManager", e);
        }
    }

    public static PasswordManager getInstance(State state) {
        try {
            return (PasswordManager)CACHED_INSTANCES.get((Object)new CachedInstanceKey(state));
        }
        catch (ExecutionException e) {
            throw new RuntimeException("Unable to get an instance of PasswordManager", e);
        }
    }

    public static PasswordManager getInstance(Properties props) {
        return PasswordManager.getInstance(new State(props));
    }

    public static PasswordManager getInstance(Path masterPwdLoc) {
        State state = new State();
        state.setProp("encrypt.key.loc", masterPwdLoc.toString());
        state.setProp("encrypt.key.fs.uri", masterPwdLoc.toUri());
        try {
            return (PasswordManager)CACHED_INSTANCES.get((Object)new CachedInstanceKey(state));
        }
        catch (ExecutionException e) {
            throw new RuntimeException("Unable to get an instance of PasswordManager", e);
        }
    }

    private static boolean shouldUseStrongEncryptor(State state) {
        return state.getPropAsBoolean("encrypt.use.strong.encryptor", false);
    }

    public String encryptPassword(String plain) {
        Preconditions.checkArgument((this.encryptors.size() > 0 ? 1 : 0) != 0, (Object)"A master password needs to be provided for encrypting passwords.");
        try {
            return this.encryptors.get(0).encrypt(plain);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to encrypt password", e);
        }
    }

    public String decryptPassword(String encrypted) {
        Preconditions.checkArgument((this.encryptors.size() > 0 ? 1 : 0) != 0, (Object)"A master password needs to be provided for decrypting passwords.");
        for (TextEncryptor encryptor : this.encryptors) {
            try {
                return encryptor.decrypt(encrypted);
            }
            catch (Exception e) {
                LOG.warn("Failed attempt to decrypt secret {}", (Object)encrypted, (Object)e);
            }
        }
        LOG.error("All {} decrypt attempt(s) failed.", (Object)this.encryptors.size());
        throw new RuntimeException("Failed to decrypt password ENC(" + encrypted + ")");
    }

    public String readPassword(String password) {
        if (password == null || this.encryptors.size() < 1) {
            return password;
        }
        Matcher matcher = PASSWORD_PATTERN.matcher(password);
        if (matcher.find()) {
            return this.decryptPassword(matcher.group(1));
        }
        return password;
    }

    public static Optional<String> getMasterPassword(Path masterPasswordFile) {
        try {
            FileSystem fs = masterPasswordFile.getFileSystem(new Configuration());
            return PasswordManager.getMasterPassword(fs, masterPasswordFile);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to obtain master password from " + masterPasswordFile, e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Optional<String> getMasterPassword(FileSystem fs, Path masterPasswordFile) {
        try (Closer closer = Closer.create();){
            if (!fs.exists(masterPasswordFile) || fs.getFileStatus(masterPasswordFile).isDirectory()) {
                LOG.warn(masterPasswordFile + " does not exist or is not a file. Cannot decrypt any encrypted password.");
                Optional optional = Optional.absent();
                return optional;
            }
            InputStream in = (InputStream)closer.register((Closeable)fs.open(masterPasswordFile));
            Optional optional = Optional.of((Object)new LineReader((Readable)new InputStreamReader(in, Charsets.UTF_8)).readLine());
            return optional;
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to obtain master password from " + masterPasswordFile, e);
        }
    }

    private static class CachedInstanceKey {
        int numOfEncryptionKeys;
        String fsURI;
        String masterPasswordFile;
        boolean useStrongEncryptor;

        public CachedInstanceKey(State state) {
            this.numOfEncryptionKeys = state.getPropAsInt("num.encrypt.keys", 2);
            this.useStrongEncryptor = PasswordManager.shouldUseStrongEncryptor(state);
            this.fsURI = state.getProp("encrypt.key.fs.uri");
            this.masterPasswordFile = state.getProp("encrypt.key.loc");
        }

        public CachedInstanceKey() {
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof CachedInstanceKey)) {
                return false;
            }
            CachedInstanceKey other = (CachedInstanceKey)o;
            if (!other.canEqual(this)) {
                return false;
            }
            if (this.numOfEncryptionKeys != other.numOfEncryptionKeys) {
                return false;
            }
            if (this.useStrongEncryptor != other.useStrongEncryptor) {
                return false;
            }
            String this$fsURI = this.fsURI;
            String other$fsURI = other.fsURI;
            if (this$fsURI == null ? other$fsURI != null : !this$fsURI.equals(other$fsURI)) {
                return false;
            }
            String this$masterPasswordFile = this.masterPasswordFile;
            String other$masterPasswordFile = other.masterPasswordFile;
            return !(this$masterPasswordFile == null ? other$masterPasswordFile != null : !this$masterPasswordFile.equals(other$masterPasswordFile));
        }

        protected boolean canEqual(Object other) {
            return other instanceof CachedInstanceKey;
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            result = result * 59 + this.numOfEncryptionKeys;
            result = result * 59 + (this.useStrongEncryptor ? 79 : 97);
            String $fsURI = this.fsURI;
            result = result * 59 + ($fsURI == null ? 43 : $fsURI.hashCode());
            String $masterPasswordFile = this.masterPasswordFile;
            result = result * 59 + ($masterPasswordFile == null ? 43 : $masterPasswordFile.hashCode());
            return result;
        }
    }
}

