/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.operators.util;

import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.core.memory.MemorySegmentFactory;
import org.apache.flink.core.memory.MemoryUtils;
import org.apache.flink.runtime.operators.util.BitSet;
import org.apache.flink.util.Preconditions;

public class BloomFilter {
    private static final int BYTE_ARRAY_BASE_OFFSET = MemoryUtils.UNSAFE.arrayBaseOffset(byte[].class);
    protected BitSet bitSet;
    protected int numHashFunctions;

    public BloomFilter(int expectedEntries, int byteSize) {
        Preconditions.checkArgument(expectedEntries > 0, "expectedEntries should be > 0");
        this.numHashFunctions = BloomFilter.optimalNumOfHashFunctions(expectedEntries, byteSize << 3);
        this.bitSet = new BitSet(byteSize);
    }

    private BloomFilter(BitSet bitSet, int numHashFunctions) {
        Preconditions.checkNotNull(bitSet, "bitSet must be not null");
        Preconditions.checkArgument(numHashFunctions > 0, "numHashFunctions should be > 0");
        this.bitSet = bitSet;
        this.numHashFunctions = numHashFunctions;
    }

    public void setBitsLocation(MemorySegment memorySegment, int offset) {
        this.bitSet.setMemorySegment(memorySegment, offset);
    }

    public static int optimalNumOfBits(long inputEntries, double fpp) {
        int numBits = (int)((double)(-inputEntries) * Math.log(fpp) / (Math.log(2.0) * Math.log(2.0)));
        return numBits;
    }

    public static double estimateFalsePositiveProbability(long inputEntries, int bitSize) {
        int numFunction = BloomFilter.optimalNumOfHashFunctions(inputEntries, bitSize);
        double p = Math.pow(Math.E, -((double)numFunction) * (double)inputEntries / (double)bitSize);
        double estimatedFPP = Math.pow(1.0 - p, numFunction);
        return estimatedFPP;
    }

    static int optimalNumOfHashFunctions(long expectEntries, long bitSize) {
        return Math.max(1, (int)Math.round((double)bitSize / (double)expectEntries * Math.log(2.0)));
    }

    public void addHash(int hash32) {
        int hash1 = hash32;
        int hash2 = hash32 >>> 16;
        for (int i = 1; i <= this.numHashFunctions; ++i) {
            int combinedHash = hash1 + i * hash2;
            if (combinedHash < 0) {
                combinedHash ^= 0xFFFFFFFF;
            }
            int pos = combinedHash % this.bitSet.bitSize();
            this.bitSet.set(pos);
        }
    }

    public boolean testHash(int hash32) {
        int hash1 = hash32;
        int hash2 = hash32 >>> 16;
        for (int i = 1; i <= this.numHashFunctions; ++i) {
            int pos;
            int combinedHash = hash1 + i * hash2;
            if (combinedHash < 0) {
                combinedHash ^= 0xFFFFFFFF;
            }
            if (this.bitSet.get(pos = combinedHash % this.bitSet.bitSize())) continue;
            return false;
        }
        return true;
    }

    public void reset() {
        this.bitSet.clear();
    }

    public String toString() {
        StringBuilder output = new StringBuilder();
        output.append("BloomFilter:\n");
        output.append("\thash function number:").append(this.numHashFunctions).append("\n");
        output.append(this.bitSet);
        return output.toString();
    }

    public static byte[] toBytes(BloomFilter filter) {
        byte[] data = filter.bitSet.toBytes();
        int byteSize = data.length;
        byte[] bytes = new byte[8 + byteSize];
        MemoryUtils.UNSAFE.putInt(bytes, BYTE_ARRAY_BASE_OFFSET, filter.numHashFunctions);
        MemoryUtils.UNSAFE.putInt(bytes, BYTE_ARRAY_BASE_OFFSET + 4, byteSize);
        MemoryUtils.UNSAFE.copyMemory(data, BYTE_ARRAY_BASE_OFFSET, bytes, BYTE_ARRAY_BASE_OFFSET + 8, byteSize);
        return bytes;
    }

    public static BloomFilter fromBytes(byte[] bytes) {
        int numHashFunctions = MemoryUtils.UNSAFE.getInt(bytes, BYTE_ARRAY_BASE_OFFSET);
        int byteSize = MemoryUtils.UNSAFE.getInt(bytes, BYTE_ARRAY_BASE_OFFSET + 4);
        byte[] data = new byte[byteSize];
        MemoryUtils.UNSAFE.copyMemory(bytes, BYTE_ARRAY_BASE_OFFSET + 8, data, BYTE_ARRAY_BASE_OFFSET, byteSize);
        BitSet bitSet = new BitSet(byteSize);
        bitSet.setMemorySegment(MemorySegmentFactory.wrap(data), 0);
        return new BloomFilter(bitSet, numHashFunctions);
    }

    public static byte[] mergeSerializedBloomFilters(byte[] bf1Bytes, byte[] bf2Bytes) {
        return BloomFilter.mergeSerializedBloomFilters(bf1Bytes, 0, bf1Bytes.length, bf2Bytes, 0, bf2Bytes.length);
    }

    private static byte[] mergeSerializedBloomFilters(byte[] bf1Bytes, int bf1Start, int bf1Length, byte[] bf2Bytes, int bf2Start, int bf2Length) {
        if (bf1Length != bf2Length) {
            throw new IllegalArgumentException(String.format("bf1Length %s does not match bf2Length %s when merging", bf1Length, bf2Length));
        }
        if (MemoryUtils.UNSAFE.getByte(bf1Bytes, BYTE_ARRAY_BASE_OFFSET + bf1Start) != MemoryUtils.UNSAFE.getByte(bf2Bytes, BYTE_ARRAY_BASE_OFFSET + bf2Start)) {
            throw new IllegalArgumentException("bf1 numHashFunctions does not match bf2 when merging");
        }
        for (int idx = 8 + BYTE_ARRAY_BASE_OFFSET; idx < bf1Length + BYTE_ARRAY_BASE_OFFSET; ++idx) {
            byte l1 = MemoryUtils.UNSAFE.getByte(bf1Bytes, bf1Start + idx);
            byte l2 = MemoryUtils.UNSAFE.getByte(bf2Bytes, bf2Start + idx);
            MemoryUtils.UNSAFE.putByte(bf1Bytes, bf1Start + idx, (byte)(l1 | l2));
        }
        return bf1Bytes;
    }
}

