/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state;

import org.apache.flink.api.common.state.StateDescriptor;
import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.api.common.typeutils.TypeSerializerConfigSnapshot;
import org.apache.flink.util.Preconditions;

import static org.apache.flink.util.Preconditions.checkNotNull;

/**
 * Compound meta information for a registered state in a keyed state backend. This combines all serializers and the
 * state name.
 *
 * @param <N> Type of namespace
 * @param <S> Type of state value
 */
public class RegisteredKeyedBackendStateMetaInfo<N, S> {

	private final StateDescriptor.Type stateType;
	private final String name;
	private final TypeSerializer<N> namespaceSerializer;
	private final TypeSerializer<S> stateSerializer;

	public RegisteredKeyedBackendStateMetaInfo(
			StateDescriptor.Type stateType,
			String name,
			TypeSerializer<N> namespaceSerializer,
			TypeSerializer<S> stateSerializer) {

		this.stateType = checkNotNull(stateType);
		this.name = checkNotNull(name);
		this.namespaceSerializer = checkNotNull(namespaceSerializer);
		this.stateSerializer = checkNotNull(stateSerializer);
	}

	public StateDescriptor.Type getStateType() {
		return stateType;
	}

	public String getName() {
		return name;
	}

	public TypeSerializer<N> getNamespaceSerializer() {
		return namespaceSerializer;
	}

	public TypeSerializer<S> getStateSerializer() {
		return stateSerializer;
	}

	public Snapshot<N, S> snapshot() {
		return new Snapshot<>(
			stateType,
			name,
			namespaceSerializer.duplicate(),
			stateSerializer.duplicate(),
			namespaceSerializer.snapshotConfiguration(),
			stateSerializer.snapshotConfiguration());
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		RegisteredKeyedBackendStateMetaInfo<?, ?> that = (RegisteredKeyedBackendStateMetaInfo<?, ?>) o;

		if (!stateType.equals(that.stateType)) {
			return false;
		}

		if (!getName().equals(that.getName())) {
			return false;
		}

		return getStateSerializer().equals(that.getStateSerializer())
				&& getNamespaceSerializer().equals(that.getNamespaceSerializer());
	}

	@Override
	public String toString() {
		return "RegisteredKeyedBackendStateMetaInfo{" +
				"stateType=" + stateType +
				", name='" + name + '\'' +
				", namespaceSerializer=" + namespaceSerializer +
				", stateSerializer=" + stateSerializer +
				'}';
	}

	@Override
	public int hashCode() {
		int result = getName().hashCode();
		result = 31 * result + getStateType().hashCode();
		result = 31 * result + getNamespaceSerializer().hashCode();
		result = 31 * result + getStateSerializer().hashCode();
		return result;
	}

	/**
	 * A consistent snapshot of a {@link RegisteredKeyedBackendStateMetaInfo}.
	 */
	public static class Snapshot<N, S> {

		private StateDescriptor.Type stateType;
		private String name;
		private TypeSerializer<N> namespaceSerializer;
		private TypeSerializer<S> stateSerializer;
		private TypeSerializerConfigSnapshot namespaceSerializerConfigSnapshot;
		private TypeSerializerConfigSnapshot stateSerializerConfigSnapshot;

		/** Empty constructor used when restoring the state meta info snapshot. */
		Snapshot() {}

		private Snapshot(
				StateDescriptor.Type stateType,
				String name,
				TypeSerializer<N> namespaceSerializer,
				TypeSerializer<S> stateSerializer,
				TypeSerializerConfigSnapshot namespaceSerializerConfigSnapshot,
				TypeSerializerConfigSnapshot stateSerializerConfigSnapshot) {

			this.stateType = Preconditions.checkNotNull(stateType);
			this.name = Preconditions.checkNotNull(name);
			this.namespaceSerializer = Preconditions.checkNotNull(namespaceSerializer);
			this.stateSerializer = Preconditions.checkNotNull(stateSerializer);
			this.namespaceSerializerConfigSnapshot = Preconditions.checkNotNull(namespaceSerializerConfigSnapshot);
			this.stateSerializerConfigSnapshot = Preconditions.checkNotNull(stateSerializerConfigSnapshot);
		}

		public StateDescriptor.Type getStateType() {
			return stateType;
		}

		void setStateType(StateDescriptor.Type stateType) {
			this.stateType = stateType;
		}

		public String getName() {
			return name;
		}

		void setName(String name) {
			this.name = name;
		}

		public TypeSerializer<N> getNamespaceSerializer() {
			return namespaceSerializer;
		}

		void setNamespaceSerializer(TypeSerializer<N> namespaceSerializer) {
			this.namespaceSerializer = namespaceSerializer;
		}

		public TypeSerializer<S> getStateSerializer() {
			return stateSerializer;
		}

		void setStateSerializer(TypeSerializer<S> stateSerializer) {
			this.stateSerializer = stateSerializer;
		}

		public TypeSerializerConfigSnapshot getNamespaceSerializerConfigSnapshot() {
			return namespaceSerializerConfigSnapshot;
		}

		void setNamespaceSerializerConfigSnapshot(TypeSerializerConfigSnapshot namespaceSerializerConfigSnapshot) {
			this.namespaceSerializerConfigSnapshot = namespaceSerializerConfigSnapshot;
		}

		public TypeSerializerConfigSnapshot getStateSerializerConfigSnapshot() {
			return stateSerializerConfigSnapshot;
		}

		void setStateSerializerConfigSnapshot(TypeSerializerConfigSnapshot stateSerializerConfigSnapshot) {
			this.stateSerializerConfigSnapshot = stateSerializerConfigSnapshot;
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) {
				return true;
			}

			if (o == null || getClass() != o.getClass()) {
				return false;
			}

			Snapshot<?, ?> that = (Snapshot<?, ?>) o;

			if (!stateType.equals(that.stateType)) {
				return false;
			}

			if (!getName().equals(that.getName())) {
				return false;
			}

			// need to check for nulls because serializer and config snapshots may be null on restore
			return
				((getStateSerializer() == null && that.getStateSerializer() == null)
					|| getStateSerializer().equals(that.getStateSerializer()))
				&& ((getNamespaceSerializer() == null && that.getNamespaceSerializer() == null)
					|| getNamespaceSerializer().equals(that.getNamespaceSerializer()))
				&& ((getNamespaceSerializerConfigSnapshot() == null && that.getNamespaceSerializerConfigSnapshot() == null)
					|| getNamespaceSerializerConfigSnapshot().equals(that.getNamespaceSerializerConfigSnapshot()))
				&& ((getStateSerializerConfigSnapshot() == null && that.getStateSerializerConfigSnapshot() == null)
					|| getStateSerializerConfigSnapshot().equals(that.getStateSerializerConfigSnapshot()));
		}

		@Override
		public int hashCode() {
			// need to check for nulls because serializer and config snapshots may be null on restore
			int result = getName().hashCode();
			result = 31 * result + getStateType().hashCode();
			result = 31 * result + (getNamespaceSerializer() != null ? getNamespaceSerializer().hashCode() : 0);
			result = 31 * result + (getStateSerializer() != null ? getStateSerializer().hashCode() : 0);
			result = 31 * result + (getNamespaceSerializerConfigSnapshot() != null ? getNamespaceSerializerConfigSnapshot().hashCode() : 0);
			result = 31 * result + (getStateSerializerConfigSnapshot() != null ? getStateSerializerConfigSnapshot().hashCode() : 0);
			return result;
		}
	}
}
