/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.api.ldap.model.password;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Date;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.apache.directory.api.ldap.model.constants.LdapSecurityConstants;
import org.apache.directory.api.ldap.model.password.EncryptionMethod;
import org.apache.directory.api.util.Base64;
import org.apache.directory.api.util.DateUtils;
import org.apache.directory.api.util.Strings;
import org.apache.directory.api.util.UnixCrypt;

public final class PasswordUtil {
    public static final int SHA1_LENGTH = 20;
    public static final int SHA256_LENGTH = 32;
    public static final int SHA384_LENGTH = 48;
    public static final int SHA512_LENGTH = 64;
    public static final int MD5_LENGTH = 16;
    public static final int PKCS5S2_LENGTH = 32;

    private PasswordUtil() {
    }

    public static LdapSecurityConstants findAlgorithm(byte[] credentials) {
        if (credentials == null || credentials.length == 0) {
            return null;
        }
        if (credentials[0] == 123) {
            int pos;
            for (pos = 1; pos < credentials.length && credentials[pos] != 125; ++pos) {
            }
            if (pos < credentials.length) {
                if (pos == 1) {
                    return null;
                }
                String algorithm = Strings.toLowerCaseAscii((String)Strings.utf8ToString((byte[])credentials, (int)1, (int)(pos - 1)));
                return LdapSecurityConstants.getAlgorithm(algorithm);
            }
            return null;
        }
        return null;
    }

    public static byte[] createStoragePassword(String credentials, LdapSecurityConstants algorithm) {
        return PasswordUtil.createStoragePassword(Strings.getBytesUtf8((String)credentials), algorithm);
    }

    public static byte[] createStoragePassword(byte[] credentials, LdapSecurityConstants algorithm) {
        byte[] salt;
        switch (algorithm) {
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SMD5: {
                salt = new byte[8];
                new SecureRandom().nextBytes(salt);
                break;
            }
            case HASH_METHOD_PKCS5S2: {
                salt = new byte[16];
                new SecureRandom().nextBytes(salt);
                break;
            }
            case HASH_METHOD_CRYPT: {
                salt = new byte[2];
                SecureRandom sr = new SecureRandom();
                int i1 = sr.nextInt(64);
                int i2 = sr.nextInt(64);
                salt[0] = (byte)(i1 < 12 ? i1 + 46 : (i1 < 38 ? i1 + 65 - 12 : i1 + 97 - 38));
                salt[1] = (byte)(i2 < 12 ? i2 + 46 : (i2 < 38 ? i2 + 65 - 12 : i2 + 97 - 38));
                break;
            }
            default: {
                salt = null;
            }
        }
        byte[] hashedPassword = PasswordUtil.encryptPassword(credentials, algorithm, salt);
        StringBuffer sb = new StringBuffer();
        sb.append('{').append(Strings.upperCase((String)algorithm.getPrefix())).append('}');
        if (algorithm == LdapSecurityConstants.HASH_METHOD_CRYPT) {
            sb.append(Strings.utf8ToString((byte[])salt));
            sb.append(Strings.utf8ToString((byte[])hashedPassword));
        } else if (salt != null) {
            byte[] hashedPasswordWithSaltBytes = new byte[hashedPassword.length + salt.length];
            if (algorithm == LdapSecurityConstants.HASH_METHOD_PKCS5S2) {
                PasswordUtil.merge(hashedPasswordWithSaltBytes, salt, hashedPassword);
            } else {
                PasswordUtil.merge(hashedPasswordWithSaltBytes, hashedPassword, salt);
            }
            sb.append(String.valueOf(Base64.encode((byte[])hashedPasswordWithSaltBytes)));
        } else {
            sb.append(String.valueOf(Base64.encode((byte[])hashedPassword)));
        }
        return Strings.getBytesUtf8((String)sb.toString());
    }

    public static boolean compareCredentials(byte[] receivedCredentials, byte[] storedCredentials) {
        LdapSecurityConstants algorithm = PasswordUtil.findAlgorithm(storedCredentials);
        if (algorithm != null) {
            EncryptionMethod encryptionMethod = new EncryptionMethod(algorithm, null);
            byte[] encryptedStored = PasswordUtil.splitCredentials(storedCredentials, encryptionMethod);
            byte[] userPassword = PasswordUtil.encryptPassword(receivedCredentials, encryptionMethod.getAlgorithm(), encryptionMethod.getSalt());
            return PasswordUtil.compareBytes(userPassword, encryptedStored);
        }
        return PasswordUtil.compareBytes(receivedCredentials, storedCredentials);
    }

    private static boolean compareBytes(byte[] provided, byte[] stored) {
        if (stored == null) {
            return provided == null;
        }
        if (provided == null) {
            return false;
        }
        if (stored.length != provided.length) {
            return false;
        }
        int result = 0;
        for (int i = 0; i < stored.length; ++i) {
            result |= stored[i] ^ provided[i];
        }
        return result == 0;
    }

    public static byte[] encryptPassword(byte[] credentials, LdapSecurityConstants algorithm, byte[] salt) {
        switch (algorithm) {
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SHA: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA, credentials, salt);
            }
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SHA256: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA256, credentials, salt);
            }
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SHA384: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA384, credentials, salt);
            }
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SHA512: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA512, credentials, salt);
            }
            case HASH_METHOD_SMD5: 
            case HASH_METHOD_MD5: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_MD5, credentials, salt);
            }
            case HASH_METHOD_CRYPT: {
                String saltWithCrypted = UnixCrypt.crypt((String)Strings.utf8ToString((byte[])credentials), (String)Strings.utf8ToString((byte[])salt));
                String crypted = saltWithCrypted.substring(2);
                return Strings.getBytesUtf8((String)crypted);
            }
            case HASH_METHOD_PKCS5S2: {
                return PasswordUtil.generatePbkdf2Hash(credentials, algorithm, salt);
            }
        }
        return credentials;
    }

    private static byte[] digest(LdapSecurityConstants algorithm, byte[] password, byte[] salt) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(algorithm.getAlgorithm());
        }
        catch (NoSuchAlgorithmException e1) {
            return null;
        }
        if (salt != null) {
            digest.update(password);
            digest.update(salt);
            return digest.digest();
        }
        return digest.digest(password);
    }

    public static byte[] splitCredentials(byte[] credentials, EncryptionMethod encryptionMethod) {
        int algoLength = encryptionMethod.getAlgorithm().getPrefix().length() + 2;
        switch (encryptionMethod.getAlgorithm()) {
            case HASH_METHOD_SHA: 
            case HASH_METHOD_MD5: {
                return Base64.decode((char[])Strings.utf8ToString((byte[])credentials, (int)algoLength, (int)(credentials.length - algoLength)).toCharArray());
            }
            case HASH_METHOD_SMD5: {
                byte[] passwordAndSalt = Base64.decode((char[])Strings.utf8ToString((byte[])credentials, (int)algoLength, (int)(credentials.length - algoLength)).toCharArray());
                int saltLength = passwordAndSalt.length - 16;
                encryptionMethod.setSalt(new byte[saltLength]);
                byte[] password = new byte[16];
                PasswordUtil.split(passwordAndSalt, 0, password, encryptionMethod.getSalt());
                return password;
            }
            case HASH_METHOD_SSHA: {
                return PasswordUtil.getCredentials(credentials, algoLength, 20, encryptionMethod);
            }
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SHA256: {
                return PasswordUtil.getCredentials(credentials, algoLength, 32, encryptionMethod);
            }
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SHA384: {
                return PasswordUtil.getCredentials(credentials, algoLength, 48, encryptionMethod);
            }
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SHA512: {
                return PasswordUtil.getCredentials(credentials, algoLength, 64, encryptionMethod);
            }
            case HASH_METHOD_PKCS5S2: {
                return PasswordUtil.getPbkdf2Credentials(credentials, algoLength, encryptionMethod);
            }
            case HASH_METHOD_CRYPT: {
                encryptionMethod.setSalt(new byte[2]);
                byte[] password2 = new byte[credentials.length - encryptionMethod.getSalt().length - algoLength];
                PasswordUtil.split(credentials, algoLength, encryptionMethod.getSalt(), password2);
                return password2;
            }
        }
        return credentials;
    }

    private static byte[] getCredentials(byte[] credentials, int algoLength, int hashLen, EncryptionMethod encryptionMethod) {
        byte[] passwordAndSalt = Base64.decode((char[])Strings.utf8ToString((byte[])credentials, (int)algoLength, (int)(credentials.length - algoLength)).toCharArray());
        int saltLength = passwordAndSalt.length - hashLen;
        encryptionMethod.setSalt(new byte[saltLength]);
        byte[] password = new byte[hashLen];
        PasswordUtil.split(passwordAndSalt, 0, password, encryptionMethod.getSalt());
        return password;
    }

    private static void split(byte[] all, int offset, byte[] left, byte[] right) {
        System.arraycopy(all, offset, left, 0, left.length);
        System.arraycopy(all, offset + left.length, right, 0, right.length);
    }

    private static void merge(byte[] all, byte[] left, byte[] right) {
        System.arraycopy(left, 0, all, 0, left.length);
        System.arraycopy(right, 0, all, left.length, right.length);
    }

    public static boolean isPwdExpired(String pwdChangedZtime, int pwdMaxAgeSec) {
        Date pwdChangeDate = DateUtils.getDate((String)pwdChangedZtime);
        long time = (long)pwdMaxAgeSec * 1000L;
        Date expiryDate = DateUtils.getDate((String)DateUtils.getGeneralizedTime((long)(time += pwdChangeDate.getTime())));
        Date now = DateUtils.getDate((String)DateUtils.getGeneralizedTime());
        boolean expired = false;
        if (expiryDate.equals(now) || expiryDate.before(now)) {
            expired = true;
        }
        return expired;
    }

    private static byte[] generatePbkdf2Hash(byte[] credentials, LdapSecurityConstants algorithm, byte[] salt) {
        try {
            SecretKeyFactory sk = SecretKeyFactory.getInstance(algorithm.getAlgorithm());
            char[] password = Strings.utf8ToString((byte[])credentials).toCharArray();
            PBEKeySpec keySpec = new PBEKeySpec(password, salt, 10000, 256);
            SecretKey key = sk.generateSecret(keySpec);
            return key.getEncoded();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static byte[] getPbkdf2Credentials(byte[] credentials, int algoLength, EncryptionMethod encryptionMethod) {
        byte[] passwordAndSalt = Base64.decode((char[])Strings.utf8ToString((byte[])credentials, (int)algoLength, (int)(credentials.length - algoLength)).toCharArray());
        int saltLength = passwordAndSalt.length - 32;
        encryptionMethod.setSalt(new byte[saltLength]);
        byte[] password = new byte[32];
        PasswordUtil.split(passwordAndSalt, 0, encryptionMethod.getSalt(), password);
        return password;
    }
}

