/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */
package org.apache.directory.api.ldap.codec.actions.request.bind;


import org.apache.directory.api.asn1.DecoderException;
import org.apache.directory.api.asn1.ber.grammar.GrammarAction;
import org.apache.directory.api.asn1.ber.tlv.BerValue;
import org.apache.directory.api.asn1.ber.tlv.IntegerDecoder;
import org.apache.directory.api.asn1.ber.tlv.IntegerDecoderException;
import org.apache.directory.api.asn1.ber.tlv.TLV;
import org.apache.directory.api.i18n.I18n;
import org.apache.directory.api.ldap.codec.api.LdapMessageContainer;
import org.apache.directory.api.ldap.model.message.BindRequest;
import org.apache.directory.api.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * The action used to store the BindRequest version.
 * <pre>
 * BindRequest ::= [APPLICATION 0] SEQUENCE {
 *     version                 INTEGER (1 ..  127),
 *     ....
 * </pre>
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class StoreVersion extends GrammarAction<LdapMessageContainer<BindRequest>>
{
    /** The logger */
    private static final Logger LOG = LoggerFactory.getLogger( StoreVersion.class );

    /**
     * Instantiates a new action.
     */
    public StoreVersion()
    {
        super( "Store BindRequest Version" );
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public void action( LdapMessageContainer<BindRequest> container ) throws DecoderException
    {
        BindRequest bindRequestMessage = container.getMessage();

        // The current TLV should be a integer between 1 and 127
        // We get it and store it in Version
        TLV tlv = container.getCurrentTLV();

        BerValue value = tlv.getValue();

        try
        {
            int version = IntegerDecoder.parse( value, 1, 127 );

            if ( LOG.isDebugEnabled() )
            {
                LOG.debug( I18n.msg( I18n.MSG_05114_LDAP_VERSION, Integer.valueOf( version ) ) );
            }

            bindRequestMessage.setVersion3( version == 3 );
        }
        catch ( IntegerDecoderException ide )
        {
            LOG.error( I18n
                .err( I18n.ERR_05117_INVALID_VERSION, Strings.dumpBytes( value.getData() ), ide.getMessage() ) );

            // This will generate a PROTOCOL_ERROR
            throw new DecoderException( ide.getMessage(), ide );
        }
    }
}
