/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cayenne.datasource;

import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import javax.sql.DataSource;
import org.apache.cayenne.CayenneRuntimeException;
import org.apache.cayenne.datasource.BadValidationQueryException;
import org.apache.cayenne.datasource.PoolAwareConnection;
import org.apache.cayenne.datasource.PoolingDataSource;
import org.apache.cayenne.datasource.PoolingDataSourceParameters;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class UnmanagedPoolingDataSource
implements PoolingDataSource {
    public static final int MAX_QUEUE_WAIT_DEFAULT = 20000;
    private static Log LOGGER = LogFactory.getLog(UnmanagedPoolingDataSource.class);
    private DataSource nonPoolingDataSource;
    private long maxQueueWaitTime;
    private Map<PoolAwareConnection, Object> pool;
    private Semaphore poolCap;
    private BlockingQueue<PoolAwareConnection> available;
    private int maxIdleConnections;
    private int minConnections;
    private int maxConnections;
    private String validationQuery;

    static void sybaseAutoCommitPatch(Connection c, SQLException e, boolean autoCommit) throws SQLException {
        String s = e.getMessage().toLowerCase();
        if (!s.contains("set chained command not allowed")) {
            throw e;
        }
        c.commit();
        c.setAutoCommit(autoCommit);
    }

    static int maxIdleConnections(int min, int max) {
        return min == max ? min : min + (int)Math.ceil((double)(max - min) / 2.0);
    }

    public UnmanagedPoolingDataSource(DataSource nonPoolingDataSource, PoolingDataSourceParameters parameters) {
        int minConnections = parameters.getMinConnections();
        int maxConnections = parameters.getMaxConnections();
        if (minConnections < 0) {
            throw new IllegalArgumentException("Negative min connections: " + minConnections);
        }
        if (maxConnections < 0) {
            throw new IllegalArgumentException("Negative max connections: " + maxConnections);
        }
        if (minConnections > maxConnections) {
            throw new IllegalArgumentException("Min connections (" + minConnections + ") is greater than max connections (" + maxConnections + ")");
        }
        this.nonPoolingDataSource = nonPoolingDataSource;
        this.maxQueueWaitTime = parameters.getMaxQueueWaitTime();
        this.validationQuery = parameters.getValidationQuery();
        this.minConnections = minConnections;
        this.maxConnections = maxConnections;
        this.pool = new ConcurrentHashMap<PoolAwareConnection, Object>((int)((double)maxConnections / 0.75));
        this.available = new ArrayBlockingQueue<PoolAwareConnection>(maxConnections);
        this.poolCap = new Semaphore(maxConnections);
        this.maxIdleConnections = UnmanagedPoolingDataSource.maxIdleConnections(minConnections, maxConnections);
        try {
            for (int i = 0; i < minConnections; ++i) {
                PoolAwareConnection c = this.createUnchecked();
                this.reclaim(c);
            }
        }
        catch (BadValidationQueryException e) {
            throw new CayenneRuntimeException("Bad validation query: " + this.validationQuery, (Throwable)e, new Object[0]);
        }
        catch (SQLException e) {
            LOGGER.info((Object)"Error creating new connection when starting connection pool, ignoring", (Throwable)e);
        }
    }

    int poolSize() {
        return this.pool.size();
    }

    int availableSize() {
        return this.available.size();
    }

    int canExpandSize() {
        return this.poolCap.availablePermits();
    }

    @Override
    public void close() {
        for (PoolAwareConnection c : this.pool.keySet()) {
            this.retire(c);
        }
        this.available.clear();
        this.pool = Collections.emptyMap();
    }

    void managePool() {
        PoolAwareConnection c;
        if (this.available.size() < this.minConnections) {
            try {
                PoolAwareConnection c2 = this.createUnchecked();
                if (c2 != null) {
                    this.reclaim(c2);
                }
            }
            catch (SQLException e) {
                LOGGER.info((Object)"Error creating new connection when managing connection pool, ignoring", (Throwable)e);
            }
        } else if (this.available.size() > this.maxIdleConnections && (c = this.uncheckNonBlocking(false)) != null) {
            this.retire(c);
        }
    }

    void retire(PoolAwareConnection connection) {
        this.pool.remove(connection);
        this.poolCap.release();
        try {
            connection.getConnection().close();
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    void reclaim(PoolAwareConnection connection) {
        if (!this.available.offer(connection)) {
            this.retire(connection);
        }
    }

    PoolAwareConnection uncheckNonBlocking(boolean validate) {
        PoolAwareConnection c = (PoolAwareConnection)this.available.poll();
        return validate ? this.validateUnchecked(c) : c;
    }

    PoolAwareConnection uncheckBlocking(boolean validate) {
        PoolAwareConnection c;
        try {
            c = this.available.poll(this.maxQueueWaitTime, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            return null;
        }
        return validate ? this.validateUnchecked(c) : c;
    }

    PoolAwareConnection validateUnchecked(PoolAwareConnection c) {
        if (c == null || c.validate()) {
            return c;
        }
        this.retire(c);
        return this.validateUnchecked((PoolAwareConnection)this.available.poll());
    }

    PoolAwareConnection createUnchecked() throws SQLException {
        PoolAwareConnection c;
        if (!this.poolCap.tryAcquire()) {
            return null;
        }
        try {
            c = this.createWrapped();
        }
        catch (SQLException e) {
            this.poolCap.release();
            throw e;
        }
        this.pool.put(c, 1);
        if (!c.validate()) {
            throw new BadValidationQueryException("Can't validate a fresh connection. Likely validation query is wrong: " + this.validationQuery);
        }
        return c;
    }

    PoolAwareConnection createWrapped() throws SQLException {
        return new PoolAwareConnection(this, this.createUnwrapped(), this.validationQuery);
    }

    Connection createUnwrapped() throws SQLException {
        return this.nonPoolingDataSource.getConnection();
    }

    Connection resetState(Connection c) throws SQLException {
        if (!c.getAutoCommit()) {
            try {
                c.setAutoCommit(true);
            }
            catch (SQLException e) {
                UnmanagedPoolingDataSource.sybaseAutoCommitPatch(c, e, true);
            }
        }
        c.clearWarnings();
        return c;
    }

    @Override
    public Connection getConnection() throws SQLException {
        PoolAwareConnection c = this.uncheckNonBlocking(true);
        if (c != null) {
            return this.resetState(c);
        }
        c = this.createUnchecked();
        if (c != null) {
            return this.resetState(c);
        }
        c = this.uncheckBlocking(true);
        if (c != null) {
            return this.resetState(c);
        }
        int poolSize = this.poolSize();
        int canGrow = this.poolCap.availablePermits();
        throw new ConnectionUnavailableException("Can't obtain connection. Request to pool timed out. Total pool size: " + poolSize + ", can expand by: " + canGrow);
    }

    @Override
    public Connection getConnection(String userName, String password) throws SQLException {
        throw new UnsupportedOperationException("Connections for a specific user are not supported by the pooled DataSource");
    }

    @Override
    public int getLoginTimeout() throws SQLException {
        return this.nonPoolingDataSource.getLoginTimeout();
    }

    @Override
    public void setLoginTimeout(int seconds) throws SQLException {
        this.nonPoolingDataSource.setLoginTimeout(seconds);
    }

    @Override
    public PrintWriter getLogWriter() throws SQLException {
        return this.nonPoolingDataSource.getLogWriter();
    }

    @Override
    public void setLogWriter(PrintWriter out) throws SQLException {
        this.nonPoolingDataSource.setLogWriter(out);
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return UnmanagedPoolingDataSource.class.equals(iface) ? true : this.nonPoolingDataSource.isWrapperFor(iface);
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        return (T)(UnmanagedPoolingDataSource.class.equals(iface) ? this : this.nonPoolingDataSource.unwrap(iface));
    }

    @Override
    public Logger getParentLogger() throws SQLFeatureNotSupportedException {
        return this.nonPoolingDataSource.getParentLogger();
    }

    String getValidationQuery() {
        return this.validationQuery;
    }

    long getMaxQueueWaitTime() {
        return this.maxQueueWaitTime;
    }

    int getMaxIdleConnections() {
        return this.maxIdleConnections;
    }

    int getMinConnections() {
        return this.minConnections;
    }

    int getMaxConnections() {
        return this.maxConnections;
    }

    public static class ConnectionUnavailableException
    extends SQLException {
        private static final long serialVersionUID = 1063973806941023165L;

        public ConnectionUnavailableException(String message) {
            super(message);
        }
    }
}

