/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.driver.api.testinfra.ccm;

import com.datastax.oss.driver.api.core.Version;
import com.datastax.oss.driver.shaded.guava.common.base.Joiner;
import com.datastax.oss.driver.shaded.guava.common.io.Resources;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteStreamHandler;
import org.apache.commons.exec.ExecuteWatchdog;
import org.apache.commons.exec.LogOutputStream;
import org.apache.commons.exec.PumpStreamHandler;
import org.assertj.core.util.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CcmBridge
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(CcmBridge.class);
    public static final Version VERSION = Objects.requireNonNull(Version.parse((String)System.getProperty("ccm.version", "4.0.0")));
    public static final String INSTALL_DIRECTORY = System.getProperty("ccm.directory");
    public static final String BRANCH = System.getProperty("ccm.branch");
    public static final Boolean DSE_ENABLEMENT = Boolean.getBoolean("ccm.dse");
    public static final String CLUSTER_NAME = "ccm_1";
    public static final String DEFAULT_CLIENT_TRUSTSTORE_PASSWORD = "fakePasswordForTests";
    public static final String DEFAULT_CLIENT_TRUSTSTORE_PATH = "/client.truststore";
    public static final File DEFAULT_CLIENT_TRUSTSTORE_FILE = CcmBridge.createTempStore("/client.truststore");
    public static final String DEFAULT_CLIENT_KEYSTORE_PASSWORD = "fakePasswordForTests";
    public static final String DEFAULT_CLIENT_KEYSTORE_PATH = "/client.keystore";
    public static final File DEFAULT_CLIENT_KEYSTORE_FILE = CcmBridge.createTempStore("/client.keystore");
    public static final File DEFAULT_CLIENT_PRIVATE_KEY_FILE = CcmBridge.createTempStore("/client.key");
    public static final File DEFAULT_CLIENT_CERT_CHAIN_FILE = CcmBridge.createTempStore("/client.crt");
    public static final String DEFAULT_SERVER_TRUSTSTORE_PASSWORD = "fakePasswordForTests";
    public static final String DEFAULT_SERVER_TRUSTSTORE_PATH = "/server.truststore";
    private static final File DEFAULT_SERVER_TRUSTSTORE_FILE = CcmBridge.createTempStore("/server.truststore");
    public static final String DEFAULT_SERVER_KEYSTORE_PASSWORD = "fakePasswordForTests";
    public static final String DEFAULT_SERVER_KEYSTORE_PATH = "/server.keystore";
    private static final File DEFAULT_SERVER_KEYSTORE_FILE = CcmBridge.createTempStore("/server.keystore");
    public static final String DEFAULT_SERVER_LOCALHOST_KEYSTORE_PATH = "/server_localhost.keystore";
    private static final File DEFAULT_SERVER_LOCALHOST_KEYSTORE_FILE = CcmBridge.createTempStore("/server_localhost.keystore");
    private static final Version V6_0_0 = Version.parse((String)"6.0.0");
    private static final Version V5_1_0 = Version.parse((String)"5.1.0");
    private static final Version V5_0_0 = Version.parse((String)"5.0.0");
    private static final Version V4_0_0 = Version.parse((String)"4.0.0");
    private static final Version V3_10 = Version.parse((String)"3.10");
    private static final Version V3_0_15 = Version.parse((String)"3.0.15");
    private static final Version V2_1_19 = Version.parse((String)"2.1.19");
    private final int[] nodes;
    private final Path configDirectory;
    private final AtomicBoolean started = new AtomicBoolean();
    private final AtomicBoolean created = new AtomicBoolean();
    private final String ipPrefix;
    private final Map<String, Object> cassandraConfiguration;
    private final Map<String, Object> dseConfiguration;
    private final List<String> rawDseYaml;
    private final List<String> createOptions;
    private final List<String> dseWorkloads;
    private final String jvmArgs;

    private CcmBridge(Path configDirectory, int[] nodes, String ipPrefix, Map<String, Object> cassandraConfiguration, Map<String, Object> dseConfiguration, List<String> dseConfigurationRawYaml, List<String> createOptions, Collection<String> jvmArgs, List<String> dseWorkloads) {
        this.configDirectory = configDirectory;
        this.nodes = nodes.length == 1 ? new int[]{nodes[0], 0} : nodes;
        this.ipPrefix = ipPrefix;
        this.cassandraConfiguration = cassandraConfiguration;
        this.dseConfiguration = dseConfiguration;
        this.rawDseYaml = dseConfigurationRawYaml;
        this.createOptions = createOptions;
        StringBuilder allJvmArgs = new StringBuilder("");
        String quote = CcmBridge.isWindows() ? "\"" : "";
        for (String jvmArg : jvmArgs) {
            allJvmArgs.append(" ");
            allJvmArgs.append(quote);
            allJvmArgs.append("--jvm_arg=");
            allJvmArgs.append(jvmArg);
            allJvmArgs.append(quote);
        }
        this.jvmArgs = allJvmArgs.toString();
        this.dseWorkloads = dseWorkloads;
    }

    private static boolean isWindows() {
        return System.getProperty("os.name", "").toLowerCase(Locale.US).contains("win");
    }

    public Optional<Version> getDseVersion() {
        return DSE_ENABLEMENT != false ? Optional.of(VERSION) : Optional.empty();
    }

    public Version getCassandraVersion() {
        if (!DSE_ENABLEMENT.booleanValue()) {
            return VERSION;
        }
        Version stableVersion = VERSION.nextStable();
        if (stableVersion.compareTo(V6_0_0) >= 0) {
            return V4_0_0;
        }
        if (stableVersion.compareTo(V5_1_0) >= 0) {
            return V3_10;
        }
        if (stableVersion.compareTo(V5_0_0) >= 0) {
            return V3_0_15;
        }
        return V2_1_19;
    }

    private String getCcmVersionString(Version version) {
        if (version.getMajor() == 4 && version.getMinor() == 0 && version.getPatch() == 0 && version.getPreReleaseLabels() != null) {
            StringBuilder sb = new StringBuilder();
            sb.append(version.getMajor()).append('.').append(version.getMinor());
            for (String preReleaseString : version.getPreReleaseLabels()) {
                sb.append('-').append(preReleaseString);
            }
            return sb.toString();
        }
        return version.toString();
    }

    public void create() {
        if (this.created.compareAndSet(false, true)) {
            if (INSTALL_DIRECTORY != null) {
                this.createOptions.add("--install-dir=" + new File(INSTALL_DIRECTORY).getAbsolutePath());
            } else if (BRANCH != null) {
                this.createOptions.add("-v git:" + BRANCH.trim().replaceAll("\"", ""));
            } else {
                this.createOptions.add("-v " + this.getCcmVersionString(VERSION));
            }
            if (DSE_ENABLEMENT.booleanValue()) {
                this.createOptions.add("--dse");
            }
            this.execute("create", CLUSTER_NAME, "-i", this.ipPrefix, "-n", Arrays.stream(this.nodes).mapToObj(n -> "" + n).collect(Collectors.joining(":")), this.createOptions.stream().collect(Collectors.joining(" ")));
            for (Map.Entry<String, Object> conf : this.cassandraConfiguration.entrySet()) {
                this.execute("updateconf", String.format("%s:%s", conf.getKey(), conf.getValue()));
            }
            if (this.getCassandraVersion().compareTo(Version.V2_2_0) >= 0) {
                this.execute("updateconf", "enable_user_defined_functions:true");
            }
            if (DSE_ENABLEMENT.booleanValue()) {
                for (Map.Entry<String, Object> conf : this.dseConfiguration.entrySet()) {
                    this.execute("updatedseconf", String.format("%s:%s", conf.getKey(), conf.getValue()));
                }
                for (String yaml : this.rawDseYaml) {
                    this.executeUnsanitized("updatedseconf", "-y", yaml);
                }
                if (!this.dseWorkloads.isEmpty()) {
                    this.execute("setworkload", String.join((CharSequence)",", this.dseWorkloads));
                }
            }
        }
    }

    public void nodetool(int node, String ... args) {
        this.execute(String.format("node%d nodetool %s", node, Joiner.on((String)" ").join((Object[])args)));
    }

    public void dsetool(int node, String ... args) {
        this.execute(String.format("node%d dsetool %s", node, Joiner.on((String)" ").join((Object[])args)));
    }

    public void reloadCore(int node, String keyspace, String table, boolean reindex) {
        this.dsetool(node, "reload_core", keyspace + "." + table, "reindex=" + reindex);
    }

    public void start() {
        if (this.started.compareAndSet(false, true)) {
            ArrayList cmdAndArgs = Lists.newArrayList((Object[])new String[]{"start", this.jvmArgs, "--wait-for-binary-proto"});
            this.overrideJvmVersionForDseWorkloads().ifPresent(jvmVersion -> cmdAndArgs.add(String.format("--jvm_version=%d", jvmVersion)));
            try {
                this.execute(cmdAndArgs.toArray(new String[0]));
            }
            catch (RuntimeException re) {
                this.executeCheckLogError();
                throw re;
            }
        }
    }

    public void stop() {
        if (this.started.compareAndSet(true, false)) {
            this.execute("stop");
        }
    }

    public void remove() {
        this.execute("remove");
    }

    public void pause(int n) {
        this.execute("node" + n, "pause");
    }

    public void resume(int n) {
        this.execute("node" + n, "resume");
    }

    public void start(int n) {
        ArrayList cmdAndArgs = Lists.newArrayList((Object[])new String[]{"node" + n, "start"});
        this.overrideJvmVersionForDseWorkloads().ifPresent(jvmVersion -> cmdAndArgs.add(String.format("--jvm_version=%d", jvmVersion)));
        this.execute(cmdAndArgs.toArray(new String[0]));
    }

    public void stop(int n) {
        this.execute("node" + n, "stop");
    }

    public void add(int n, String dc) {
        if (this.getDseVersion().isPresent()) {
            this.execute("add", "-i", this.ipPrefix + n, "-d", dc, "node" + n, "--dse");
        } else {
            this.execute("add", "-i", this.ipPrefix + n, "-d", dc, "node" + n);
        }
        this.start(n);
    }

    public void decommission(int n) {
        this.nodetool(n, "decommission");
    }

    synchronized void execute(String ... args) {
        String command = "ccm " + String.join((CharSequence)" ", args) + " --config-dir=" + this.configDirectory.toFile().getAbsolutePath();
        this.execute(CommandLine.parse((String)command));
    }

    synchronized void executeUnsanitized(String ... args) {
        String command = "ccm ";
        CommandLine cli = CommandLine.parse((String)command);
        for (String arg : args) {
            cli.addArgument(arg, false);
        }
        cli.addArgument("--config-dir=" + this.configDirectory.toFile().getAbsolutePath());
        this.execute(cli);
    }

    private void execute(CommandLine cli) {
        this.execute(cli, false);
    }

    private void executeCheckLogError() {
        String command = "ccm checklogerror --config-dir=" + this.configDirectory.toFile().getAbsolutePath();
        this.execute(CommandLine.parse((String)command), true);
    }

    private void execute(CommandLine cli, final boolean forceErrorLogging) {
        if (forceErrorLogging) {
            LOG.error("Executing: " + cli);
        } else {
            LOG.debug("Executing: " + cli);
        }
        ExecuteWatchdog watchDog = new ExecuteWatchdog(TimeUnit.MINUTES.toMillis(10L));
        try (LogOutputStream outStream = new LogOutputStream(){

            protected void processLine(String line, int logLevel) {
                if (forceErrorLogging) {
                    LOG.error("ccmout> {}", (Object)line);
                } else {
                    LOG.debug("ccmout> {}", (Object)line);
                }
            }
        };
             LogOutputStream errStream = new LogOutputStream(){

            protected void processLine(String line, int logLevel) {
                LOG.error("ccmerr> {}", (Object)line);
            }
        };){
            DefaultExecutor executor = new DefaultExecutor();
            PumpStreamHandler streamHandler = new PumpStreamHandler((OutputStream)outStream, (OutputStream)errStream);
            executor.setStreamHandler((ExecuteStreamHandler)streamHandler);
            executor.setWatchdog(watchDog);
            int retValue = executor.execute(cli);
            if (retValue != 0) {
                LOG.error("Non-zero exit code ({}) returned from executing ccm command: {}", (Object)retValue, (Object)cli);
            }
        }
        catch (IOException ex) {
            if (watchDog.killedProcess()) {
                throw new RuntimeException("The command '" + cli + "' was killed after 10 minutes");
            }
            throw new RuntimeException("The command '" + cli + "' failed to execute", ex);
        }
    }

    @Override
    public void close() {
        this.remove();
    }

    private static File createTempStore(String storePath) {
        File f = null;
        try {
            f = File.createTempFile("server", ".store");
            try (FileOutputStream os = new FileOutputStream(f);){
                f.deleteOnExit();
                Resources.copy((URL)CcmBridge.class.getResource(storePath), (OutputStream)os);
            }
        }
        catch (IOException e) {
            LOG.warn("Failure to write keystore, SSL-enabled servers may fail to start.", (Throwable)e);
        }
        return f;
    }

    private static int getCurrentJvmMajorVersion() {
        String version = System.getProperty("java.version");
        if (version.startsWith("1.")) {
            version = version.substring(2, 3);
        } else {
            int dot = version.indexOf(".");
            if (dot != -1) {
                version = version.substring(0, dot);
            }
        }
        return Integer.parseInt(version);
    }

    private Optional<Integer> overrideJvmVersionForDseWorkloads() {
        if (CcmBridge.getCurrentJvmMajorVersion() <= 8) {
            return Optional.empty();
        }
        if (!DSE_ENABLEMENT.booleanValue() || !this.getDseVersion().isPresent()) {
            return Optional.empty();
        }
        if (this.getDseVersion().get().compareTo(Version.parse((String)"6.8.19")) < 0) {
            return Optional.empty();
        }
        if (this.dseWorkloads.contains("graph")) {
            return Optional.of(8);
        }
        return Optional.empty();
    }

    public static Builder builder() {
        return new Builder();
    }

    static {
        if (DSE_ENABLEMENT.booleanValue()) {
            LOG.info("CCM Bridge configured with DSE version {}", (Object)VERSION);
        } else {
            LOG.info("CCM Bridge configured with Apache Cassandra version {}", (Object)VERSION);
        }
    }

    public static class Builder {
        private int[] nodes = new int[]{1};
        private final Map<String, Object> cassandraConfiguration = new LinkedHashMap<String, Object>();
        private final Map<String, Object> dseConfiguration = new LinkedHashMap<String, Object>();
        private final List<String> dseRawYaml = new ArrayList<String>();
        private final List<String> jvmArgs = new ArrayList<String>();
        private String ipPrefix = "127.0.0.";
        private final List<String> createOptions = new ArrayList<String>();
        private final List<String> dseWorkloads = new ArrayList<String>();
        private final Path configDirectory;

        private Builder() {
            try {
                this.configDirectory = Files.createTempDirectory("ccm", new FileAttribute[0]);
                this.configDirectory.toFile().deleteOnExit();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            this.withCassandraConfiguration("auto_snapshot", "false");
        }

        public Builder withCassandraConfiguration(String key, Object value) {
            this.cassandraConfiguration.put(key, value);
            return this;
        }

        public Builder withDseConfiguration(String key, Object value) {
            this.dseConfiguration.put(key, value);
            return this;
        }

        public Builder withDseConfiguration(String rawYaml) {
            this.dseRawYaml.add(rawYaml);
            return this;
        }

        public Builder withJvmArgs(String ... jvmArgs) {
            Collections.addAll(this.jvmArgs, jvmArgs);
            return this;
        }

        public Builder withNodes(int ... nodes) {
            this.nodes = nodes;
            return this;
        }

        public Builder withIpPrefix(String ipPrefix) {
            this.ipPrefix = ipPrefix;
            return this;
        }

        public Builder withCreateOption(String option) {
            this.createOptions.add(option);
            return this;
        }

        public Builder withSsl() {
            this.cassandraConfiguration.put("client_encryption_options.enabled", "true");
            this.cassandraConfiguration.put("client_encryption_options.optional", "false");
            this.cassandraConfiguration.put("client_encryption_options.keystore", DEFAULT_SERVER_KEYSTORE_FILE.getAbsolutePath());
            this.cassandraConfiguration.put("client_encryption_options.keystore_password", "fakePasswordForTests");
            return this;
        }

        public Builder withSslLocalhostCn() {
            this.cassandraConfiguration.put("client_encryption_options.enabled", "true");
            this.cassandraConfiguration.put("client_encryption_options.optional", "false");
            this.cassandraConfiguration.put("client_encryption_options.keystore", DEFAULT_SERVER_LOCALHOST_KEYSTORE_FILE.getAbsolutePath());
            this.cassandraConfiguration.put("client_encryption_options.keystore_password", "fakePasswordForTests");
            return this;
        }

        public Builder withSslAuth() {
            this.withSsl();
            this.cassandraConfiguration.put("client_encryption_options.require_client_auth", "true");
            this.cassandraConfiguration.put("client_encryption_options.truststore", DEFAULT_SERVER_TRUSTSTORE_FILE.getAbsolutePath());
            this.cassandraConfiguration.put("client_encryption_options.truststore_password", "fakePasswordForTests");
            return this;
        }

        public Builder withDseWorkloads(String ... workloads) {
            this.dseWorkloads.addAll(Arrays.asList(workloads));
            return this;
        }

        public CcmBridge build() {
            return new CcmBridge(this.configDirectory, this.nodes, this.ipPrefix, this.cassandraConfiguration, this.dseConfiguration, this.dseRawYaml, this.createOptions, this.jvmArgs, this.dseWorkloads);
        }
    }
}

