/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.support.processor.idempotent;

import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.api.management.ManagedAttribute;
import org.apache.camel.api.management.ManagedOperation;
import org.apache.camel.api.management.ManagedResource;
import org.apache.camel.spi.IdempotentRepository;
import org.apache.camel.support.LRUCache;
import org.apache.camel.support.LRUCacheFactory;
import org.apache.camel.support.service.ServiceSupport;
import org.apache.camel.util.FileUtil;
import org.apache.camel.util.IOHelper;
import org.apache.camel.util.ObjectHelper;
import org.apache.camel.util.Scanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedResource(description="File based idempotent repository")
public class FileIdempotentRepository
extends ServiceSupport
implements IdempotentRepository {
    private static final Logger LOG = LoggerFactory.getLogger(FileIdempotentRepository.class);
    private static final String STORE_DELIMITER = "\n";
    private final AtomicBoolean init = new AtomicBoolean();
    private Map<String, Object> cache;
    private File fileStore;
    private long maxFileStoreSize = 32768000L;
    private long dropOldestFileStore = 1000L;

    public FileIdempotentRepository() {
    }

    public FileIdempotentRepository(File fileStore, Map<String, Object> set) {
        this.fileStore = fileStore;
        this.cache = set;
    }

    public static IdempotentRepository fileIdempotentRepository(File fileStore) {
        return FileIdempotentRepository.fileIdempotentRepository(fileStore, 1000);
    }

    public static IdempotentRepository fileIdempotentRepository(File fileStore, int cacheSize) {
        return FileIdempotentRepository.fileIdempotentRepository(fileStore, LRUCacheFactory.newLRUCache(cacheSize));
    }

    public static IdempotentRepository fileIdempotentRepository(File fileStore, int cacheSize, long maxFileStoreSize) {
        FileIdempotentRepository repository = new FileIdempotentRepository(fileStore, LRUCacheFactory.newLRUCache(cacheSize));
        repository.setMaxFileStoreSize(maxFileStoreSize);
        return repository;
    }

    public static IdempotentRepository fileIdempotentRepository(File store, Map<String, Object> cache) {
        return new FileIdempotentRepository(store, cache);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ManagedOperation(description="Adds the key to the store")
    public boolean add(String key) {
        Map<String, Object> map = this.cache;
        synchronized (map) {
            if (this.cache.containsKey(key)) {
                return false;
            }
            this.cache.put(key, key);
            boolean containsInFile = this.containsStore(key);
            if (containsInFile) {
                return false;
            }
            this.appendToStore(key);
            if (this.maxFileStoreSize > 0L && this.fileStore.length() > this.maxFileStoreSize) {
                LOG.warn("Maximum capacity of file store: {} hit at {} bytes. Dropping {} oldest entries from the file store", new Object[]{this.fileStore, this.maxFileStoreSize, this.dropOldestFileStore});
                this.trunkStore();
            }
            return true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ManagedOperation(description="Does the store contain the given key")
    public boolean contains(String key) {
        Map<String, Object> map = this.cache;
        synchronized (map) {
            return this.cache.containsKey(key) || this.containsStore(key);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ManagedOperation(description="Remove the key from the store")
    public boolean remove(String key) {
        boolean answer;
        Map<String, Object> map = this.cache;
        synchronized (map) {
            answer = this.cache.remove(key) != null;
            this.removeFromStore(key);
        }
        return answer;
    }

    public boolean confirm(String key) {
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ManagedOperation(description="Clear the store (danger this removes all entries)")
    public void clear() {
        Map<String, Object> map = this.cache;
        synchronized (map) {
            this.cache.clear();
            if (this.cache instanceof LRUCache) {
                ((LRUCache)this.cache).cleanUp();
            }
            this.clearStore();
        }
    }

    public File getFileStore() {
        return this.fileStore;
    }

    public void setFileStore(File fileStore) {
        this.fileStore = fileStore;
    }

    @ManagedAttribute(description="The file path for the store")
    public String getFilePath() {
        return this.fileStore.getPath();
    }

    public Map<String, Object> getCache() {
        return this.cache;
    }

    public void setCache(Map<String, Object> cache) {
        this.cache = cache;
    }

    @ManagedAttribute(description="The maximum file size for the file store in bytes")
    public long getMaxFileStoreSize() {
        return this.maxFileStoreSize;
    }

    @ManagedAttribute(description="The maximum file size for the file store in bytes")
    public void setMaxFileStoreSize(long maxFileStoreSize) {
        this.maxFileStoreSize = maxFileStoreSize;
    }

    public long getDropOldestFileStore() {
        return this.dropOldestFileStore;
    }

    @ManagedAttribute(description="Number of oldest elements to drop from file store if maximum file size reached")
    public void setDropOldestFileStore(long dropOldestFileStore) {
        this.dropOldestFileStore = dropOldestFileStore;
    }

    public void setCacheSize(int size) {
        if (this.cache != null && !(this.cache instanceof LRUCache)) {
            throw new IllegalArgumentException("Setting cache size is only possible when using the default LRUCache cache implementation");
        }
        if (this.cache != null) {
            this.cache.clear();
        }
        this.cache = LRUCacheFactory.newLRUCache(size);
    }

    @ManagedAttribute(description="The current 1st-level cache size")
    public int getCacheSize() {
        if (this.cache != null) {
            return this.cache.size();
        }
        return 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @ManagedOperation(description="Reset and reloads the file store")
    public synchronized void reset() throws IOException {
        Map<String, Object> map = this.cache;
        synchronized (map) {
            if (this.cache instanceof LRUCache) {
                ((LRUCache)this.cache).cleanUp();
            }
            this.cache.clear();
            this.loadStore();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected boolean containsStore(String key) {
        if (this.fileStore == null) return false;
        if (!this.fileStore.exists()) {
            return false;
        }
        try (Scanner scanner = new Scanner(this.fileStore, null, STORE_DELIMITER);){
            String line;
            do {
                if (!scanner.hasNext()) return false;
            } while (!(line = scanner.next()).equals(key));
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            throw RuntimeCamelException.wrapRuntimeCamelException((Throwable)e);
        }
    }

    protected void appendToStore(String key) {
        LOG.debug("Appending: {} to idempotent filestore: {}", (Object)key, (Object)this.fileStore);
        FileOutputStream fos = null;
        try {
            File storeParentDirectory = this.fileStore.getParentFile();
            if (storeParentDirectory != null && !storeParentDirectory.exists()) {
                LOG.info("Parent directory of file store {} doesn't exist. Creating.", (Object)this.fileStore);
                if (this.fileStore.getParentFile().mkdirs()) {
                    LOG.info("Parent directory of filestore: {} successfully created.", (Object)this.fileStore);
                } else {
                    LOG.warn("Parent directory of filestore: {} cannot be created.", (Object)this.fileStore);
                }
            }
            if (!this.fileStore.exists()) {
                FileUtil.createNewFile((File)this.fileStore);
            }
            fos = new FileOutputStream(this.fileStore, true);
            fos.write(key.getBytes());
            fos.write(STORE_DELIMITER.getBytes());
        }
        catch (IOException e) {
            try {
                throw RuntimeCamelException.wrapRuntimeCamelException((Throwable)e);
            }
            catch (Throwable throwable) {
                IOHelper.close(fos, (String)"Appending to file idempotent repository", (Logger)LOG);
                throw throwable;
            }
        }
        IOHelper.close((Closeable)fos, (String)"Appending to file idempotent repository", (Logger)LOG);
    }

    protected synchronized void removeFromStore(String key) {
        LOG.debug("Removing: {} from idempotent filestore: {}", (Object)key, (Object)this.fileStore);
        ArrayList<String> lines = new ArrayList<String>();
        boolean found = false;
        try {
            Scanner scanner = new Scanner(this.fileStore, null, STORE_DELIMITER);
            Object object = null;
            try {
                while (scanner.hasNext()) {
                    String line = scanner.next();
                    if (key.equals(line)) {
                        found = true;
                        continue;
                    }
                    lines.add(line);
                }
            }
            catch (Throwable line) {
                object = line;
                throw line;
            }
            finally {
                if (scanner != null) {
                    if (object != null) {
                        try {
                            scanner.close();
                        }
                        catch (Throwable line) {
                            ((Throwable)object).addSuppressed(line);
                        }
                    } else {
                        scanner.close();
                    }
                }
            }
        }
        catch (IOException e) {
            throw RuntimeCamelException.wrapRuntimeCamelException((Throwable)e);
        }
        if (found) {
            LOG.debug("Rewriting idempotent filestore: {} due to key: {} removed", (Object)this.fileStore, (Object)key);
            FileOutputStream fos = null;
            try {
                fos = new FileOutputStream(this.fileStore);
                for (String line : lines) {
                    fos.write(line.getBytes());
                    fos.write(STORE_DELIMITER.getBytes());
                }
            }
            catch (IOException e) {
                throw RuntimeCamelException.wrapRuntimeCamelException((Throwable)e);
            }
            finally {
                IOHelper.close((Closeable)fos, (String)"Rewriting file idempotent repository", (Logger)LOG);
            }
        }
    }

    protected void clearStore() {
        try {
            FileUtil.deleteFile((File)this.fileStore);
            FileUtil.createNewFile((File)this.fileStore);
        }
        catch (IOException e) {
            throw RuntimeCamelException.wrapRuntimeCamelException((Throwable)e);
        }
    }

    protected synchronized void trunkStore() {
        if (this.fileStore == null || !this.fileStore.exists()) {
            return;
        }
        LOG.debug("Trunking: {} oldest entries from idempotent filestore: {}", (Object)this.dropOldestFileStore, (Object)this.fileStore);
        ArrayList<String> lines = new ArrayList<String>();
        int count = 0;
        try {
            Scanner scanner = new Scanner(this.fileStore, null, STORE_DELIMITER);
            Object object = null;
            try {
                while (scanner.hasNext()) {
                    String line = scanner.next();
                    if ((long)(++count) <= this.dropOldestFileStore) continue;
                    lines.add(line);
                }
            }
            catch (Throwable line) {
                object = line;
                throw line;
            }
            finally {
                if (scanner != null) {
                    if (object != null) {
                        try {
                            scanner.close();
                        }
                        catch (Throwable line) {
                            ((Throwable)object).addSuppressed(line);
                        }
                    } else {
                        scanner.close();
                    }
                }
            }
        }
        catch (IOException e) {
            throw RuntimeCamelException.wrapRuntimeCamelException((Throwable)e);
        }
        if (!lines.isEmpty()) {
            LOG.debug("Rewriting idempotent filestore: {} with {} entries:", (Object)this.fileStore, (Object)lines.size());
            FileOutputStream fos = null;
            try {
                fos = new FileOutputStream(this.fileStore);
                for (String line : lines) {
                    fos.write(line.getBytes());
                    fos.write(STORE_DELIMITER.getBytes());
                }
            }
            catch (IOException e) {
                throw RuntimeCamelException.wrapRuntimeCamelException((Throwable)e);
            }
            finally {
                IOHelper.close((Closeable)fos, (String)"Rewriting file idempotent repository", (Logger)LOG);
            }
        } else {
            LOG.debug("Clearing idempotent filestore: {}", (Object)this.fileStore);
            this.clearStore();
        }
    }

    protected void cleanup() {
        if (this.cache instanceof LRUCache) {
            ((LRUCache)this.cache).cleanUp();
        }
    }

    protected void loadStore() throws IOException {
        if (!this.fileStore.exists()) {
            boolean created;
            boolean mkdirsResult;
            LOG.debug("Creating filestore: {}", (Object)this.fileStore);
            File parent = this.fileStore.getParentFile();
            if (parent != null && !(mkdirsResult = parent.mkdirs())) {
                LOG.error("Couldn't create the filestore at {} because creating the directory has failed", (Object)parent);
            }
            if (!(created = FileUtil.createNewFile((File)this.fileStore))) {
                throw new IOException("Cannot create filestore: " + this.fileStore);
            }
        }
        LOG.trace("Loading to 1st level cache from idempotent filestore: {}", (Object)this.fileStore);
        this.cache.clear();
        try (Scanner scanner = new Scanner(this.fileStore, null, STORE_DELIMITER);){
            while (scanner.hasNext()) {
                String line = scanner.next();
                this.cache.put(line, line);
            }
        }
        catch (IOException e) {
            throw RuntimeCamelException.wrapRuntimeCamelException((Throwable)e);
        }
        LOG.debug("Loaded {} to the 1st level cache from idempotent filestore: {}", (Object)this.cache.size(), (Object)this.fileStore);
    }

    protected void doStart() throws Exception {
        ObjectHelper.notNull((Object)this.fileStore, (String)"fileStore", (Object)((Object)this));
        if (this.cache == null) {
            this.cache = LRUCacheFactory.newLRUCache(1000);
        }
        if (this.init.compareAndSet(false, true)) {
            this.loadStore();
        }
    }

    protected void doStop() throws Exception {
        if (this.cache instanceof LRUCache) {
            ((LRUCache)this.cache).cleanUp();
        }
        this.cache.clear();
        this.init.set(false);
    }
}

