
/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

    package org.apache.arrow.vector.complex.impl;

import java.util.Locale;

import static org.apache.arrow.util.Preconditions.checkArgument;
import static org.apache.arrow.util.Preconditions.checkState;

import com.google.flatbuffers.FlatBufferBuilder;

import org.apache.arrow.memory.*;
import org.apache.arrow.util.Preconditions;
import org.apache.arrow.vector.types.Types;
import org.apache.arrow.vector.types.Types.*;
import org.apache.arrow.vector.types.pojo.*;
import org.apache.arrow.vector.types.pojo.ArrowType.*;
import org.apache.arrow.vector.types.*;
import org.apache.arrow.vector.*;
import org.apache.arrow.vector.holders.*;
import org.apache.arrow.vector.util.*;
import org.apache.arrow.vector.complex.*;
import org.apache.arrow.vector.complex.reader.*;
import org.apache.arrow.vector.complex.impl.*;
import org.apache.arrow.vector.complex.writer.*;
import org.apache.arrow.vector.complex.writer.BaseWriter.StructWriter;
import org.apache.arrow.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.arrow.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.arrow.vector.util.JsonStringArrayList;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.ZonedDateTime;

/**
 * This FieldWriter implementation delegates all FieldWriter API calls to an inner FieldWriter. This
 * inner field writer can start as a specific type, and this class will promote the writer to a
 * UnionWriter if a call is made that the specifically typed writer cannot handle. A new UnionVector
 * is created, wrapping the original vector, and replaces the original vector in the parent vector,
 * which can be either an AbstractStructVector or a ListViewVector.
 *
 * <p>The writer used can either be for single elements (struct) or lists.
 */
public class PromotableViewWriter extends PromotableWriter {

  public PromotableViewWriter(ValueVector v, FixedSizeListVector fixedListVector) {
    super(v, fixedListVector);
  }

  public PromotableViewWriter(ValueVector v, FixedSizeListVector fixedListVector,
      NullableStructWriterFactory nullableStructWriterFactory) {
    super(v, fixedListVector, nullableStructWriterFactory);
  }

  public PromotableViewWriter(ValueVector v, LargeListVector largeListVector) {
    super(v, largeListVector);
  }

  public PromotableViewWriter(ValueVector v, LargeListVector largeListVector,
      NullableStructWriterFactory nullableStructWriterFactory) {
    super(v, largeListVector, nullableStructWriterFactory);
  }

  public PromotableViewWriter(ValueVector v, ListVector listVector) {
    super(v, listVector);
  }

  public PromotableViewWriter(ValueVector v, ListVector listVector,
      NullableStructWriterFactory nullableStructWriterFactory) {
    super(v, listVector, nullableStructWriterFactory);
  }

  public PromotableViewWriter(ValueVector v, ListViewVector listViewVector,
      NullableStructWriterFactory nullableStructWriterFactory) {
    super(v, listViewVector, nullableStructWriterFactory);
  }

  public PromotableViewWriter(ValueVector v, LargeListViewVector largeListViewVector) {
    super(v, largeListViewVector);
  }

  public PromotableViewWriter(ValueVector v, LargeListViewVector largeListViewVector,
      NullableStructWriterFactory nullableStructWriterFactory) {
    super(v, largeListViewVector, nullableStructWriterFactory);
  }

  public PromotableViewWriter(ValueVector v, AbstractStructVector parentContainer) {
    super(v, parentContainer);
  }

  public PromotableViewWriter(ValueVector v, AbstractStructVector parentContainer,
      NullableStructWriterFactory nullableStructWriterFactory) {
    super(v, parentContainer, nullableStructWriterFactory);
  }

  @Override
  protected FieldWriter getWriter(MinorType type, ArrowType arrowType) {
    if (state == State.UNION) {
      if (requiresArrowType(type)) {
        writer = ((UnionWriter) writer).toViewWriter();
        ((UnionViewWriter) writer).getWriter(type, arrowType);
      } else {
        writer = ((UnionWriter) writer).toViewWriter();
        ((UnionViewWriter) writer).getWriter(type);
      }
    } else if (state == State.UNTYPED) {
      if (type == null) {
        // ???
        return null;
      }
      if (arrowType == null) {
        arrowType = type.getType();
      }
      FieldType fieldType = new FieldType(addVectorAsNullable, arrowType, null, null);
      ValueVector v;
      if (listVector != null) {
        v = listVector.addOrGetVector(fieldType).getVector();
      } else if (fixedListVector != null) {
        v = fixedListVector.addOrGetVector(fieldType).getVector();
      } else if (listViewVector != null) {
        v = listViewVector.addOrGetVector(fieldType).getVector();
      } else if (largeListVector != null) {
        v = largeListVector.addOrGetVector(fieldType).getVector();
      } else {
        v = largeListViewVector.addOrGetVector(fieldType).getVector();
      }
      v.allocateNew();
      setWriter(v);
      writer.setPosition(position);
    } else if (type != this.type) {
      promoteToUnion();
      if (requiresArrowType(type)) {
        writer = ((UnionWriter) writer).toViewWriter();
        ((UnionViewWriter) writer).getWriter(type, arrowType);
      } else {
        writer = ((UnionWriter) writer).toViewWriter();
        ((UnionViewWriter) writer).getWriter(type);
      }
    }
    return writer;
  }

  @Override
  public StructWriter struct() {
    return getWriter(MinorType.LISTVIEW).struct();
  }


  @Override
  public TinyIntWriter tinyInt() {
    return getWriter(MinorType.LISTVIEW).tinyInt();
  }


  @Override
  public UInt1Writer uInt1() {
    return getWriter(MinorType.LISTVIEW).uInt1();
  }


  @Override
  public UInt2Writer uInt2() {
    return getWriter(MinorType.LISTVIEW).uInt2();
  }


  @Override
  public SmallIntWriter smallInt() {
    return getWriter(MinorType.LISTVIEW).smallInt();
  }


  @Override
  public Float2Writer float2() {
    return getWriter(MinorType.LISTVIEW).float2();
  }


  @Override
  public IntWriter integer() {
    return getWriter(MinorType.LISTVIEW).integer();
  }


  @Override
  public UInt4Writer uInt4() {
    return getWriter(MinorType.LISTVIEW).uInt4();
  }


  @Override
  public Float4Writer float4() {
    return getWriter(MinorType.LISTVIEW).float4();
  }


  @Override
  public DateDayWriter dateDay() {
    return getWriter(MinorType.LISTVIEW).dateDay();
  }


  @Override
  public IntervalYearWriter intervalYear() {
    return getWriter(MinorType.LISTVIEW).intervalYear();
  }


  @Override
  public TimeSecWriter timeSec() {
    return getWriter(MinorType.LISTVIEW).timeSec();
  }


  @Override
  public TimeMilliWriter timeMilli() {
    return getWriter(MinorType.LISTVIEW).timeMilli();
  }


  @Override
  public BigIntWriter bigInt() {
    return getWriter(MinorType.LISTVIEW).bigInt();
  }


  @Override
  public UInt8Writer uInt8() {
    return getWriter(MinorType.LISTVIEW).uInt8();
  }


  @Override
  public Float8Writer float8() {
    return getWriter(MinorType.LISTVIEW).float8();
  }


  @Override
  public DateMilliWriter dateMilli() {
    return getWriter(MinorType.LISTVIEW).dateMilli();
  }


  @Override
  public DurationWriter duration() {
    return getWriter(MinorType.LISTVIEW).duration();
  }


  @Override
  public TimeStampSecWriter timeStampSec() {
    return getWriter(MinorType.LISTVIEW).timeStampSec();
  }


  @Override
  public TimeStampMilliWriter timeStampMilli() {
    return getWriter(MinorType.LISTVIEW).timeStampMilli();
  }


  @Override
  public TimeStampMicroWriter timeStampMicro() {
    return getWriter(MinorType.LISTVIEW).timeStampMicro();
  }


  @Override
  public TimeStampNanoWriter timeStampNano() {
    return getWriter(MinorType.LISTVIEW).timeStampNano();
  }


  @Override
  public TimeStampSecTZWriter timeStampSecTZ() {
    return getWriter(MinorType.LISTVIEW).timeStampSecTZ();
  }


  @Override
  public TimeStampMilliTZWriter timeStampMilliTZ() {
    return getWriter(MinorType.LISTVIEW).timeStampMilliTZ();
  }


  @Override
  public TimeStampMicroTZWriter timeStampMicroTZ() {
    return getWriter(MinorType.LISTVIEW).timeStampMicroTZ();
  }


  @Override
  public TimeStampNanoTZWriter timeStampNanoTZ() {
    return getWriter(MinorType.LISTVIEW).timeStampNanoTZ();
  }


  @Override
  public TimeMicroWriter timeMicro() {
    return getWriter(MinorType.LISTVIEW).timeMicro();
  }


  @Override
  public TimeNanoWriter timeNano() {
    return getWriter(MinorType.LISTVIEW).timeNano();
  }


  @Override
  public IntervalDayWriter intervalDay() {
    return getWriter(MinorType.LISTVIEW).intervalDay();
  }


  @Override
  public IntervalMonthDayNanoWriter intervalMonthDayNano() {
    return getWriter(MinorType.LISTVIEW).intervalMonthDayNano();
  }


  @Override
  public Decimal256Writer decimal256() {
    return getWriter(MinorType.LISTVIEW).decimal256();
  }


  @Override
  public DecimalWriter decimal() {
    return getWriter(MinorType.LISTVIEW).decimal();
  }


  @Override
  public FixedSizeBinaryWriter fixedSizeBinary() {
    return getWriter(MinorType.LISTVIEW).fixedSizeBinary();
  }


  @Override
  public VarBinaryWriter varBinary() {
    return getWriter(MinorType.LISTVIEW).varBinary();
  }


  @Override
  public VarCharWriter varChar() {
    return getWriter(MinorType.LISTVIEW).varChar();
  }


  @Override
  public ViewVarBinaryWriter viewVarBinary() {
    return getWriter(MinorType.LISTVIEW).viewVarBinary();
  }


  @Override
  public ViewVarCharWriter viewVarChar() {
    return getWriter(MinorType.LISTVIEW).viewVarChar();
  }


  @Override
  public LargeVarCharWriter largeVarChar() {
    return getWriter(MinorType.LISTVIEW).largeVarChar();
  }


  @Override
  public LargeVarBinaryWriter largeVarBinary() {
    return getWriter(MinorType.LISTVIEW).largeVarBinary();
  }


  @Override
  public BitWriter bit() {
    return getWriter(MinorType.LISTVIEW).bit();
  }


  @Override
  public void allocate() {
    getWriter().allocate();
  }

  @Override
  public void clear() {
    getWriter().clear();
  }

  @Override
  public Field getField() {
    return getWriter().getField();
  }

  @Override
  public int getValueCapacity() {
    return getWriter().getValueCapacity();
  }

  @Override
  public void close() throws Exception {
    getWriter().close();
  }
}
