/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import org.apache.accumulo.core.client.AccumuloException;
import org.apache.accumulo.core.conf.AccumuloConfiguration;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.util.AddressUtil;
import org.apache.accumulo.core.volume.Volume;
import org.apache.accumulo.fate.ReadOnlyStore;
import org.apache.accumulo.fate.TStore;
import org.apache.accumulo.fate.ZooStore;
import org.apache.accumulo.fate.util.UtilWaitThread;
import org.apache.accumulo.fate.zookeeper.IZooReaderWriter;
import org.apache.accumulo.server.ServerConstants;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.server.fs.VolumeManager;
import org.apache.accumulo.server.util.time.SimpleTimer;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServerUtil {
    private static final Logger log = LoggerFactory.getLogger(ServerUtil.class);

    public static synchronized void updateAccumuloVersion(VolumeManager fs, int oldVersion) {
        for (Volume volume : fs.getVolumes()) {
            try {
                if (ServerUtil.getAccumuloPersistentVersion(volume) != oldVersion) continue;
                log.debug("Attempting to upgrade {}", (Object)volume);
                Path dataVersionLocation = ServerConstants.getDataVersionLocation(volume);
                fs.create(new Path(dataVersionLocation, Integer.toString(9))).close();
                Path prevDataVersionLoc = new Path(dataVersionLocation, Integer.toString(oldVersion));
                if (fs.delete(prevDataVersionLoc)) continue;
                throw new RuntimeException("Could not delete previous data version location (" + prevDataVersionLoc + ") for " + volume);
            }
            catch (IOException e) {
                throw new RuntimeException("Unable to set accumulo version: an error occurred.", e);
            }
        }
    }

    public static synchronized int getAccumuloPersistentVersion(FileSystem fs, Path path) {
        try {
            FileStatus[] files = fs.listStatus(path);
            int dataVersion = files == null || files.length == 0 ? -1 : Integer.parseInt(files[0].getPath().getName());
            return dataVersion;
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to read accumulo version: an error occurred.", e);
        }
    }

    public static synchronized int getAccumuloPersistentVersion(Volume v) {
        Path path = ServerConstants.getDataVersionLocation(v);
        return ServerUtil.getAccumuloPersistentVersion(v.getFileSystem(), path);
    }

    public static synchronized int getAccumuloPersistentVersion(VolumeManager fs) {
        return ServerUtil.getAccumuloPersistentVersion(fs.getVolumes().iterator().next());
    }

    public static synchronized Path getAccumuloInstanceIdPath(VolumeManager fs) {
        Volume v = fs.getVolumes().iterator().next();
        return ServerConstants.getInstanceIdLocation(v);
    }

    public static void init(ServerContext context, String application) {
        AccumuloConfiguration conf = context.getConfiguration();
        log.info("{} starting", (Object)application);
        log.info("Instance {}", (Object)context.getInstanceID());
        int dataVersion = ServerUtil.getAccumuloPersistentVersion(context.getVolumeManager());
        log.info("Data Version {}", (Object)dataVersion);
        ServerUtil.waitForZookeeperAndHdfs(context);
        ServerUtil.ensureDataVersionCompatible(dataVersion);
        TreeMap<String, String> sortedProps = new TreeMap<String, String>();
        for (Map.Entry entry : conf) {
            sortedProps.put((String)entry.getKey(), (String)entry.getValue());
        }
        for (Map.Entry entry : sortedProps.entrySet()) {
            String key;
            log.info("{} = {}", (Object)key, (Object)(Property.isSensitive((String)(key = (String)entry.getKey())) ? "<hidden>" : entry.getValue()));
            Property prop = Property.getPropertyByKey((String)key);
            if (prop == null || !conf.isPropertySet(prop, false) || !prop.isDeprecated()) continue;
            Property replacedBy = prop.replacedBy();
            if (replacedBy != null) {
                log.warn("{} is deprecated, use {} instead.", (Object)prop.getKey(), (Object)replacedBy.getKey());
                continue;
            }
            log.warn("{} is deprecated", (Object)prop.getKey());
        }
        ServerUtil.monitorSwappiness(conf);
        String SSL = "SSL";
        for (Property sslProtocolProperty : Arrays.asList(Property.RPC_SSL_CLIENT_PROTOCOL, Property.RPC_SSL_ENABLED_PROTOCOLS, Property.MONITOR_SSL_INCLUDE_PROTOCOLS)) {
            String value = conf.get(sslProtocolProperty);
            if (!value.contains("SSL")) continue;
            log.warn("It is recommended that {} only allow TLS", (Object)sslProtocolProperty);
        }
    }

    public static void ensureDataVersionCompatible(int dataVersion) {
        if (!ServerConstants.CAN_RUN.contains(dataVersion)) {
            throw new IllegalStateException("This version of accumulo (2.0.0) is not compatible with files stored using data version " + dataVersion);
        }
    }

    public static boolean persistentVersionNeedsUpgrade(int accumuloPersistentVersion) {
        return ServerConstants.NEEDS_UPGRADE.contains(accumuloPersistentVersion);
    }

    public static void monitorSwappiness(AccumuloConfiguration config) {
        SimpleTimer.getInstance(config).schedule(() -> {
            block8: {
                try {
                    String procFile = "/proc/sys/vm/swappiness";
                    File swappiness = new File(procFile);
                    if (!swappiness.exists() || !swappiness.canRead()) break block8;
                    try (FileInputStream is = new FileInputStream(procFile);){
                        byte[] buffer = new byte[10];
                        int bytes = ((InputStream)is).read(buffer);
                        String setting = new String(buffer, 0, bytes, StandardCharsets.UTF_8);
                        setting = setting.trim();
                        if (bytes > 0 && Integer.parseInt(setting) > 10) {
                            log.warn("System swappiness setting is greater than ten ({}) which can cause time-sensitive operations to be delayed. Accumulo is time sensitive because it needs to maintain distributed lock agreement.", (Object)setting);
                        }
                    }
                }
                catch (Throwable t) {
                    log.error("", t);
                }
            }
        }, 1000L, 600000L);
    }

    public static void waitForZookeeperAndHdfs(ServerContext context) {
        log.info("Attempting to talk to zookeeper");
        while (true) {
            try {
                context.getZooReaderWriter().getChildren("/accumulo");
            }
            catch (InterruptedException interruptedException) {
                continue;
            }
            catch (KeeperException ex) {
                log.info("Waiting for accumulo to be initialized");
                UtilWaitThread.sleepUninterruptibly((long)1L, (TimeUnit)TimeUnit.SECONDS);
                continue;
            }
            break;
        }
        log.info("ZooKeeper connected and initialized, attempting to talk to HDFS");
        long sleep = 1000L;
        int unknownHostTries = 3;
        while (true) {
            try {
                if (context.getVolumeManager().isReady()) break;
                log.warn("Waiting for the NameNode to leave safemode");
            }
            catch (IOException ex) {
                log.warn("Unable to connect to HDFS", (Throwable)ex);
            }
            catch (IllegalArgumentException e) {
                if (e.getCause() instanceof UnknownHostException) {
                    if (unknownHostTries <= 0) {
                        log.error("Unable to connect to HDFS and exceeded the maximum number of retries.", (Throwable)e);
                        throw e;
                    }
                    log.warn("Unable to connect to HDFS, will retry. cause: {}", e.getCause());
                    int ttl = AddressUtil.getAddressCacheNegativeTtl((UnknownHostException)((UnknownHostException)e.getCause()));
                    sleep = Math.max(sleep, (long)(ttl + 1) * 1000L);
                    --unknownHostTries;
                }
                throw e;
            }
            log.info("Backing off due to failure; current sleep period is {} seconds", (Object)((double)sleep / 1000.0));
            UtilWaitThread.sleepUninterruptibly((long)sleep, (TimeUnit)TimeUnit.MILLISECONDS);
            sleep = Math.min(60000L, sleep * 2L);
        }
        log.info("Connected to HDFS");
    }

    public static void abortIfFateTransactions(ServerContext context) {
        try {
            ReadOnlyStore fate = new ReadOnlyStore((TStore)new ZooStore(context.getZooKeeperRoot() + "/fate", (IZooReaderWriter)context.getZooReaderWriter()));
            if (!fate.list().isEmpty()) {
                throw new AccumuloException("Aborting upgrade because there are outstanding FATE transactions from a previous Accumulo version. You can start the tservers and then use the shell to delete completed  transactions. If there are uncomplete transactions, you will need to roll back and fix those issues. Please see the following page for more information:  https://accumulo.apache.org/docs/2.x/troubleshooting/advanced#upgrade-issues");
            }
        }
        catch (Exception exception) {
            log.error("Problem verifying Fate readiness", (Throwable)exception);
            System.exit(1);
        }
    }
}

