/*===========================================================================
  Copyright (C) 2016 by the Okapi Framework contributors
-----------------------------------------------------------------------------
  This library is free software; you can redistribute it and/or modify it 
  under the terms of the GNU Lesser General Public License as published by 
  the Free Software Foundation; either version 2.1 of the License, or (at 
  your option) any later version.

  This library is distributed in the hope that it will be useful, but 
  WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
  General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License 
  along with this library; if not, write to the Free Software Foundation, 
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

  See also the full LGPL text here: http://www.gnu.org/copyleft/lesser.html
===========================================================================*/

package net.sf.okapi.lib.serialization.filter;

import com.google.protobuf.TextFormat;
import com.google.protobuf.util.JsonFormat;
import net.sf.okapi.common.Event;
import net.sf.okapi.common.EventType;
import net.sf.okapi.common.IParameters;
import net.sf.okapi.common.exceptions.OkapiBadFilterInputException;
import net.sf.okapi.common.filters.AbstractFilter;
import net.sf.okapi.common.filters.EventBuilder;
import net.sf.okapi.common.filters.FilterConfiguration;
import net.sf.okapi.common.filters.IFilter;
import net.sf.okapi.common.resource.RawDocument;
import net.sf.okapi.lib.serialization.textunitflat.Proto2TextUnitFlat;
import net.sf.okapi.lib.serialization.writer.ProtoBufferTextUnitFlatWriter;
import net.sf.okapi.proto.textunitflat.TextUnit;
import net.sf.okapi.proto.textunitflat.TextUnits;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;

/**
 * An {@link IFilter} that returns serialized {@link Event}s generated by
 * {@link ProtoBufferTextUnitFlatWriter}
 * 
 * @author jimh
 *
 */
public class TextUnitFlatFilter extends AbstractFilter {
	private static final String MIMETYPE = "text/x-flat-text-unit";
	private RawDocument input;	
	private EventBuilder eventBuilder;
	private boolean binary = false;

	public TextUnitFlatFilter() {
		binary = false;
		setMimeType(MIMETYPE);
		setMultilingual(false);
		setName("okf_textunitflat");
		setDisplayName("Flat TextUnit Filter");
		addConfiguration(new FilterConfiguration(getName(), MIMETYPE, getClass().getName(),
				"Protobuffer Flat TextUnit", "Configuration for Flat TextUnit files of various types", null, ".json;"));
	}

	@Override
	public void open(RawDocument input) {
		open(input, false);
	}

	@Override
	public void open(RawDocument input, boolean generateSkeleton) {		
		this.input = input;
		this.input.setEncoding(StandardCharsets.UTF_8);

		// create EventBuilder with document name as rootId
		if (eventBuilder == null) {
			eventBuilder = new EventBuilder(getParentId(), this);
		} else {
			eventBuilder.reset(getParentId(), this);
		}
		eventBuilder.setMimeType(MIMETYPE);
		eventBuilder.setPreserveWhitespace(true);
		eventBuilder.addFilterEvent(createStartFilterEvent());

		TextUnits tus;
		try {
			tus = loadFromText(input.getStream());
		} catch (OkapiBadFilterInputException e1) {
			try {
				tus = loadFromBinary(input.getStream());
				binary = true;
			} catch (OkapiBadFilterInputException e2) {
				try {
					tus = loadFromJson(input.getStream());
				} catch (OkapiBadFilterInputException e3) {
					throw new OkapiBadFilterInputException("Error loading TextUnits. Bad input format.");
				}
			}
		}

		for (TextUnit tu : tus.getTextUnitsList()) {
			eventBuilder.addFilterEvent(new Event(EventType.TEXT_UNIT,
					Proto2TextUnitFlat.toTextUnit(tu)));
		}

		eventBuilder.addFilterEvent(createEndFilterEvent());
	}

	@Override
	public void setParameters(IParameters params) {
	}
	
	@Override
	public IParameters getParameters() {
		return null;
	}

	@Override
	public boolean hasNext() {
		return eventBuilder.hasNext();
	}

	@Override
	public Event next() {
		return eventBuilder.next();
	}
	private TextUnits loadFromBinary(InputStream inputStream) {
		try {
			return TextUnits.parseFrom(inputStream);
		} catch (IOException e) {
			throw new OkapiBadFilterInputException("Error parsing or converting Protobuf TextUnits", e);
		}
	}
	private TextUnits loadFromText(InputStream inputStream) {
		try (Reader reader = new InputStreamReader(inputStream, StandardCharsets.UTF_8)) {
			TextUnits.Builder builder = TextUnits.newBuilder();
			TextFormat.Parser.newBuilder().build().merge(reader, builder);
			return builder.build();
		} catch (IOException e) {
			throw new OkapiBadFilterInputException("Error parsing or converting Protobuf Text TextUnits", e);
		}
	}
	private TextUnits loadFromJson(InputStream inputStream) {
		try (Reader reader = new InputStreamReader(inputStream, StandardCharsets.UTF_8)) {
			TextUnits.Builder builder = TextUnits.newBuilder();
			JsonFormat.parser().merge(reader, builder);
			return builder.build();
		} catch (IOException e) {
			throw new OkapiBadFilterInputException("Error parsing or converting JSON TextUnits", e);
		}
	}

	@Override
	protected boolean isUtf8Encoding() {
		return binary ? false : true;
	}

	@Override
	protected boolean isUtf8Bom() {
		return false;
	}

	@Override
	public void close() {
		if (input != null) {
			input.close();
		}
	}
}
