package it.smartdust.entitydtomapper;

import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.modelmapper.ModelMapper;
import org.modelmapper.convention.MatchingStrategies;
import org.springframework.data.jpa.repository.JpaRepository;

import jakarta.persistence.EntityNotFoundException;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.OneToOne;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class EntityDtoMapperImpl<T, S> implements EntityDtoMapper<T, S> {

	private final ModelMapper mapper;

	public EntityDtoMapperImpl() {
		this.mapper = new ModelMapper();

		mapper.getConfiguration().setCollectionsMergeEnabled(false);
		mapper.getConfiguration().setMatchingStrategy(MatchingStrategies.STRICT);
	}

	@SuppressWarnings("rawtypes")
	@Override
	public T toEntity(S sourceDto, Class<T> targetEntityClass) {

		Object entity = null;

		Method method;
		try {
			method = sourceDto.getClass().getMethod("getId");

			Long dtoId = (Long) method.invoke(sourceDto, new Object[0]);

			StringBuilder sb = new StringBuilder();

			if (dtoId != null) {
				// Update the existing entity

				String mainEntityRepositoryName = targetEntityClass.getSimpleName();
				mainEntityRepositoryName = mainEntityRepositoryName.substring(0, 1).toLowerCase()
						+ mainEntityRepositoryName.substring(1);
				mainEntityRepositoryName = sb.append(mainEntityRepositoryName).append("Repository").toString();
				JpaRepository mainEntityRepo = (JpaRepository) ApplicationContextProvider.getApplicationContext()
						.getBean(mainEntityRepositoryName);

				entity = mainEntityRepo.findById(dtoId)
						.orElseThrow(() -> new EntityNotFoundException("Ue not found with id: " + dtoId));

			} else {
				// Insert a new entity
				entity = targetEntityClass.getDeclaredConstructor().newInstance();
			}

			mapper.map(sourceDto, entity);

			// Managing nested entities
			List<Method> persistentAggregateSetters = findPersistentAggregateProperties(targetEntityClass, false);

			for (Method curMethod : persistentAggregateSetters) {

				sb.setLength(0);

				String dtoAccessorName = sb.append(curMethod.getName().replaceFirst("^set", "get")).append("Id")
						.toString();

				try {
					Method dtoMethodGetter = sourceDto.getClass().getMethod(dtoAccessorName);

					Object aggregateId = dtoMethodGetter.invoke(sourceDto, new Object[0]);

					if (aggregateId != null) {

						Method entityMethodGetter = entity.getClass()
								.getMethod(curMethod.getName().replaceFirst("^set", "get"));
						Object entityAggregateValue = entityMethodGetter.invoke(entity);

						Object entityAggregateId = null;
						if (entityAggregateValue != null) {

							Method entityAggregateIdGetter = entityAggregateValue.getClass().getMethod("getId");

							entityAggregateId = entityAggregateIdGetter.invoke(entityAggregateValue);

						}
						if (!aggregateId.equals(entityAggregateId)) {

							sb.setLength(0);

							String aggregateClassSimpleName = curMethod.getParameterTypes()[0].getSimpleName();
							aggregateClassSimpleName = aggregateClassSimpleName.substring(0, 1).toLowerCase()
									+ aggregateClassSimpleName.substring(1);
							String aggregateEntityRepositoryName = sb.append(aggregateClassSimpleName)
									.append("Repository").toString();

							JpaRepository aggregateEntityRepo = (JpaRepository) ApplicationContextProvider
									.getApplicationContext().getBean(aggregateEntityRepositoryName);

							Object aggregateEntity = aggregateEntityRepo.findById(aggregateId)
									.orElseThrow(() -> new EntityNotFoundException(
											curMethod.getParameterTypes()[0] + " not found with id: " + dtoId));

							curMethod.invoke(entity, aggregateEntity);
						}

					} else {
						curMethod.invoke(entity, (Object) null);
					}
				} catch (NoSuchMethodException ex) {
					// If there is not corresponding aggregate Id setter on DTO, skip the property
					log.debug("No {} method found in Dto, skipping mapping of {}", dtoAccessorName,
							curMethod.getName());
					continue;
				}

			}

			return (T) entity;
		} catch (Throwable e) {
			log.error("Unable to map dto to entity - Dto: {}", sourceDto, e);
			throw new IllegalArgumentException("Unable to map dto to entity", e);
		}

	}

	@Override
	public S toDto(T sourceEntity, S targetDto) {
		try {

			mapper.map(sourceEntity, targetDto);

			List<Method> persistentAggregateGetters = findPersistentAggregateProperties(sourceEntity.getClass(), true);

			persistentAggregateGetters.forEach(el -> log.debug(el.getName()));

			StringBuilder sb = new StringBuilder();

			for (Method curMethod : persistentAggregateGetters) {

				Object aggregateObject = curMethod.invoke(sourceEntity, new Object[0]);

				String objectAccessorName = curMethod.getName().replaceFirst("^get", "");

				sb.setLength(0);

				String aggregatSetterMethodName = sb.append("set").append(objectAccessorName).append("Id").toString();

				try {
					Method dtoMethodSetter = targetDto.getClass().getMethod(aggregatSetterMethodName, Long.class);

					if (aggregateObject != null) {

						Method method = aggregateObject.getClass().getMethod("getId");
						Object aggregateId = method.invoke(aggregateObject, new Object[0]);

						if (aggregateId != null && aggregateId instanceof Long) {

							dtoMethodSetter.invoke(targetDto, aggregateId);

						} else {
							log.error("Unable to extract id from entity aggregate - Entity: {} - Aggregate: {}",
									sourceEntity, aggregateObject);
							throw new IllegalArgumentException("Unable to map entity to dto");

						}
					} else {
						dtoMethodSetter.invoke(targetDto, new Object[] { null });
					}
				} catch (NoSuchMethodException ex) {
					// If there is not corresponding aggregate Id setter on DTO, skip the property
					log.debug("No {} method found in Dto, skipping mapping of {}", aggregatSetterMethodName,
							objectAccessorName);
					continue;
				}

			}

			return targetDto;
		} catch (Exception e) {
			log.error("Unable to map entity to dto - Entity: {}", sourceEntity, e);
			throw new IllegalArgumentException("Unable to map entity to dto", e);
		}

	}

	@Override
	public S toDto(T sourceEntity, Class<S> targetDtoClass) {
		try {

			S dto = targetDtoClass.getDeclaredConstructor().newInstance();

			return toDto(sourceEntity, dto);

		} catch (Exception e) {
			log.error("Unable to map entity to dto - Entity: {}", sourceEntity, e);
			throw new IllegalArgumentException("Unable to map entity to dto", e);
		}

	}

	private List<Method> findPersistentAggregateProperties(Class<? extends Object> class1, boolean readMethod) {

		List<Method> result = new ArrayList<>();

		// Assume 'bean' is the instance of your Java bean
		PropertyDescriptor[] propertyDescriptors;
		try {
			propertyDescriptors = Introspector.getBeanInfo(class1).getPropertyDescriptors();
			for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
				String propertyName = propertyDescriptor.getName();
				log.debug("Processing property: {}", propertyName);
				if (!"class".equals(propertyName)) {

					try {
						Field field = class1.getDeclaredField(propertyName);
						if (field != null) {
							Annotation[] annotations = field.getDeclaredAnnotations();
							for (Annotation annotation : annotations) {
								if (annotation instanceof ManyToOne || annotation instanceof OneToOne) {
									// Process the property with the specific annotation
									log.debug("Found manyToOne or oneToOne field: {}", field.getName());
									result.add(readMethod ? propertyDescriptor.getReadMethod()
											: propertyDescriptor.getWriteMethod());

								}
							}
						}
					} catch (NoSuchFieldException ex) {
						log.debug("Field not found for property: {} - Skipping", propertyName);
					}
				}
			}
			return result;
		} catch (IntrospectionException | SecurityException e) {
			log.error("Error:", e);
			throw new RuntimeException(e);
		}
	}

}
